# 機能設計書: node:dgram - UDPソケット

## 1. 機能概要

### 1.1 機能名
node:dgram - UDPソケット

### 1.2 機能ID
94

### 1.3 概要説明
BunにおけるNode.js互換のdgramモジュール実装。UDP（User Datagram Protocol）を使用したコネクションレスなデータグラム通信を提供する。マルチキャスト、ブロードキャストもサポートし、低遅延・軽量な通信を必要とするアプリケーションに適している。

### 1.4 関連画面/API
- CLI: `bun run`コマンドでUDPアプリケーションを実行
- API: `require("node:dgram")`

## 2. 機能要件

### 2.1 ユースケース

#### UC-94-1: UDPサーバーの作成
- **アクター**: 開発者
- **事前条件**: Bunランタイムがインストールされている
- **基本フロー**:
  1. `dgram.createSocket('udp4')`でソケット作成
  2. `socket.bind(port)`でポートにバインド
  3. 'message'イベントでデータグラム受信
- **事後条件**: UDPソケットがリッスン状態

#### UC-94-2: UDPクライアントの作成
- **アクター**: 開発者
- **事前条件**: 対象ホストがアクセス可能
- **基本フロー**:
  1. `dgram.createSocket('udp4')`でソケット作成
  2. `socket.send(msg, port, address)`でデータ送信
- **事後条件**: データグラムが送信される

#### UC-94-3: マルチキャスト通信
- **アクター**: 開発者
- **事前条件**: マルチキャストアドレスが設定可能
- **基本フロー**:
  1. `socket.addMembership(multicastAddress)`でグループ参加
  2. マルチキャストアドレスにデータ送信
  3. グループメンバーからデータ受信
- **事後条件**: マルチキャストグループで通信

### 2.2 機能仕様

#### 2.2.1 ソケット作成
| 項目 | 仕様 |
|------|------|
| ソケット作成 | `dgram.createSocket(type[, callback])` |
| ソケットタイプ | 'udp4' (IPv4), 'udp6' (IPv6) |
| オプション | `reuseAddr`, `ipv6Only`, `recvBufferSize`, `sendBufferSize` |

#### 2.2.2 基本操作
| 項目 | 仕様 |
|------|------|
| バインド | `socket.bind([port][, address][, callback])` |
| 送信 | `socket.send(msg[, offset, length], port[, address][, callback])` |
| 切断 | `socket.close([callback])` |
| アドレス取得 | `socket.address()` |

#### 2.2.3 マルチキャスト
| 項目 | 仕様 |
|------|------|
| グループ参加 | `socket.addMembership(multicastAddress[, multicastInterface])` |
| グループ離脱 | `socket.dropMembership(multicastAddress[, multicastInterface])` |
| ループバック | `socket.setMulticastLoopback(flag)` |
| TTL設定 | `socket.setMulticastTTL(ttl)` |

#### 2.2.4 ブロードキャスト
| 項目 | 仕様 |
|------|------|
| ブロードキャスト有効化 | `socket.setBroadcast(flag)` |

#### 2.2.5 ソケットイベント
| イベント | 説明 |
|---------|------|
| `message` | データグラム受信 (msg, rinfo) |
| `listening` | バインド完了 |
| `close` | ソケット終了 |
| `error` | エラー発生 |

## 3. 技術設計

### 3.1 アーキテクチャ

```
┌─────────────────────────────────────────────────────────────┐
│                    JavaScript Layer                          │
│  ┌─────────────────────────────────────────────────────────┐│
│  │  node:dgram (src/js/node/dgram.ts)                      ││
│  │  ┌───────────────────────────────────────────────────┐  ││
│  │  │  Socket (extends EventEmitter)                    │  ││
│  │  │  - bind(), send(), close()                        │  ││
│  │  │  - addMembership(), dropMembership()              │  ││
│  │  │  - setBroadcast(), setMulticastTTL()              │  ││
│  │  └───────────────────────────────────────────────────┘  ││
│  └─────────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────────┘
                            │
┌───────────────────────────▼─────────────────────────────────┐
│                     Native Layer                             │
│  ┌─────────────────────────────────────────────────────────┐│
│  │  UDP Socket Implementation                               ││
│  │  - Datagram sending/receiving                            ││
│  │  - Multicast group management                            ││
│  │  - Socket options (TTL, Loopback, etc.)                  ││
│  └─────────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────────┘
```

### 3.2 クラス構成

```
dgram
├── createSocket(type, callback) - ソケット作成ファクトリ
└── Socket (extends EventEmitter)
    ├── bind([port][, address][, callback])
    │   └── ポートにバインド
    ├── send(msg, [offset, length,] port, address, [callback])
    │   └── データグラム送信
    ├── close([callback])
    │   └── ソケット終了
    ├── address()
    │   └── バインドアドレス取得
    ├── addMembership(multicastAddr, [multicastInterface])
    │   └── マルチキャストグループ参加
    ├── dropMembership(multicastAddr, [multicastInterface])
    │   └── マルチキャストグループ離脱
    ├── setMulticastTTL(ttl)
    │   └── マルチキャストTTL設定
    ├── setMulticastLoopback(flag)
    │   └── マルチキャストループバック設定
    ├── setBroadcast(flag)
    │   └── ブロードキャスト有効化
    ├── setTTL(ttl)
    │   └── ユニキャストTTL設定
    ├── setRecvBufferSize(size)
    │   └── 受信バッファサイズ設定
    ├── setSendBufferSize(size)
    │   └── 送信バッファサイズ設定
    ├── ref()
    │   └── イベントループ参照保持
    └── unref()
        └── イベントループ参照解除
```

### 3.3 処理フロー

#### データグラム受信フロー
```
1. dgram.createSocket('udp4')
   └── new Socket()
       └── EventEmitter.call(this)

2. socket.bind(port)
   └── [native] UDPソケットバインド
       └── 'listening' イベント発火

3. データグラム受信時
   └── 'message' イベント発火
       └── callback(msg, rinfo)
           ├── msg: Buffer - 受信データ
           └── rinfo: { address, family, port, size }
```

#### データグラム送信フロー
```
1. socket.send(msg, port, address)
   └── [native] sendto() 呼び出し
       └── callback(error, bytes)

2. 複数バッファ送信
   └── socket.send([buf1, buf2], port, address)
       └── [native] sendmsg() 呼び出し
```

## 4. コードリーディングガイド

### 4.1 推奨読解順序

1. **モジュール構造** (`src/js/node/dgram.ts`)
   - EventEmitterのインポート
   - createSocket()ファクトリ関数

2. **Socket クラス**
   - EventEmitterを継承
   - ソケットタイプ（udp4/udp6）の管理
   - bind(), send(), close()メソッド

3. **マルチキャスト機能**
   - addMembership(), dropMembership()
   - setMulticastTTL(), setMulticastLoopback()

4. **ユーティリティ**
   - address()によるアドレス情報取得
   - ref()/unref()によるイベントループ制御

### 4.2 プログラム呼び出し階層図

```
dgram.createSocket(type, callback)
├── new Socket(type)
│   ├── EventEmitter.call(this)
│   ├── this.type = type
│   └── 'message' リスナー設定 (callback)
└── socket.bind(port, address)
    ├── [native] socket() システムコール
    ├── [native] bind() システムコール
    └── 'listening' イベント発火

socket.send(msg, port, address, callback)
├── Buffer変換 (文字列の場合)
├── [native] sendto() システムコール
└── callback(err, bytes)

socket.addMembership(multicastAddr, interface)
├── [native] setsockopt(IP_ADD_MEMBERSHIP)
└── グループ参加完了

socket.close()
├── [native] close() システムコール
└── 'close' イベント発火
```

### 4.3 データフロー図

```
[UDP Client]                           [UDP Server]
     │                                       │
     │ ─────── Datagram ───────────────────>│
     │        (connectionless)               │
     │                                       │
     │        ┌─────────────────────┐        │
     │        │ 'message' event     │        │
     │        │ (msg, rinfo)        │        │
     │        │   - msg: Buffer     │        │
     │        │   - rinfo.address   │        │
     │        │   - rinfo.port      │        │
     │        └─────────────────────┘        │
     │                                       │
     │ <────── Datagram ─────────────────────│
     │        (response)                     │
     │                                       │


[Multicast Group]
     ┌────────────────────────────────────────┐
     │                                        │
   ┌─┴─┐    ┌───┐    ┌───┐    ┌───┐         │
   │ A │    │ B │    │ C │    │ D │         │
   └─┬─┘    └─┬─┘    └─┬─┘    └─┬─┘         │
     │        │        │        │           │
     └────────┴────────┴────────┘           │
              │                              │
              ▼                              │
     ┌─────────────────────┐                 │
     │ Multicast Address   │                 │
     │ (224.0.0.x)         │◄────────────────┘
     └─────────────────────┘
```

### 4.4 関連ファイル一覧

| ファイルパス | 種別 | 役割 |
|-------------|------|------|
| `src/js/node/dgram.ts` | TypeScript | node:dgramモジュールのメイン実装 |
| `src/js/node/events.ts` | TypeScript | EventEmitter基底クラス |
| `src/bun.js/` | Zig | UDPソケットネイティブ実装 |

## 5. 設計上の考慮事項

### 5.1 Node.js互換性
- Node.jsのdgram APIと完全互換を目指す
- EventEmitterベースのイベントモデル
- rinfoオブジェクトの形式互換

### 5.2 パフォーマンス特性
- コネクションレス通信による低オーバーヘッド
- 小さなデータグラムの高速送受信
- マルチキャストによる効率的なグループ通信

### 5.3 プラットフォーム対応
- IPv4/IPv6デュアルスタック
- マルチキャストインターフェース選択
- ブロードキャスト対応

### 5.4 エラーハンドリング
- ポートバインドエラー（EADDRINUSE等）
- 送信エラー（ENETUNREACH, EHOSTUNREACH等）
- マルチキャストエラー（無効なアドレス等）

## 6. テスト方針

### 6.1 単体テスト
- createSocket()でudp4/udp6ソケット作成
- bind(), send(), close()の基本動作
- アドレス情報取得のテスト

### 6.2 統合テスト
- データグラム送受信テスト
- マルチキャスト通信テスト
- ブロードキャスト通信テスト

### 6.3 互換性テスト
- Node.jsのdgramテストスイートの実行
- 他のUDPアプリケーションとの相互運用テスト

## 7. 変更履歴

| 日付 | バージョン | 変更内容 |
|------|-----------|---------|
| 2026-01-27 | 1.0 | 初版作成 |
