# 機能設計書: node:child_process - 子プロセス管理

## 1. 機能概要

### 1.1 機能名
node:child_process - 子プロセス管理

### 1.2 機能ID
95

### 1.3 概要説明
BunにおけるNode.js互換のchild_processモジュール実装。子プロセスの生成、実行、通信を管理し、外部コマンドの実行やサブプロセスとのIPC通信を可能にする。spawn, exec, fork等の複数のAPIを提供し、同期/非同期の両方の実行モードをサポートする。Bunの高速なプロセス生成機能と連携して効率的な子プロセス管理を実現する。

### 1.4 関連画面/API
- CLI: `bun run`コマンドで子プロセスを利用するアプリケーションを実行
- API: `require("node:child_process")`

## 2. 機能要件

### 2.1 ユースケース

#### UC-95-1: 外部コマンドの実行（exec）
- **アクター**: 開発者
- **事前条件**: 実行対象のコマンドがシステムに存在
- **基本フロー**:
  1. `exec(command)`でシェル経由でコマンド実行
  2. コールバックで標準出力/エラー出力を取得
- **事後条件**: コマンド実行結果が取得される

#### UC-95-2: プロセスの生成（spawn）
- **アクター**: 開発者
- **事前条件**: 実行対象のプログラムが存在
- **基本フロー**:
  1. `spawn(command, args)`でプロセス生成
  2. stdin/stdout/stderrストリームでI/O処理
  3. 'exit'イベントで終了を検知
- **事後条件**: 子プロセスが起動し、通信可能

#### UC-95-3: Node.jsプロセスのフォーク
- **アクター**: 開発者
- **事前条件**: 対象のJavaScriptファイルが存在
- **基本フロー**:
  1. `fork(modulePath)`で新しいNode.jsプロセス生成
  2. IPCチャンネル経由でメッセージ送受信
  3. `process.send()`/`process.on('message')`で通信
- **事後条件**: IPCで通信可能な子プロセスが起動

### 2.2 機能仕様

#### 2.2.1 非同期API
| 関数 | 説明 |
|------|------|
| `spawn(command[, args][, options])` | ストリームベースのプロセス生成 |
| `exec(command[, options], callback)` | シェル経由でコマンド実行 |
| `execFile(file[, args][, options], callback)` | 直接ファイル実行 |
| `fork(modulePath[, args][, options])` | Node.jsプロセスのフォーク |

#### 2.2.2 同期API
| 関数 | 説明 |
|------|------|
| `spawnSync(command[, args][, options])` | 同期的プロセス生成 |
| `execSync(command[, options])` | 同期的コマンド実行 |
| `execFileSync(file[, args][, options])` | 同期的ファイル実行 |

#### 2.2.3 ChildProcessオブジェクト
| プロパティ/メソッド | 説明 |
|---------------------|------|
| `pid` | プロセスID |
| `stdin` | 標準入力ストリーム |
| `stdout` | 標準出力ストリーム |
| `stderr` | 標準エラーストリーム |
| `kill([signal])` | プロセス終了 |
| `send(message[, sendHandle][, options][, callback])` | IPCメッセージ送信 |
| `disconnect()` | IPCチャンネル切断 |
| `connected` | IPC接続状態 |

#### 2.2.4 spawnオプション
| オプション | 説明 |
|-----------|------|
| `cwd` | 作業ディレクトリ |
| `env` | 環境変数 |
| `stdio` | I/O設定（'pipe', 'inherit', 'ignore'等） |
| `detached` | 親プロセスからデタッチ |
| `shell` | シェル経由で実行 |
| `timeout` | タイムアウト（ミリ秒） |
| `uid`, `gid` | ユーザー/グループID |

## 3. 技術設計

### 3.1 アーキテクチャ

```
┌─────────────────────────────────────────────────────────────┐
│                    JavaScript Layer                          │
│  ┌─────────────────────────────────────────────────────────┐│
│  │  node:child_process (src/js/node/child_process.ts)      ││
│  │  ┌───────────────┐ ┌───────────────┐ ┌───────────────┐  ││
│  │  │    spawn      │ │     exec      │ │     fork      │  ││
│  │  └───────────────┘ └───────────────┘ └───────────────┘  ││
│  │  ┌───────────────────────────────────────────────────┐  ││
│  │  │  ChildProcess (extends EventEmitter)              │  ││
│  │  │  - stdin/stdout/stderr streams                    │  ││
│  │  │  - IPC channel                                    │  ││
│  │  └───────────────────────────────────────────────────┘  ││
│  └─────────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────────┘
                            │
┌───────────────────────────▼─────────────────────────────────┐
│                     Native Layer                             │
│  ┌─────────────────────────────────────────────────────────┐│
│  │  Bun Process Spawn Implementation                        ││
│  │  - fork() / vfork() / posix_spawn()                      ││
│  │  - Pipe creation and management                          ││
│  │  - Signal handling                                       ││
│  └─────────────────────────────────────────────────────────┘│
│  ┌─────────────────────────────────────────────────────────┐│
│  │  IPC Implementation                                      ││
│  │  - Unix domain socket / Named pipe                       ││
│  │  - Message serialization                                 ││
│  └─────────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────────┘
```

### 3.2 クラス構成

```
child_process
├── spawn(command, args, options)
│   └── ChildProcess インスタンス返却
├── exec(command, options, callback)
│   └── ChildProcess インスタンス返却
├── execFile(file, args, options, callback)
│   └── ChildProcess インスタンス返却
├── fork(modulePath, args, options)
│   └── ChildProcess インスタンス返却（IPC有効）
├── spawnSync(command, args, options)
│   └── 同期結果オブジェクト返却
├── execSync(command, options)
│   └── stdout Buffer/String 返却
├── execFileSync(file, args, options)
│   └── stdout Buffer/String 返却
└── ChildProcess (extends EventEmitter)
    ├── Properties
    │   ├── pid - プロセスID
    │   ├── stdin - Writable Stream
    │   ├── stdout - Readable Stream
    │   ├── stderr - Readable Stream
    │   ├── stdio - [stdin, stdout, stderr, ...]
    │   ├── connected - IPC接続状態
    │   ├── killed - kill()が呼ばれたか
    │   ├── exitCode - 終了コード
    │   └── signalCode - 終了シグナル
    ├── Methods
    │   ├── kill([signal]) - シグナル送信
    │   ├── send(message) - IPCメッセージ送信
    │   ├── disconnect() - IPC切断
    │   ├── ref() - イベントループ参照保持
    │   └── unref() - イベントループ参照解除
    └── Events
        ├── 'spawn' - プロセス生成成功
        ├── 'exit' - プロセス終了
        ├── 'close' - I/O終了
        ├── 'error' - エラー発生
        ├── 'message' - IPCメッセージ受信
        └── 'disconnect' - IPC切断
```

### 3.3 処理フロー

#### spawn処理フロー
```
1. spawn(command, args, options)
   └── new ChildProcess()
       ├── [native] プロセス生成準備
       ├── パイプ作成 (stdin/stdout/stderr)
       └── [native] fork() / posix_spawn()

2. プロセス生成成功
   └── 'spawn' イベント発火

3. I/O処理
   ├── stdin.write() → 子プロセスへ入力
   ├── stdout 'data' → 子プロセス出力受信
   └── stderr 'data' → 子プロセスエラー受信

4. プロセス終了
   ├── 'exit' イベント発火 (code, signal)
   └── 'close' イベント発火
```

#### fork処理フロー
```
1. fork(modulePath, args, options)
   └── spawn('bun', [modulePath, ...args], {
           stdio: ['pipe', 'pipe', 'pipe', 'ipc']
       })

2. IPCチャンネル確立
   └── Unix domain socket / Named pipe

3. メッセージ通信
   ├── parent.send(message) → child.on('message')
   └── child.send(message) → parent.on('message')

4. IPC切断
   ├── disconnect() 呼び出し
   └── 'disconnect' イベント発火
```

## 4. コードリーディングガイド

### 4.1 推奨読解順序

1. **モジュール構造** (`src/js/node/child_process.ts`)
   - EventEmitter, Streamのインポート
   - Bunネイティブプロセス生成連携

2. **spawn関数**
   - オプション解析とデフォルト値設定
   - ChildProcessインスタンス作成
   - ネイティブプロセス生成呼び出し

3. **ChildProcessクラス**
   - EventEmitterを継承
   - stdin/stdout/stderrストリーム管理
   - kill(), send(), disconnect()メソッド

4. **exec/execFile関数**
   - spawnをラップして簡易API提供
   - 出力バッファリングとコールバック

5. **fork関数**
   - IPCチャンネル付きのspawn
   - メッセージシリアライゼーション

6. **同期API**
   - spawnSync, execSync, execFileSync
   - ブロッキング実行と結果取得

### 4.2 プログラム呼び出し階層図

```
spawn(command, args, options)
├── normalizeSpawnArguments()
│   └── オプション正規化
├── new ChildProcess()
│   ├── EventEmitter.call(this)
│   └── stdio 設定
└── [native] Bun.spawn() 連携
    ├── プロセス生成
    ├── パイプ設定
    └── 'spawn' イベント発火

exec(command, options, callback)
├── spawn(command, { shell: true })
└── stdout/stderr バッファリング
    └── callback(error, stdout, stderr)

fork(modulePath, args, options)
├── spawn('bun', [modulePath, ...args], {
│       stdio: ['pipe', 'pipe', 'pipe', 'ipc']
│   })
└── IPC チャンネル設定
    └── send(), on('message') 有効化

spawnSync(command, args, options)
├── [native] 同期プロセス生成
└── { stdout, stderr, status, signal, error }
```

### 4.3 データフロー図

```
[Parent Process]                    [Child Process]
       │                                   │
       │ ─────── spawn() ─────────────────>│
       │                                   │
       │        ┌─────────────────────┐    │
       │        │ ChildProcess        │    │
       │        │ .stdin  ────────────────>│ stdin
       │        │ .stdout <───────────────│ stdout
       │        │ .stderr <───────────────│ stderr
       │        └─────────────────────┘    │
       │                                   │


[Parent Process]                    [Child Process (fork)]
       │                                   │
       │ ════════ IPC Channel ════════════│
       │                                   │
       │  .send(message)                   │
       │  ─────────────────────────────────>  process.on('message')
       │                                   │
       │  .on('message')                   │
       │  <─────────────────────────────────  process.send(message)
       │                                   │
```

### 4.4 関連ファイル一覧

| ファイルパス | 種別 | 役割 |
|-------------|------|------|
| `src/js/node/child_process.ts` | TypeScript | node:child_processモジュールのメイン実装 |
| `src/js/node/events.ts` | TypeScript | EventEmitter基底クラス |
| `src/js/internal/streams/` | TypeScript | stdin/stdout/stderrストリーム |
| `src/bun.js/` | Zig | プロセス生成ネイティブ実装 |

## 5. 設計上の考慮事項

### 5.1 Node.js互換性
- Node.jsのchild_process APIと完全互換を目指す
- ChildProcessオブジェクトの振る舞い互換
- IPC通信の互換性確保

### 5.2 パフォーマンス最適化
- Bunの高速プロセス生成機能の活用
- 効率的なパイプ管理
- 最小限のオーバーヘッドでのIPC

### 5.3 プラットフォーム対応
- Unix系: fork(), vfork(), posix_spawn()
- Windows: CreateProcess()
- シグナルハンドリングの差異対応

### 5.4 セキュリティ
- シェルインジェクション対策（execFile推奨）
- 環境変数の適切な継承
- 権限昇格の防止

### 5.5 エラーハンドリング
- プロセス生成エラー（ENOENT, EACCES等）
- シグナルによる終了の検出
- タイムアウト処理

## 6. テスト方針

### 6.1 単体テスト
- spawn(), exec(), fork()の基本動作
- ChildProcessの各メソッド/プロパティ
- 同期APIの動作テスト

### 6.2 統合テスト
- 外部コマンド実行テスト
- IPC通信テスト
- 大量出力のストリーム処理テスト

### 6.3 互換性テスト
- Node.jsのchild_processテストスイートの実行
- 各種シェルコマンドの実行テスト

## 7. 変更履歴

| 日付 | バージョン | 変更内容 |
|------|-----------|---------|
| 2026-01-27 | 1.0 | 初版作成 |
