# 機能設計書: node:stream - ストリームAPI

## 1. 機能概要

### 1.1 機能名
node:stream - ストリームAPI

### 1.2 機能ID
97

### 1.3 概要説明
BunにおけるNode.js互換のstreamモジュール実装。データの読み書きを抽象化したストリームインターフェースを提供し、大量データの効率的な処理を可能にする。Readable, Writable, Duplex, Transformの4つの基本ストリームタイプと、pipeline, finished等のユーティリティ関数を提供する。fs, http, net等の多くのコアモジュールがこのストリームAPIを基盤として使用している。

### 1.4 関連画面/API
- CLI: `bun run`コマンドでストリームを利用するアプリケーションを実行
- API: `require("node:stream")`

## 2. 機能要件

### 2.1 ユースケース

#### UC-97-1: ファイル読み込みストリーム
- **アクター**: 開発者
- **事前条件**: 読み込み対象のファイルが存在
- **基本フロー**:
  1. `fs.createReadStream()`でReadable作成
  2. 'data'イベントでチャンク受信
  3. 'end'イベントで読み込み完了を検知
- **事後条件**: ファイル内容がストリームで読み込まれる

#### UC-97-2: ストリームパイプライン
- **アクター**: 開発者
- **事前条件**: 複数のストリームが準備されている
- **基本フロー**:
  1. `stream.pipeline(src, transform, dest, callback)`で連結
  2. 自動的なバックプレッシャー処理
  3. エラー伝播と自動クリーンアップ
- **事後条件**: データが変換されて出力

#### UC-97-3: カスタムTransformストリーム
- **アクター**: 開発者
- **事前条件**: データ変換ロジックが定義されている
- **基本フロー**:
  1. Transformクラスを継承
  2. `_transform(chunk, encoding, callback)`を実装
  3. パイプラインに組み込み
- **事後条件**: カスタム変換が適用される

### 2.2 機能仕様

#### 2.2.1 Readable Stream
| メソッド/プロパティ | 説明 |
|--------------------|------|
| `read([size])` | データ読み取り |
| `pause()` | 'data'イベント一時停止 |
| `resume()` | 'data'イベント再開 |
| `pipe(destination)` | 出力先に接続 |
| `unpipe([destination])` | 接続解除 |
| `readable` | 読み取り可能状態 |
| `readableLength` | バッファ内データサイズ |

#### 2.2.2 Writable Stream
| メソッド/プロパティ | 説明 |
|--------------------|------|
| `write(chunk[, encoding][, callback])` | データ書き込み |
| `end([chunk][, encoding][, callback])` | 書き込み終了 |
| `cork()` | 書き込みバッファリング開始 |
| `uncork()` | バッファリング解除 |
| `writable` | 書き込み可能状態 |
| `writableLength` | バッファ内データサイズ |

#### 2.2.3 Duplex Stream
| 特徴 | 説明 |
|------|------|
| 継承 | Readable + Writable |
| 用途 | 双方向通信（Socket等） |
| 独立性 | 読み取り/書き込みは独立 |

#### 2.2.4 Transform Stream
| 特徴 | 説明 |
|------|------|
| 継承 | Duplex |
| 用途 | データ変換（圧縮、暗号化等） |
| メソッド | `_transform(chunk, encoding, callback)` |

#### 2.2.5 ユーティリティ関数
| 関数 | 説明 |
|------|------|
| `pipeline(...streams, callback)` | ストリーム連結 |
| `finished(stream[, options], callback)` | 終了検知 |
| `Readable.from(iterable)` | イテラブルからReadable作成 |
| `compose(...streams)` | ストリーム合成 |
| `addAbortSignal(signal, stream)` | AbortSignal接続 |

## 3. 技術設計

### 3.1 アーキテクチャ

```
┌─────────────────────────────────────────────────────────────┐
│                    JavaScript Layer                          │
│  ┌─────────────────────────────────────────────────────────┐│
│  │  node:stream (src/js/node/stream.ts)                    ││
│  │       │                                                  ││
│  │       └──> internal/stream (src/js/internal/stream.ts)  ││
│  │            ┌───────────────────────────────────────────┐││
│  │            │  Stream (Legacy Base)                     │││
│  │            │  Readable / Writable / Duplex / Transform │││
│  │            │  pipeline / finished / compose            │││
│  │            └───────────────────────────────────────────┘││
│  └─────────────────────────────────────────────────────────┘│
│  ┌─────────────────────────────────────────────────────────┐│
│  │  internal/streams/* (各ストリーム実装)                   ││
│  │  - readable.ts, writable.ts, duplex.ts, transform.ts   ││
│  │  - pipeline.ts, end-of-stream.ts, destroy.ts           ││
│  └─────────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────────┘
```

### 3.2 クラス構成

```
stream
├── Stream (Legacy Base - extends EventEmitter)
│   └── 基本ストリームクラス
├── Readable (extends Stream)
│   ├── _read(size) - オーバーライド用
│   ├── read([size]) - データ読み取り
│   ├── push(chunk) - データ追加
│   ├── pause() / resume() - フロー制御
│   ├── pipe(dest) / unpipe([dest])
│   └── Events: 'data', 'end', 'readable', 'error', 'close'
├── Writable (extends Stream)
│   ├── _write(chunk, encoding, callback) - オーバーライド用
│   ├── _writev(chunks, callback) - 複数チャンク書き込み
│   ├── write(chunk) - データ書き込み
│   ├── end([chunk]) - 終了
│   ├── cork() / uncork() - バッファリング
│   └── Events: 'drain', 'finish', 'error', 'close'
├── Duplex (extends Readable implements Writable)
│   └── 双方向ストリーム
├── Transform (extends Duplex)
│   ├── _transform(chunk, encoding, callback)
│   └── _flush(callback)
├── PassThrough (extends Transform)
│   └── データ変換なしのパススルー
├── pipeline(...streams, callback)
│   └── ストリーム連結とエラー処理
├── finished(stream, callback)
│   └── ストリーム終了検知
├── compose(...streams)
│   └── ストリーム合成
└── Utility
    ├── Readable.from(iterable)
    ├── addAbortSignal(signal, stream)
    ├── getDefaultHighWaterMark(objectMode)
    └── setDefaultHighWaterMark(objectMode, value)
```

### 3.3 処理フロー

#### Readableストリームフロー
```
1. new Readable(options)
   ├── highWaterMark 設定（デフォルト16KB）
   └── 内部バッファ初期化

2. read() 呼び出し（流動モード）
   ├── 内部バッファからデータ取得
   ├── バッファ不足時 _read() 呼び出し
   └── 'data' イベント発火

3. pipe(destination)
   ├── 'data' リスナー登録
   ├── dest.write(chunk) 呼び出し
   ├── バックプレッシャー処理
   │   ├── dest.write() が false → pause()
   │   └── dest 'drain' → resume()
   └── 'end' で dest.end()
```

#### Writableストリームフロー
```
1. new Writable(options)
   ├── highWaterMark 設定
   └── 内部バッファ初期化

2. write(chunk) 呼び出し
   ├── バッファに追加
   ├── _write() 呼び出し
   ├── バッファ超過時 false 返却
   └── バッファ空時 'drain' イベント

3. end([chunk]) 呼び出し
   ├── 最終データ書き込み
   ├── 全書き込み完了待機
   └── 'finish' イベント発火
```

#### pipelineフロー
```
1. pipeline(src, ...transforms, dest, callback)
   ├── 各ストリームを pipe() で連結
   ├── エラーリスナー設定
   └── 終了リスナー設定

2. エラー発生時
   ├── 全ストリームを destroy()
   └── callback(error)

3. 正常終了時
   └── callback()
```

## 4. コードリーディングガイド

### 4.1 推奨読解順序

1. **エントリーポイント** (`src/js/node/stream.ts`)
   - **1-8行目**: internal/streamからエクスポート
   - シンプルな再エクスポート構造

2. **内部ストリーム実装** (`src/js/internal/stream.ts`)
   - **1-14行目**: 各種ユーティリティのインポート
   - **16-23行目**: Stream基底クラスへのユーティリティ追加
   - **24-68行目**: Readableプロトタイプへのオペレータ追加
   - **65-77行目**: Stream.Writable, Duplex, Transform, PassThrough設定
   - **104-113行目**: Stream.pipeline, finished, compose設定

3. **Readableストリーム** (`src/js/internal/streams/readable.ts`)
   - read(), push(), pause(), resume()メソッド
   - 'data', 'end', 'readable'イベント

4. **Writableストリーム** (`src/js/internal/streams/writable.ts`)
   - write(), end(), cork(), uncork()メソッド
   - 'drain', 'finish'イベント

5. **パイプラインユーティリティ** (`src/js/internal/streams/pipeline.ts`)
   - pipeline関数の実装
   - エラー伝播とクリーンアップ

### 4.2 プログラム呼び出し階層図

```
require('node:stream')
└── src/js/node/stream.ts
    └── require('internal/stream')
        └── src/js/internal/stream.ts
            ├── require('internal/streams/operators')
            ├── require('internal/streams/compose')
            ├── require('internal/streams/state')
            ├── require('internal/streams/pipeline')
            ├── require('internal/streams/destroy')
            ├── require('internal/streams/end-of-stream')
            ├── require('internal/streams/legacy').Stream
            ├── require('internal/streams/readable')
            ├── require('internal/streams/writable')
            ├── require('internal/streams/duplex')
            ├── require('internal/streams/transform')
            ├── require('internal/streams/passthrough')
            └── require('internal/streams/duplexpair')

readable.pipe(writable)
├── writable にデータ書き込み
├── バックプレッシャー検出
│   └── readable.pause() / resume()
└── 'end' で writable.end()

pipeline(src, transform, dest, callback)
├── src.pipe(transform)
├── transform.pipe(dest)
├── エラーリスナー設定
└── 終了/エラー時 callback 呼び出し
```

### 4.3 データフロー図

```
[Readable]                    [Writable]
    │                             │
    │ ──── read() ────┐           │
    │ <─── push() ────┘           │
    │                             │
    │ ═══════ pipe() ════════════>│
    │    'data' event             │ write()
    │    chunk →                  │
    │                             │
    │ <── false (backpressure) ───│
    │    pause()                  │
    │                             │
    │ <── 'drain' ────────────────│
    │    resume()                 │
    │                             │
    │ ──── 'end' ────────────────>│
    │                             │ end()
    │                             │
    │                             │ 'finish'


[Pipeline]
┌──────────┐     ┌───────────┐     ┌──────────┐
│ Readable │────>│ Transform │────>│ Writable │
│  (src)   │pipe │   (zlib)  │pipe │  (dest)  │
└──────────┘     └───────────┘     └──────────┘
     │                │                 │
     └────────────────┼─────────────────┘
                      │
              Error propagation
              Auto cleanup (destroy)
```

### 4.4 関連ファイル一覧

| ファイルパス | 種別 | 役割 |
|-------------|------|------|
| `src/js/node/stream.ts` | TypeScript | node:streamモジュールエントリーポイント |
| `src/js/internal/stream.ts` | TypeScript | ストリームAPI統合と設定 |
| `src/js/internal/streams/readable.ts` | TypeScript | Readableストリーム実装 |
| `src/js/internal/streams/writable.ts` | TypeScript | Writableストリーム実装 |
| `src/js/internal/streams/duplex.ts` | TypeScript | Duplexストリーム実装 |
| `src/js/internal/streams/transform.ts` | TypeScript | Transformストリーム実装 |
| `src/js/internal/streams/passthrough.ts` | TypeScript | PassThroughストリーム実装 |
| `src/js/internal/streams/pipeline.ts` | TypeScript | pipeline関数実装 |
| `src/js/internal/streams/end-of-stream.ts` | TypeScript | finished関数実装 |
| `src/js/internal/streams/destroy.ts` | TypeScript | destroy処理実装 |
| `src/js/internal/streams/operators.ts` | TypeScript | ストリームオペレータ（map, filter等） |
| `src/js/internal/streams/state.ts` | TypeScript | highWaterMark管理 |

## 5. 設計上の考慮事項

### 5.1 Node.js互換性
- Node.jsのstream APIと完全互換を目指す
- Web Streams APIとの相互運用性
- readable-streamパッケージとの互換性

### 5.2 パフォーマンス最適化
- highWaterMarkによるバッファリング制御
- バックプレッシャーによるメモリ効率化
- 効率的なバッファコピー

### 5.3 エラーハンドリング
- pipeline()での自動エラー伝播
- destroy()での適切なリソース解放
- finished()での終了状態検知

### 5.4 メモリ管理
- highWaterMarkによるバッファサイズ制限
- バックプレッシャーによる生産者制御
- 適切なストリーム破棄

## 6. テスト方針

### 6.1 単体テスト
- Readable/Writable/Duplex/Transformの基本動作
- pipe()のバックプレッシャー処理
- pipeline()のエラー伝播

### 6.2 統合テスト
- ファイルストリームとの連携
- HTTPストリームとの連携
- 圧縮ストリームとの連携

### 6.3 互換性テスト
- Node.jsのstreamテストスイートの実行
- readable-streamパッケージとの互換性テスト

## 7. 変更履歴

| 日付 | バージョン | 変更内容 |
|------|-----------|---------|
| 2026-01-27 | 1.0 | 初版作成 |
