# 機能設計書: node:buffer - Bufferオブジェクト

## 1. 機能概要

### 1.1 機能名
node:buffer - Bufferオブジェクト

### 1.2 機能ID
98

### 1.3 概要説明
BunにおけるNode.js互換のbufferモジュール実装。固定長のバイナリデータを効率的に扱うBufferクラスを提供する。文字列とバイナリデータ間のエンコード/デコード、バイナリデータの操作（読み書き、比較、検索等）を行う。Uint8Arrayを継承し、TypedArrayとの互換性を持つ。Zigで実装されたネイティブ関数による高速なバッファ操作を提供する。

### 1.4 関連画面/API
- CLI: `bun run`コマンドでBufferを利用するアプリケーションを実行
- API: `require("node:buffer")`, グローバル`Buffer`

## 2. 機能要件

### 2.1 ユースケース

#### UC-98-1: 文字列からBufferへの変換
- **アクター**: 開発者
- **事前条件**: 変換対象の文字列がある
- **基本フロー**:
  1. `Buffer.from(string, encoding)`で変換
  2. 指定エンコーディングでバイト列に変換
- **事後条件**: 文字列がBufferに変換される

#### UC-98-2: バイナリデータの読み書き
- **アクター**: 開発者
- **事前条件**: Bufferが作成されている
- **基本フロー**:
  1. `buffer.writeUInt32BE(value, offset)`で書き込み
  2. `buffer.readUInt32BE(offset)`で読み込み
- **事後条件**: バイナリデータが操作される

#### UC-98-3: Base64エンコード/デコード
- **アクター**: 開発者
- **事前条件**: エンコード/デコード対象のデータがある
- **基本フロー**:
  1. `Buffer.from(string, 'base64')`でデコード
  2. `buffer.toString('base64')`でエンコード
- **事後条件**: Base64変換が完了

### 2.2 機能仕様

#### 2.2.1 Buffer作成メソッド
| メソッド | 説明 |
|---------|------|
| `Buffer.alloc(size[, fill[, encoding]])` | 指定サイズで初期化済みBuffer作成 |
| `Buffer.allocUnsafe(size)` | 未初期化Buffer作成（高速） |
| `Buffer.allocUnsafeSlow(size)` | プール未使用の未初期化Buffer |
| `Buffer.from(array)` | 配列からBuffer作成 |
| `Buffer.from(buffer)` | 既存Bufferからコピー |
| `Buffer.from(string[, encoding])` | 文字列からBuffer作成 |
| `Buffer.from(arrayBuffer[, offset[, length]])` | ArrayBufferからBuffer作成 |

#### 2.2.2 サポートエンコーディング
| エンコーディング | 説明 |
|-----------------|------|
| `utf8` | UTF-8（デフォルト） |
| `utf16le` / `ucs2` | UTF-16 Little Endian |
| `latin1` / `binary` | Latin-1 |
| `ascii` | 7-bit ASCII |
| `base64` | Base64 |
| `base64url` | URL-safe Base64 |
| `hex` | 16進数文字列 |

#### 2.2.3 読み込みメソッド
| メソッド | 説明 |
|---------|------|
| `readUInt8(offset)` | 符号なし8ビット整数読み込み |
| `readUInt16BE/LE(offset)` | 符号なし16ビット整数読み込み |
| `readUInt32BE/LE(offset)` | 符号なし32ビット整数読み込み |
| `readBigUInt64BE/LE(offset)` | 符号なし64ビット整数読み込み |
| `readInt8/16/32(offset)` | 符号付き整数読み込み |
| `readFloatBE/LE(offset)` | 32ビット浮動小数点読み込み |
| `readDoubleBE/LE(offset)` | 64ビット浮動小数点読み込み |

#### 2.2.4 書き込みメソッド
| メソッド | 説明 |
|---------|------|
| `writeUInt8(value, offset)` | 符号なし8ビット整数書き込み |
| `writeUInt16BE/LE(value, offset)` | 符号なし16ビット整数書き込み |
| `writeUInt32BE/LE(value, offset)` | 符号なし32ビット整数書き込み |
| `writeBigUInt64BE/LE(value, offset)` | 符号なし64ビット整数書き込み |
| `writeInt8/16/32(value, offset)` | 符号付き整数書き込み |
| `writeFloatBE/LE(value, offset)` | 32ビット浮動小数点書き込み |
| `writeDoubleBE/LE(value, offset)` | 64ビット浮動小数点書き込み |

#### 2.2.5 操作メソッド
| メソッド | 説明 |
|---------|------|
| `toString([encoding[, start[, end]]])` | 文字列に変換 |
| `slice([start[, end]])` | 部分Buffer取得 |
| `copy(target[, targetStart[, sourceStart[, sourceEnd]]])` | コピー |
| `fill(value[, offset[, end]][, encoding])` | 値で埋める |
| `compare(target)` | 比較 |
| `equals(otherBuffer)` | 等価判定 |
| `indexOf(value[, byteOffset][, encoding])` | 検索 |
| `includes(value[, byteOffset][, encoding])` | 含有判定 |
| `swap16/32/64()` | バイトスワップ |

## 3. 技術設計

### 3.1 アーキテクチャ

```
┌─────────────────────────────────────────────────────────────┐
│                    JavaScript Layer                          │
│  ┌─────────────────────────────────────────────────────────┐│
│  │  node:buffer (global Buffer)                            ││
│  │  ┌───────────────────────────────────────────────────┐  ││
│  │  │  Buffer (extends Uint8Array)                      │  ││
│  │  │  - alloc(), allocUnsafe(), from()                 │  ││
│  │  │  - read/write methods                             │  ││
│  │  │  - toString(), slice(), copy(), fill()            │  ││
│  │  └───────────────────────────────────────────────────┘  ││
│  │  ┌───────────────────────────────────────────────────┐  ││
│  │  │  Blob, File, atob(), btoa()                       │  ││
│  │  └───────────────────────────────────────────────────┘  ││
│  └─────────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────────┘
                            │
┌───────────────────────────▼─────────────────────────────────┐
│                     Native Layer                             │
│  ┌─────────────────────────────────────────────────────────┐│
│  │  buffer.zig (src/bun.js/node/buffer.zig)                ││
│  │  ┌───────────────────────────────────────────────────┐  ││
│  │  │  BufferVectorized                                 │  ││
│  │  │  - fill() - 高速バッファ充填                      │  ││
│  │  │  - エンコーディング変換                           │  ││
│  │  └───────────────────────────────────────────────────┘  ││
│  └─────────────────────────────────────────────────────────┘│
│  ┌─────────────────────────────────────────────────────────┐│
│  │  Encoder (jsc.WebCore.encoding)                         ││
│  │  - writeU8/U16 - エンコード処理                         ││
│  └─────────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────────┘
```

### 3.2 クラス構成

```
buffer
├── Buffer (extends Uint8Array)
│   ├── Static Methods
│   │   ├── alloc(size, fill, encoding)
│   │   ├── allocUnsafe(size)
│   │   ├── allocUnsafeSlow(size)
│   │   ├── from(data, encoding)
│   │   ├── concat(list, totalLength)
│   │   ├── compare(buf1, buf2)
│   │   ├── isBuffer(obj)
│   │   ├── isEncoding(encoding)
│   │   ├── byteLength(string, encoding)
│   │   └── copyBytesFrom(view, offset, length)
│   ├── Instance Methods (Read)
│   │   ├── readUInt8/16/32/BigUInt64
│   │   ├── readInt8/16/32/BigInt64
│   │   └── readFloat/Double (BE/LE)
│   ├── Instance Methods (Write)
│   │   ├── writeUInt8/16/32/BigUInt64
│   │   ├── writeInt8/16/32/BigInt64
│   │   └── writeFloat/Double (BE/LE)
│   └── Instance Methods (Utility)
│       ├── toString(encoding, start, end)
│       ├── toJSON()
│       ├── slice(start, end)
│       ├── subarray(start, end)
│       ├── copy(target, targetStart, sourceStart, sourceEnd)
│       ├── fill(value, offset, end, encoding)
│       ├── compare(target, targetStart, targetEnd, sourceStart, sourceEnd)
│       ├── equals(otherBuffer)
│       ├── indexOf/lastIndexOf(value, byteOffset, encoding)
│       ├── includes(value, byteOffset, encoding)
│       ├── write(string, offset, length, encoding)
│       └── swap16/32/64()
├── Blob
│   └── Web API互換Blob
├── File
│   └── Web API互換File
├── atob(encodedData)
│   └── Base64デコード
├── btoa(stringToEncode)
│   └── Base64エンコード
├── constants
│   └── MAX_LENGTH, MAX_STRING_LENGTH
└── kMaxLength
    └── 最大Bufferサイズ
```

### 3.3 処理フロー

#### Buffer.from(string, encoding)フロー
```
1. Buffer.from(string, encoding)
   ├── encoding 検証
   ├── バイト長計算
   ├── Uint8Array作成
   └── エンコード処理
       ├── utf8 → UTF-8エンコード
       ├── base64 → Base64デコード
       ├── hex → 16進デコード
       └── etc.

2. ネイティブエンコード処理
   └── [native] Encoder.writeU8/U16
```

#### buffer.fill(value)フロー
```
1. buffer.fill(value, offset, end, encoding)
   ├── パラメータ検証
   ├── 値のエンコード
   └── [native] BufferVectorized.fill()
       ├── パターン長別最適化
       │   ├── 1バイト → memset
       │   ├── 4/8/16バイト → memset_pattern (Mac)
       │   └── その他 → ループコピー
       └── バッファ充填完了
```

## 4. コードリーディングガイド

### 4.1 推奨読解順序

1. **ネイティブバッファ実装** (`src/bun.js/node/buffer.zig`)
   - **1-79行目**: BufferVectorized.fill関数
   - **12-45行目**: エンコーディング別の書き込み処理
   - **49-77行目**: パターンサイズ別の最適化処理

2. **fill関数の最適化パターン**
   - **50-54行目**: written == 0 の場合はtrue返却
   - **52-54行目**: written == 1 の場合はmemset使用
   - **55-61行目**: 4/8/16バイトパターンはmemset_pattern使用（Mac）
   - **67-77行目**: その他はループでコピー

3. **エンコーディング処理**
   - **12-45行目**: encoding別のEncoder呼び出し
   - utf8, ascii, latin1, utf16le, base64, hex等

### 4.2 プログラム呼び出し階層図

```
Buffer.from(string, encoding)
├── JavaScript層
│   ├── encoding 正規化
│   └── Uint8Array作成
└── Native層
    └── Encoder.writeU8/writeU16
        └── エンコード処理

buffer.fill(value, offset, end, encoding)
├── JavaScript層
│   ├── パラメータ検証
│   └── 値のエンコード準備
└── Native層
    └── BufferVectorized.fill()
        ├── Encoder.writeU8/writeU16 (値をバイト列に)
        └── パターン最適化充填
            ├── memset (1バイト)
            ├── memset_pattern4/8/16 (Mac)
            └── ループコピー (その他)

Buffer.concat(list, totalLength)
├── 合計サイズ計算
├── 新Buffer作成
└── 各Bufferをコピー
```

### 4.3 データフロー図

```
[String]                              [Buffer]
    │                                     │
    │ ─── Buffer.from() ─────────────────>│
    │     (encoding: utf8/base64/hex)     │
    │                                     │
    │ <── buffer.toString() ──────────────│
    │     (decoding)                      │


[Buffer Fill Optimization]
┌─────────────────────────────────────────────────┐
│ fill(value, offset, end, encoding)              │
│                                                 │
│  value → Encode → pattern                       │
│                                                 │
│  ┌─────────────────────────────────────────┐    │
│  │ Pattern Size        │ Method            │    │
│  ├─────────────────────┼───────────────────┤    │
│  │ 1 byte              │ memset            │    │
│  │ 4/8/16 bytes (Mac)  │ memset_pattern*   │    │
│  │ other               │ loop copy         │    │
│  └─────────────────────┴───────────────────┘    │
└─────────────────────────────────────────────────┘


[Binary Data Operations]
┌──────────┐      write*()      ┌──────────┐
│  Number  │ ──────────────────>│  Buffer  │
│ (value)  │                    │ (bytes)  │
└──────────┘      read*()       └──────────┘
     ^      <──────────────────      │
     └───────────────────────────────┘
```

### 4.4 関連ファイル一覧

| ファイルパス | 種別 | 役割 |
|-------------|------|------|
| `src/bun.js/node/buffer.zig` | Zig | BufferVectorized.fill実装（91行） |
| `src/js/builtins/` | TypeScript | Bufferグローバル実装 |
| `src/bun.js/bindings/` | C++ | JSCバインディング |

## 5. 設計上の考慮事項

### 5.1 Node.js互換性
- Node.jsのBuffer APIと完全互換を目指す
- Uint8Arrayの継承による互換性
- 全エンコーディングのサポート

### 5.2 パフォーマンス最適化
- **BufferVectorized.fill**: パターンサイズ別最適化
- **memset_pattern**: Macでの高速パターン充填
- **Zig実装**: ネイティブ速度でのバッファ操作

### 5.3 メモリ管理
- allocUnsafe: プールからの高速割り当て
- allocUnsafeSlow: プール未使用の直接割り当て
- kMaxLength: 最大Bufferサイズ制限

### 5.4 セキュリティ
- alloc: 初期化済み（ゼロクリア）
- allocUnsafe: 未初期化（機密データ注意）
- fill: 明示的なバッファクリア

## 6. テスト方針

### 6.1 単体テスト
- alloc/allocUnsafe/fromの基本動作
- 各エンコーディングの変換テスト
- read/writeメソッドのテスト

### 6.2 統合テスト
- 大容量データのバッファ操作
- ストリームとの連携テスト
- ファイルI/Oとの連携テスト

### 6.3 互換性テスト
- Node.jsのbufferテストスイートの実行
- TypedArrayとの相互運用テスト

## 7. 変更履歴

| 日付 | バージョン | 変更内容 |
|------|-----------|---------|
| 2026-01-27 | 1.0 | 初版作成 |
