# 機能設計書: node:util - ユーティリティ関数

## 1. 機能概要

### 1.1 機能名
node:util - ユーティリティ関数

### 1.2 機能ID
99

### 1.3 概要説明
BunにおけるNode.js互換のutilモジュール実装。デバッグ、文字列フォーマット、オブジェクト検査、型チェック、非同期ユーティリティなど、Node.js開発で広く使用される汎用関数を提供する。inspect()によるオブジェクト文字列化、promisify()によるコールバック関数のPromise化、types サブモジュールによる型判定など、開発効率を向上させる多数のユーティリティを含む。

### 1.4 関連画面/API
- CLI: `bun run`コマンドでutilを利用するアプリケーションを実行
- API: `require("node:util")`, `require("node:util/types")`

## 2. 機能要件

### 2.1 ユースケース

#### UC-99-1: オブジェクトの文字列化（inspect）
- **アクター**: 開発者
- **事前条件**: 検査対象のオブジェクトがある
- **基本フロー**:
  1. `util.inspect(object, options)`で文字列化
  2. depth, colors, maxArrayLengthなどのオプション指定
- **事後条件**: オブジェクトが読みやすい文字列で表示

#### UC-99-2: コールバック関数のPromise化
- **アクター**: 開発者
- **事前条件**: コールバックパターンの関数がある
- **基本フロー**:
  1. `util.promisify(fn)`でPromiseラッパー作成
  2. async/awaitで使用可能に
- **事後条件**: Promise形式で使用可能

#### UC-99-3: 型判定
- **アクター**: 開発者
- **事前条件**: 型チェック対象の値がある
- **基本フロー**:
  1. `util.types.isTypedArray(value)`等で型判定
  2. 判定結果（boolean）を取得
- **事後条件**: 型が正しく判定される

### 2.2 機能仕様

#### 2.2.1 フォーマット関数
| 関数 | 説明 |
|------|------|
| `format(format[, ...args])` | printf形式の文字列フォーマット |
| `formatWithOptions(inspectOptions, format[, ...args])` | オプション付きフォーマット |
| `inspect(object[, options])` | オブジェクト文字列化 |
| `styleText(format, text)` | ANSIスタイル適用 |
| `stripVTControlCharacters(str)` | ANSI制御文字除去 |

#### 2.2.2 非同期ユーティリティ
| 関数 | 説明 |
|------|------|
| `promisify(original)` | コールバック関数をPromise化 |
| `callbackify(original)` | async関数をコールバック形式に |
| `aborted(signal, resource)` | AbortSignal待機 |

#### 2.2.3 型チェック関数（deprecated）
| 関数 | 説明 |
|------|------|
| `isArray(value)` | 配列判定 |
| `isBoolean(value)` | 真偽値判定 |
| `isBuffer(value)` | Buffer判定 |
| `isDate(value)` | Date判定 |
| `isError(value)` | Error判定 |
| `isFunction(value)` | 関数判定 |
| `isNull(value)` | null判定 |
| `isNumber(value)` | 数値判定 |
| `isObject(value)` | オブジェクト判定 |
| `isRegExp(value)` | RegExp判定 |
| `isString(value)` | 文字列判定 |
| `isSymbol(value)` | Symbol判定 |
| `isUndefined(value)` | undefined判定 |

#### 2.2.4 その他のユーティリティ
| 関数 | 説明 |
|------|------|
| `debuglog(section)` | デバッグログ関数作成 |
| `deprecate(fn, msg)` | 非推奨警告ラッパー |
| `inherits(constructor, superConstructor)` | プロトタイプ継承 |
| `getSystemErrorName(err)` | システムエラー名取得 |
| `isDeepStrictEqual(val1, val2)` | 深い等価判定 |
| `parseArgs(config)` | コマンドライン引数解析 |
| `parseEnv(content)` | 環境変数形式解析 |
| `toUSVString(string)` | USV文字列変換 |

#### 2.2.5 util.types サブモジュール
| 関数 | 説明 |
|------|------|
| `isTypedArray(value)` | TypedArray判定 |
| `isUint8Array(value)` | Uint8Array判定 |
| `isArrayBuffer(value)` | ArrayBuffer判定 |
| `isSharedArrayBuffer(value)` | SharedArrayBuffer判定 |
| `isPromise(value)` | Promise判定 |
| `isMap(value)` | Map判定 |
| `isSet(value)` | Set判定 |
| `isWeakMap(value)` | WeakMap判定 |
| `isWeakSet(value)` | WeakSet判定 |
| `isRegExp(value)` | RegExp判定 |
| `isDate(value)` | Date判定 |
| `isNativeError(value)` | ネイティブError判定 |
| `isProxy(value)` | Proxy判定 |

## 3. 技術設計

### 3.1 アーキテクチャ

```
┌─────────────────────────────────────────────────────────────┐
│                    JavaScript Layer                          │
│  ┌─────────────────────────────────────────────────────────┐│
│  │  node:util (src/js/node/util.ts)                        ││
│  │  ┌───────────────────────────────────────────────────┐  ││
│  │  │  Core Functions                                   │  ││
│  │  │  - format, formatWithOptions, inspect             │  ││
│  │  │  - promisify, callbackify, deprecate              │  ││
│  │  │  - debuglog, inherits, isDeepStrictEqual          │  ││
│  │  └───────────────────────────────────────────────────┘  ││
│  │  ┌───────────────────────────────────────────────────┐  ││
│  │  │  util.types (require("node:util/types"))          │  ││
│  │  │  - isTypedArray, isPromise, isMap, isSet, etc.    │  ││
│  │  └───────────────────────────────────────────────────┘  ││
│  └─────────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────────┘
                            │
┌───────────────────────────▼─────────────────────────────────┐
│                     Native Layer                             │
│  ┌─────────────────────────────────────────────────────────┐│
│  │  node_util_binding.zig                                   ││
│  │  - internalErrorName() - システムエラー名変換            ││
│  │  - extractedSplitNewLines() - 改行分割                  ││
│  │  - normalizeEncoding() - エンコーディング正規化          ││
│  │  - parseEnv() - 環境変数形式解析                        ││
│  └─────────────────────────────────────────────────────────┘│
│  ┌─────────────────────────────────────────────────────────┐│
│  │  parse_args.zig                                          ││
│  │  - parseArgs() - コマンドライン引数解析                  ││
│  └─────────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────────┘
```

### 3.2 クラス構成

```
util
├── Formatting
│   ├── format(format, ...args)
│   ├── formatWithOptions(options, format, ...args)
│   ├── inspect(object, options)
│   ├── styleText(format, text)
│   └── stripVTControlCharacters(str)
├── Async Utilities
│   ├── promisify(fn)
│   ├── callbackify(fn)
│   └── aborted(signal, resource)
├── Type Checking (deprecated)
│   ├── isArray, isBoolean, isBuffer, etc.
│   └── isError, isFunction, isNull, etc.
├── Debug/Development
│   ├── debuglog(section)
│   ├── debug (alias for debuglog)
│   ├── deprecate(fn, msg)
│   └── getSystemErrorName(err)
├── Object Utilities
│   ├── inherits(ctor, superCtor)
│   ├── isDeepStrictEqual(val1, val2)
│   └── _extend(origin, add)
├── Parsing
│   ├── parseArgs(config)
│   ├── parseEnv(content)
│   └── toUSVString(string)
├── Classes
│   ├── TextDecoder
│   ├── TextEncoder
│   ├── MIMEType
│   └── MIMEParams
└── types (submodule)
    ├── isTypedArray, isUint8Array, isArrayBuffer
    ├── isPromise, isMap, isSet, isWeakMap, isWeakSet
    ├── isRegExp, isDate, isNativeError
    └── isProxy, isEventTarget, etc.
```

### 3.3 処理フロー

#### promisify処理フロー
```
1. util.promisify(original)
   ├── validateFunction(original)
   └── 新しい関数を作成して返却

2. 返却された関数の呼び出し
   ├── Promise作成
   ├── 元の関数を呼び出し
   │   └── コールバック引数追加
   └── コールバック実行時
       ├── エラー → reject(error)
       └── 成功 → resolve(result)
```

#### inspect処理フロー
```
1. util.inspect(object, options)
   ├── オプション正規化
   │   ├── depth (デフォルト: 2)
   │   ├── colors (デフォルト: false)
   │   ├── maxArrayLength (デフォルト: 100)
   │   └── etc.
   └── オブジェクト走査
       ├── プリミティブ → 文字列化
       ├── 配列 → 要素列挙
       ├── オブジェクト → プロパティ列挙
       └── 特殊オブジェクト → カスタム表示
```

#### debuglog処理フロー
```
1. util.debuglog(section)
   ├── NODE_DEBUG環境変数確認
   ├── sectionがマッチするか判定
   └── マッチする場合
       └── デバッグ出力関数を返却
   └── マッチしない場合
       └── 空関数を返却

2. debuglog関数の呼び出し
   └── console.error(section, pid, message)
```

## 4. コードリーディングガイド

### 4.1 推奨読解順序

1. **エクスポートとインポート** (`src/js/node/util.ts`)
   - **1-8行目**: types, utl(inspect), promisify等のインポート
   - **10-11行目**: internalErrorName, parseEnvのZig関数インポート

2. **コア関数**
   - **18-26行目**: isBuffer, isFunction関数
   - **30行目**: inspect（internal/util/inspectから）
   - **31行目**: formatWithOptions, format

3. **debuglog実装**
   - **35-61行目**: debuglog関数とNODE_DEBUG環境変数処理

4. **型チェック関数**
   - **63-104行目**: isBoolean, isNull, isNumber等の基本型チェック

5. **callbackify実装**
   - **148-193行目**: async関数をコールバック形式に変換

6. **styleText実装**
   - **198-222行目**: ANSIカラーコード適用

7. **getSystemErrorName実装**
   - **224-228行目**: internalErrorName Zig関数呼び出し

8. **aborted実装**
   - **240-283行目**: AbortSignal待機のPromise

9. **ネイティブ関数** (`src/bun.js/node/node_util_binding.zig`)
   - **1-99行目**: internalErrorName - エラーコード→名前変換
   - **109-195行目**: extractedSplitNewLinesFastPath
   - **197-205行目**: normalizeEncoding
   - **207-226行目**: parseEnv

### 4.2 プログラム呼び出し階層図

```
require('node:util')
└── src/js/node/util.ts
    ├── require('node:util/types')
    ├── require('internal/util/inspect')
    ├── require('internal/promisify')
    ├── require('internal/validators')
    └── $newZigFunction()
        ├── 'node_util_binding.zig' → internalErrorName
        ├── 'node_util_binding.zig' → parseEnv
        └── 'parse_args.zig' → parseArgs

util.promisify(fn)
├── validateFunction(fn)
└── return promisified function
    └── 呼び出し時
        ├── Promise作成
        ├── fn(...args, callback)
        └── callback(err, result)
            ├── err → reject
            └── result → resolve

util.inspect(object, options)
└── internal/util/inspect
    └── オブジェクト走査・文字列化

util.getSystemErrorName(err)
└── [native] internalErrorName(err)
    └── エラーコード→文字列マッピング
```

### 4.3 データフロー図

```
[promisify]
┌─────────────────┐     ┌──────────────────┐     ┌─────────────────┐
│ Callback-style  │────>│ util.promisify() │────>│ Promise-style   │
│ function        │     │                  │     │ function        │
│                 │     │                  │     │                 │
│ fn(arg, cb)     │     │ Wrapper creation │     │ fn(arg)         │
│ cb(err, result) │     │                  │     │ .then(result)   │
└─────────────────┘     └──────────────────┘     └─────────────────┘


[inspect]
┌─────────────────┐     ┌──────────────────┐     ┌─────────────────┐
│ JavaScript      │────>│ util.inspect()   │────>│ Formatted       │
│ Object          │     │                  │     │ String          │
│                 │     │ - depth          │     │                 │
│ { a: 1, b: 2 }  │     │ - colors         │     │ "{ a: 1, ... }" │
└─────────────────┘     └──────────────────┘     └─────────────────┘


[debuglog]
┌─────────────────┐     ┌──────────────────┐     ┌─────────────────┐
│ NODE_DEBUG=http │────>│ util.debuglog()  │────>│ Debug Function  │
│ (env variable)  │     │                  │     │                 │
│                 │     │ Section: 'http'  │     │ console.error() │
│                 │     │ Match: Yes       │     │ or no-op        │
└─────────────────┘     └──────────────────┘     └─────────────────┘
```

### 4.4 関連ファイル一覧

| ファイルパス | 種別 | 役割 |
|-------------|------|------|
| `src/js/node/util.ts` | TypeScript | node:utilモジュールのメイン実装（339行） |
| `src/bun.js/node/node_util_binding.zig` | Zig | internalErrorName, parseEnv等（238行） |
| `src/bun.js/parse_args.zig` | Zig | parseArgs実装 |
| `src/js/internal/util/inspect.ts` | TypeScript | inspect関数実装 |
| `src/js/internal/promisify.ts` | TypeScript | promisify関数実装 |
| `src/js/internal/util/deprecate.ts` | TypeScript | deprecate関数実装 |
| `src/js/internal/util/mime.ts` | TypeScript | MIMEType, MIMEParams実装 |
| `src/js/node/util/types.ts` | TypeScript | util.types実装 |

## 5. 設計上の考慮事項

### 5.1 Node.js互換性
- Node.jsのutil APIと完全互換を目指す
- 非推奨関数も互換性のため維持
- util.typesサブモジュールの完全実装

### 5.2 パフォーマンス最適化
- Zigネイティブ関数による高速処理
- internalErrorName: エラーコード直接マッピング
- parseEnv: 効率的な環境変数解析

### 5.3 デバッグ支援
- NODE_DEBUG環境変数によるログ制御
- inspect()の豊富なオプション
- deprecate()による移行支援

## 6. テスト方針

### 6.1 単体テスト
- format(), inspect()の各種フォーマットテスト
- promisify(), callbackify()の変換テスト
- 型チェック関数の正確性テスト

### 6.2 統合テスト
- NODE_DEBUGによるdebuglog動作テスト
- parseArgsのコマンドライン解析テスト
- isDeepStrictEqualの複雑なオブジェクトテスト

### 6.3 互換性テスト
- Node.jsのutilテストスイートの実行
- util.typesの全関数テスト

## 7. 変更履歴

| 日付 | バージョン | 変更内容 |
|------|-----------|---------|
| 2026-01-27 | 1.0 | 初版作成 |
