# 通知設計書 9-クライアント接続通知

## 概要

本ドキュメントは、Bun の DevServer 機能において、クライアント（ブラウザ）が WebSocket 経由で接続した際に、Inspector（デバッガ）に対して送信される接続通知の設計を記載する。

### 本通知の処理概要

この通知は、ブラウザクライアントが DevServer の HMR エンドポイントに WebSocket 接続を確立した際に、Inspector フロントエンド（Bun のデバッガツール）に対してクライアント接続を通知するサーバーサイドの内部通知である。これにより、デバッグツールでどのクライアントが接続しているかを追跡できる。

**業務上の目的・背景**：開発者がデバッガを使用している場合、どのブラウザタブ/ウィンドウが DevServer に接続しているかを把握することは重要である。`notifyClientConnected` によりクライアント接続を Inspector に通知することで、デバッグツールがアクティブなクライアント一覧を表示し、特定のクライアントに対してデバッグ操作を行えるようにする。

**通知の送信タイミング**：`HmrSocket.onOpen` 関数内で、WebSocket 接続が確立され、バージョンメッセージの送信が成功した後（`send_status != .dropped`）に送信される。

**通知の受信者**：Inspector フロントエンド（Bun のデバッガ UI）。Inspector が有効な場合（`dev.inspector()` が non-null）のみ通知が送信される。

**通知内容の概要**：DevServer ID と接続 ID（connection ID）が通知される。接続 ID は `BunFrontendDevServerAgent.nextConnectionID()` によって生成される一意の識別子。

**期待されるアクション**：Inspector フロントエンドがクライアント接続リストを更新し、開発者がデバッグ対象のクライアントを選択できるようにする。

## 通知種別

内部通知（Inspector プロトコル）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（C++ 関数呼び出し） |
| 優先度 | 中（デバッグ機能） |
| リトライ | 無 |

### 送信先決定ロジック

- `dev.inspector()` が non-null の場合のみ通知が送信される
- Inspector が無効な場合は通知はスキップされる
- 接続 ID は `agent.nextConnectionID()` で生成され、`HmrSocket.inspector_connection_id` に保存される

## 通知テンプレート

### 関数シグネチャ

```zig
pub fn notifyClientConnected(
    this: *const BunFrontendDevServerAgent,
    devServerId: DebuggerId,
    connectionId: i32
) void
```

### C++ バインディング

```cpp
extern "c" fn InspectorBunFrontendDevServerAgent__notifyClientConnected(
    agent: *InspectorBunFrontendDevServerAgentHandle,
    devServerId: i32,
    connectionId: i32
) void
```

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| `devServerId` | DevServer の識別子 | `dev.inspector_server_id` | Yes |
| `connectionId` | クライアント接続の識別子 | `agent.nextConnectionID()` | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| WebSocket 接続 | `HmrSocket.onOpen` | `send_status != .dropped` かつ Inspector 有効 | クライアントが正常に接続したとき |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| Inspector 無効 | `dev.inspector()` が null の場合 |
| バージョン送信失敗 | `send_status == .dropped` の場合 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[クライアント WebSocket 接続] --> B[HmrSocket.onOpen 呼び出し]
    B --> C[バージョンメッセージ送信]
    C --> D{send_status?}
    D -->|.dropped| E[終了]
    D -->|success| F{Inspector 有効?}
    F -->|No| G[inspector_connection_id = -1 のまま]
    F -->|Yes| H[nextConnectionID 取得]
    H --> I[inspector_connection_id に保存]
    I --> J[notifyClientConnected 呼び出し]
    J --> K[C++ バインディング経由で通知]
    G --> L[終了]
    K --> L
    E --> L
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（メモリ上のデータ構造を使用）

### サーバーサイドデータ構造

| 変数名 | 型 | 用途 | 定義場所 |
|--------|------|------|---------|
| `inspector_connection_id` | `i32` | クライアント接続 ID（-1 は未設定） | `HmrSocket` |
| `next_inspector_connection_id` | `i32` | 次の接続 ID | `BunFrontendDevServerAgent` |
| `inspector_server_id` | `DebuggerId` | DevServer ID | `DevServer` |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| Inspector ハンドル null | Inspector が初期化されていない | 通知をスキップ |
| バージョン送信失敗 | WebSocket 送信バッファフル | 接続処理を中断、通知なし |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 該当なし |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし（開発時に随時発生）

## セキュリティ考慮事項

- この通知は開発環境でのみ使用されることを想定している
- Inspector は通常 localhost からのみアクセス可能
- 接続 ID は単純なカウンタベースの識別子で、セキュリティトークンではない
- 本番環境では Inspector 機能自体が無効化されるべきである

## 備考

- `inspector_connection_id` の初期値は `-1` で、これは「未設定」を意味する
- 接続 ID は `next_inspector_connection_id` をインクリメント（+= 1）して生成される
- オーバーフロー時は wrap around（%=）される
- 同じ接続 ID が `notifyClientDisconnected` でも使用される

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Inspector 通知に関連するデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | HmrSocket.zig | `src/bake/DevServer/HmrSocket.zig` | 12行目の `inspector_connection_id: i32 = -1` で接続 ID 管理を理解する |
| 1-2 | InspectorBunFrontendDevServerAgent.zig | `src/bun.js/api/server/InspectorBunFrontendDevServerAgent.zig` | 24-26行目の `BunFrontendDevServerAgent` 構造体で ID 管理を理解する |

**読解のコツ**: `inspector_connection_id` が `-1` の場合は Inspector への通知が行われていないことを意味する。

#### Step 2: 接続処理を理解する

WebSocket 接続時の処理フローを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | HmrSocket.zig | `src/bake/DevServer/HmrSocket.zig` | 30-41行目の `onOpen` 関数で接続処理を理解する |

**主要処理フロー**:
1. **31行目**: バージョンメッセージ送信
2. **34行目**: `send_status != .dropped` チェック
3. **36行目**: `dev.inspector()` で Inspector 取得
4. **37行目**: `agent.nextConnectionID()` で接続 ID 生成
5. **38行目**: `agent.notifyClientConnected()` で通知

#### Step 3: 通知関数の実装を理解する

`notifyClientConnected` 関数の実装を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | InspectorBunFrontendDevServerAgent.zig | `src/bun.js/api/server/InspectorBunFrontendDevServerAgent.zig` | 38-42行目の `notifyClientConnected` 関数 |
| 3-2 | InspectorBunFrontendDevServerAgent.zig | `src/bun.js/api/server/InspectorBunFrontendDevServerAgent.zig` | 3行目の extern 宣言で C++ バインディングを理解 |

**主要処理フロー**:
- **39行目**: `if (this.handle) |handle|` で Inspector ハンドルチェック
- **40行目**: `handle.notifyClientConnected(devServerId.get(), connectionId)` で C++ 関数呼び出し

### プログラム呼び出し階層図

```
クライアント WebSocket 接続
    │
    ▼
HmrSocket.onOpen(ws)
    │
    ├─ ws.send(MessageId.version + configuration_hash_key)
    │
    ├─ if (send_status != .dropped)
    │      │
    │      └─ if (dev.inspector()) |agent|
    │             │
    │             ├─ agent.nextConnectionID()
    │             │      └─ next_inspector_connection_id += 1
    │             │
    │             ├─ inspector_connection_id = id
    │             │
    │             └─ agent.notifyClientConnected(server_id, connection_id)
    │                    │
    │                    └─ handle.notifyClientConnected() [C++ 呼び出し]
    │                           │
    │                           └─ Inspector フロントエンドに通知
    │
    └─ underlying = ws
```

### データフロー図

```
[入力]                    [処理]                         [出力]

クライアント
WebSocket接続 ─────────▶ HmrSocket.onOpen()
                              │
                              ▼
                       バージョンメッセージ送信
                              │
                              ▼
                       send_status チェック
                              │
                    ┌─────────┴─────────┐
                    ▼                   ▼
              [dropped]            [success]
                    │                   │
                    ▼                   ▼
                 終了            dev.inspector() チェック
                                        │
                              ┌─────────┴─────────┐
                              ▼                   ▼
                         [null]              [agent]
                              │                   │
                              ▼                   ▼
                           終了           nextConnectionID()
                                                │
                                                ▼
                                        notifyClientConnected()
                                                │
                                                ▼
                                        C++ バインディング
                                                │
                                                ▼
                                     Inspector フロントエンド通知
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| HmrSocket.zig | `src/bake/DevServer/HmrSocket.zig` | ソース | WebSocket 接続処理、`onOpen` 関数 |
| InspectorBunFrontendDevServerAgent.zig | `src/bun.js/api/server/InspectorBunFrontendDevServerAgent.zig` | ソース | `notifyClientConnected` 関数、ID 管理 |
| InspectorBunFrontendDevServerAgent.cpp | `src/bun.js/bindings/InspectorBunFrontendDevServerAgent.cpp` | ソース | C++ バインディング実装 |
| InspectorBunFrontendDevServerAgent.h | `src/bun.js/bindings/InspectorBunFrontendDevServerAgent.h` | ソース | C++ ヘッダー |
| DevServer.zig | `src/bake/DevServer.zig` | ソース | `inspector_server_id` の管理 |
