# コードリーディングガイドライン

本ガイドラインはプロジェクト(CICS GenApp)のコードを読み解けるようにするためのリファレンスです。

---

## 目次

1. [COBOLの基本構造](#1-cobolの基本構造)
2. [データ型とPIC句](#2-データ型とpic句)
3. [プロジェクト固有の命名規則](#3-プロジェクト固有の命名規則)
4. [CICSコマンドの読み方](#4-cicsコマンドの読み方)
5. [DB2/SQL文の読み方](#5-db2sql文の読み方)
6. [コピーブック（共通定義）](#6-コピーブック共通定義)
7. [処理フローの追い方](#7-処理フローの追い方)
8. [頻出パターンと定型句](#8-頻出パターンと定型句)
9. [ファイル別リファレンス](#9-ファイル別リファレンス)
10. [業務フロー追跡の実践例](#10-業務フロー追跡の実践例)
11. [設計書の参照順序ガイド](#11-設計書の参照順序ガイド)

---

## 1. COBOLの基本構造

COBOLプログラムは4つの「DIVISION（部）」で構成されます。

```cobol
IDENTIFICATION DIVISION.    ← プログラムの識別情報
ENVIRONMENT DIVISION.       ← 環境設定（省略されることが多い）
DATA DIVISION.              ← データ定義（変数宣言に相当）
PROCEDURE DIVISION.         ← 処理ロジック（関数本体に相当）
```

### 各DIVISIONの役割

| DIVISION | 役割 | 現代言語での対応 |
|----------|------|------------------|
| IDENTIFICATION | プログラム名等の宣言 | クラス名・ファイル名 |
| ENVIRONMENT | 外部ファイル等の設定 | 設定ファイル・環境変数 |
| DATA | 変数・構造体の定義 | 型定義・変数宣言 |
| PROCEDURE | 実際の処理ロジック | メソッド・関数本体 |

### DATA DIVISIONのセクション

```cobol
DATA DIVISION.
    WORKING-STORAGE SECTION.    ← ローカル変数（プログラム内で使用）
    LINKAGE SECTION.            ← 外部から渡されるパラメータ
```

### 実例: lgupdb01.cbl の構造

**ファイル**: `base/src/lgupdb01.cbl`（ポリシー更新のDB2アクセス）

```cobol
* 1行目: SQL前処理指示（DB2使用時に必要）
      PROCESS SQL

* 9-10行目: IDENTIFICATION DIVISION
      IDENTIFICATION DIVISION.
      PROGRAM-ID. LGUPDB01.          ← プログラム名の宣言

* 11-12行目: ENVIRONMENT DIVISION
      ENVIRONMENT DIVISION.
      CONFIGURATION SECTION.          ← 最小限の設定のみ

* 14-16行目: DATA DIVISION開始
      DATA DIVISION.
      WORKING-STORAGE SECTION.        ← ここから変数定義

* 45行目以降: LINKAGE SECTION
      LINKAGE SECTION.                ← 外部から渡されるパラメータ
      01  DFHCOMMAREA.                ← CICSが渡すCOMMarea

* 52行目以降: PROCEDURE DIVISION
      PROCEDURE DIVISION.             ← ここから処理ロジック
      MAINLINE SECTION.               ← メイン処理の開始
```

**ポイント**:
- `PROCESS SQL`は1行目に記述（DB2プリコンパイラへの指示）
- `PROGRAM-ID`はファイル名と一致（`LGUPDB01`）
- 本プロジェクトでは`ENVIRONMENT DIVISION`は最小限

---

## 2. データ型とPIC句

COBOLでは変数を「PIC句」で定義します。PICは「PICTURE」の略です。

### 基本的なPIC句

| PIC句 | 意味 | 例 |
|-------|------|-----|
| `PIC X(n)` | 英数字n桁 | `PIC X(10)` → 10文字の文字列 |
| `PIC 9(n)` | 数字n桁 | `PIC 9(4)` → 4桁の数字 |
| `PIC S9(n)` | 符号付き数字 | `PIC S9(4)` → -9999〜9999 |
| `PIC S9(n) COMP` | バイナリ整数 | `PIC S9(9) COMP` → 32bit整数 |

### 具体例

```cobol
01  WS-HEADER.
    03 WS-EYECATCHER            PIC X(16).     ← 16文字の文字列
    03 WS-TRANSID               PIC X(4).      ← 4文字の文字列
    03 WS-TASKNUM               PIC 9(7).      ← 7桁の数字
    03 WS-CALEN                 PIC S9(4) COMP. ← 2バイト整数
```

### レベル番号の意味

```cobol
01  親構造体名.           ← 最上位レベル（構造体のルート）
    03 子要素1.           ← 第2レベル
       05 孫要素1.        ← 第3レベル
       05 孫要素2.
    03 子要素2.
```

- `01` : 最上位のレコード（構造体）
- `03`, `05`, `07`... : ネスト構造を表す（奇数を使用）
- `77` : 単独の変数（構造体に属さない）
- `88` : 条件名（定数の別名）

### REDEFINESの理解

```cobol
03 CA-REQUEST-SPECIFIC      PIC X(32482).
03 CA-CUSTOMER-REQUEST REDEFINES CA-REQUEST-SPECIFIC.
   05 CA-FIRST-NAME         PIC X(10).
   05 CA-LAST-NAME          PIC X(20).
```

`REDEFINES`は同じメモリ領域を別の構造で参照します（C言語のunionに相当）。
上記では、`CA-REQUEST-SPECIFIC`の先頭30バイトを`CA-FIRST-NAME`と`CA-LAST-NAME`として参照できます。

### 実例: lgpolicy.cpy のデータ定義

**ファイル**: `base/src/lgpolicy.cpy`（ポリシー関連の共通データ構造）

```cobol
* 16-29行目: 各データ長の定義（WS-POLICY-LENGTHS）
      01  WS-POLICY-LENGTHS.
          03 WS-CUSTOMER-LEN    PIC S9(4) COMP VALUE +72.   ← 顧客データ長
          03 WS-POLICY-LEN      PIC S9(4) COMP VALUE +72.   ← ポリシーデータ長
          03 WS-ENDOW-LEN       PIC S9(4) COMP VALUE +52.   ← 養老保険データ長
          03 WS-HOUSE-LEN       PIC S9(4) COMP VALUE +58.   ← 住宅保険データ長
          03 WS-MOTOR-LEN       PIC S9(4) COMP VALUE +65.   ← 自動車保険データ長

* 31-40行目: 顧客データ構造（DB2-CUSTOMER）
      01  DB2-CUSTOMER.
          03 DB2-FIRSTNAME      PIC X(10).    ← 名（10文字）
          03 DB2-LASTNAME       PIC X(20).    ← 姓（20文字）
          03 DB2-DATEOFBIRTH    PIC X(10).    ← 生年月日（YYYY-MM-DD形式）
          03 DB2-HOUSENAME      PIC X(20).    ← 住所1
          03 DB2-HOUSENUMBER    PIC X(4).     ← 住所番号
          03 DB2-POSTCODE       PIC X(8).     ← 郵便番号
          03 DB2-PHONE-MOBILE   PIC X(20).    ← 携帯電話
          03 DB2-PHONE-HOME     PIC X(20).    ← 自宅電話
          03 DB2-EMAIL-ADDRESS  PIC X(100).   ← メールアドレス

* 72-81行目: 自動車保険データ構造（DB2-MOTOR）
      01  DB2-MOTOR.
          03 DB2-M-MAKE         PIC X(15).    ← 車メーカー
          03 DB2-M-MODEL        PIC X(15).    ← 車種
          03 DB2-M-VALUE        PIC 9(6).     ← 車両価値（6桁数値）
          03 DB2-M-REGNUMBER    PIC X(7).     ← 登録番号
          03 DB2-M-COLOUR       PIC X(8).     ← 色
          03 DB2-M-CC           PIC 9(4).     ← 排気量（cc）
          03 DB2-M-MANUFACTURED PIC X(10).    ← 製造日
          03 DB2-M-PREMIUM      PIC 9(6).     ← 保険料
          03 DB2-M-ACCIDENTS    PIC 9(6).     ← 事故回数
```

**ポイント**:
- `PIC X(n)` は文字列フィールド
- `PIC 9(n)` は数値フィールド（表示形式）
- `PIC S9(4) COMP` は符号付き2バイト整数
- `VALUE +72` で初期値を設定
- フィールド名の接頭辞で用途を識別（`DB2-`はDB2用、`WS-`は作業用）

### 実例: lgstsq.cbl のREDEFINES

**ファイル**: `base/src/lgstsq.cbl`（一時記憶キュー処理）

```cobol
* 25-37行目: REDEFINESによる同一メモリ領域の再定義
      01  WRITE-MSG.
        03 WRITE-MSG-SYSID      PIC X(4).     ← システムID
        03 FILLER-X             PIC X Value SPACES.
        03 WRITE-MSG-MSG        PIC X(90).    ← メッセージ本文

      01  FILLER REDEFINES WRITE-MSG.         ← 同じ95バイトを別構造で参照
        03 FILLER               PIC X(5).     ← 先頭5バイト（読み飛ばし）
        03 FILLER               PIC X(7).     ← 次の7バイト（読み飛ばし）
        03 WRITE-MSG-REST       PIC X(83).    ← 残り83バイト

      01  STSQ.
        03  STSQ-NAME           PIC X(8).     ← キュー名（8文字）

      01  FILLER REDEFINES STSQ.              ← 同じ8バイトを別構造で参照
        03 FILLER               PIC X(4).     ← 先頭4文字は固定
        03 STSQ-EXT             PIC X(4).     ← 後半4文字を拡張子として使用
```

**ポイント**:
- `REDEFINES`で同じメモリ領域に別名でアクセス
- `FILLER`は名前なしの埋め草（参照不要な領域に使用）
- メッセージ解析や部分取得に活用

---

## 3. プロジェクト固有の命名規則

### プログラム名の構成

```
LG[機能][対象][種類][番号]
```

| 位置 | 意味 | 値 |
|------|------|-----|
| LG | 接頭辞 | 固定 |
| 機能 | 処理種別 | A=追加, D=削除, I=照会, U=更新 |
| 対象 | 処理対象 | C=顧客, P=ポリシー |
| 種類 | 処理層 | US=UI, DB=DB2, VS=VSAM, OL=ロジック |
| 番号 | 連番 | 01, 02... |

### 例

| プログラム名 | 解読 |
|-------------|------|
| `LGICDB01` | LG + I(照会) + C(顧客) + DB(DB2) + 01 → 顧客照会のDB2アクセス |
| `LGAPOL01` | LG + A(追加) + P(ポリシー) + OL(ロジック) + 01 → ポリシー追加のビジネスロジック |
| `LGUCVS01` | LG + U(更新) + C(顧客) + VS(VSAM) + 01 → 顧客更新のVSAM処理 |

### 変数名の接頭辞

| 接頭辞 | 意味 | 使用場所 |
|--------|------|----------|
| `WS-` | Working Storage | ローカル変数 |
| `CA-` | CommArea | プログラム間通信データ |
| `DB2-` | DB2関連 | DB2ホスト変数 |
| `EM-` | Error Message | エラーメッセージ用 |

### 実例: 命名規則から機能を読み解く

**プロジェクト内の全プログラムを命名規則で分類**:

| プログラム群 | 命名パターン | 機能 | ファイル例 |
|-------------|-------------|------|-----------|
| 顧客照会 | `LGIC*01` | Inquire Customer | `lgicus01.cbl`, `lgicdb01.cbl`, `lgicvs01.cbl` |
| 顧客追加 | `LGAC*01` | Add Customer | `lgacus01.cbl`, `lgacdb01.cbl`, `lgacvs01.cbl` |
| 顧客更新 | `LGUC*01` | Update Customer | `lgucus01.cbl`, `lgucdb01.cbl`, `lgucvs01.cbl` |
| ポリシー照会 | `LGIP*01` | Inquire Policy | `lgipol01.cbl`, `lgipdb01.cbl`, `lgipvs01.cbl` |
| ポリシー追加 | `LGAP*01` | Add Policy | `lgapol01.cbl`, `lgapdb01.cbl`, `lgapvs01.cbl` |
| ポリシー更新 | `LGUP*01` | Update Policy | `lgupol01.cbl`, `lgupdb01.cbl`, `lgupvs01.cbl` |
| ポリシー削除 | `LGDP*01` | Delete Policy | `lgdpol01.cbl`, `lgdpdb01.cbl`, `lgdpvs01.cbl` |

### 実例: lgupdb01.cbl の変数命名

**ファイル**: `base/src/lgupdb01.cbl`（ポリシー更新DB2アクセス）

```cobol
* 22-31行目: WS-（Working Storage）接頭辞の変数群
      01  WS-HEADER.                           ← 作業用ヘッダ構造
         03 WS-EYECATCHER    PIC X(16)         ← デバッグ用識別子
                              VALUE 'LGUPDB01------WS'.
         03 WS-TRANSID       PIC X(4).         ← トランザクションID
         03 WS-TERMID        PIC X(4).         ← 端末ID
         03 WS-TASKNUM       PIC 9(7).         ← タスク番号
         03 WS-ADDR-DFHCOMMAREA USAGE is POINTER. ← COMMareaポインタ
         03 WS-CALEN         PIC S9(4) COMP.   ← COMMarea長

* 39-51行目: EM-（Error Message）接頭辞の変数群
      01  ERROR-MSG.
         03 EM-DATE          PIC X(8).         ← エラー発生日
         03 EM-TIME          PIC X(6).         ← エラー発生時刻
         03 EM-VARIABLE.
           05 EM-CUSNUM      PIC X(10).        ← エラー時の顧客番号
           05 EM-POLNUM      PIC X(10).        ← エラー時のポリシー番号
           05 EM-SQLREQ      PIC X(16).        ← 実行中のSQL種別
           05 EM-SQLRC       PIC +9(5).        ← SQLリターンコード

* 86-98行目: DB2-（DB2ホスト変数）接頭辞の変数群
      01 DB2-IN-INTEGERS.
         03 DB2-CUSTOMERNUM-INT   PIC S9(9) COMP.  ← 顧客番号（整数型）
         03 DB2-POLICYNUM-INT     PIC S9(9) COMP.  ← ポリシー番号（整数型）
         03 DB2-BROKERID-INT      PIC S9(9) COMP.  ← ブローカーID
         03 DB2-PAYMENT-INT       PIC S9(9) COMP.  ← 支払額
         03 DB2-E-TERM-SINT       PIC S9(4) COMP.  ← 養老保険期間（短整数）
         03 DB2-M-VALUE-INT       PIC S9(9) COMP.  ← 自動車価値
```

**命名規則のまとめ**:
- `WS-*`: プログラム内でのみ使用する作業変数
- `EM-*`: エラーメッセージ構築用の変数
- `DB2-*`: DB2とのデータ受け渡し用変数
- `CA-*`: プログラム間でCOMMareaを通じて受け渡す変数
- `*-INT`/`*-SINT`: 整数型（Integer/Short Integer）を示す接尾辞

---

## 4. CICSコマンドの読み方

CICSコマンドは`EXEC CICS ... END-EXEC`で囲まれます。

### 頻出コマンド

#### LINK（他プログラムの呼び出し）
```cobol
EXEC CICS LINK PROGRAM('LGSTSQ')
          COMMAREA(ERROR-MSG)
          LENGTH(LENGTH OF ERROR-MSG)
END-EXEC.
```
→ `LGSTSQ`プログラムを呼び出し、`ERROR-MSG`をパラメータとして渡す

#### RETURN（呼び出し元への復帰）
```cobol
EXEC CICS RETURN END-EXEC.
```
→ 呼び出し元に制御を戻す

#### ABEND（異常終了）
```cobol
EXEC CICS ABEND ABCODE('LGCA') NODUMP END-EXEC
```
→ エラーコード'LGCA'で異常終了（ダンプなし）

#### ASKTIME / FORMATTIME（日時取得）
```cobol
EXEC CICS ASKTIME ABSTIME(WS-ABSTIME) END-EXEC
EXEC CICS FORMATTIME ABSTIME(WS-ABSTIME)
          MMDDYYYY(WS-DATE)
          TIME(WS-TIME)
END-EXEC
```
→ 現在時刻を取得してフォーマット

### EIB変数（CICS実行時情報）

| 変数名 | 意味 |
|--------|------|
| `EIBTRNID` | トランザクションID |
| `EIBTRMID` | 端末ID |
| `EIBTASKN` | タスク番号 |
| `EIBCALEN` | COMMareaの長さ |

### 実例: lgstsq.cbl のCICSコマンド

**ファイル**: `base/src/lgstsq.cbl`（一時記憶キュー処理）

```cobol
* 60-62行目: ASSIGN - システム情報の取得
      EXEC CICS ASSIGN SYSID(WRITE-MSG-SYSID)   ← システムIDを取得
           RESP(WS-RESP)                         ← 応答コードを格納
      END-EXEC.

* 64-66行目: ASSIGN - 呼び出し元プログラム名の取得
      EXEC CICS ASSIGN INVOKINGPROG(WS-INVOKEPROG)  ← 呼び出し元を取得
           RESP(WS-RESP)
      END-EXEC.

* 73-76行目: RECEIVE - 端末からデータ受信
      EXEC CICS RECEIVE INTO(WS-RECV)           ← 受信データ格納先
          LENGTH(WS-RECV-LEN)                    ← 受信長
          RESP(WS-RESP)                          ← 応答コード
      END-EXEC

* 94-99行目: WRITEQ TD - 一時データキューへ書き込み
      EXEC CICS WRITEQ TD QUEUE(STDQ-NAME)      ← キュー名（CSMT）
                FROM(WRITE-MSG)                  ← 書き込みデータ
                RESP(WS-RESP)
                LENGTH(WS-RECV-LEN)              ← データ長
      END-EXEC.

* 105-111行目: WRITEQ TS - 一時記憶キューへ書き込み
      EXEC CICS WRITEQ TS QUEUE(STSQ-NAME)      ← キュー名（GENAERRS等）
                FROM(WRITE-MSG)
                RESP(WS-RESP)
                NOSUSPEND                        ← 空き待ちしない
                LENGTH(WS-RECV-LEN)
      END-EXEC.

* 114-119行目: SEND TEXT - 端末へテキスト送信
      EXEC CICS SEND TEXT FROM(FILLER-X)        ← 送信データ
       WAIT                                      ← 送信完了を待つ
       ERASE                                     ← 画面クリア
       LENGTH(1)                                 ← 送信長
       FREEKB                                    ← キーボード解放
      END-EXEC.

* 121-122行目: RETURN - 呼び出し元へ戻る
      EXEC CICS RETURN
      END-EXEC.
```

**CICSコマンドの構造**:
- `EXEC CICS [コマンド] [オプション...] END-EXEC`
- `RESP(変数)`: 応答コードを受け取る（エラーハンドリング用）
- `TD`（Transient Data）: 一時データキュー
- `TS`（Temporary Storage）: 一時記憶キュー

### 実例: lgacdb01.cbl のGET COUNTERコマンド

**ファイル**: `base/src/lgacdb01.cbl`（顧客追加DB2アクセス）

```cobol
* 199-211行目: 名前付きカウンタから一意番号を取得
      Obtain-CUSTOMER-Number.
         Exec CICS Get Counter(GENAcount)       ← カウンタ名（GENACUSTNUM）
                       Pool(GENApool)            ← プール名（GENA）
                       Value(LastCustNum)        ← 取得した値を格納
                       Resp(WS-RESP)             ← 応答コード
         End-Exec.

         If WS-RESP Not = DFHRESP(NORMAL)       ← 正常応答でない場合
           MOVE 'NO' TO LGAC-NCS                ← フラグをOFFに
           Initialize DB2-CUSTOMERNUM-INT       ← DB2自動採番モードへ
         ELSE
           Move LastCustNum To DB2-CUSTOMERNUM-INT
         End-If.
```

**ポイント**:
- `DFHRESP(NORMAL)`: CICSの正常応答を示す定数
- カップリングファシリティのカウンタで一意番号を生成
- 失敗時はDB2の`IDENTITY`機能にフォールバック

---

## 5. DB2/SQL文の読み方

SQL文は`EXEC SQL ... END-EXEC`で囲まれます。

### SELECT文の例

```cobol
EXEC SQL
    SELECT FIRSTNAME,
           LASTNAME,
           DATEOFBIRTH
    INTO  :CA-FIRST-NAME,
          :CA-LAST-NAME,
          :CA-DOB
    FROM CUSTOMER
    WHERE CUSTOMERNUMBER = :DB2-CUSTOMERNUMBER-INT
END-EXEC.
```

**ポイント**:
- `:変数名` はCOBOL変数（ホスト変数）を表す
- `INTO`句で結果を受け取る変数を指定
- SQLCODEで結果を判定

### SQLCODE判定

```cobol
Evaluate SQLCODE
  When 0        ← 正常終了
    MOVE '00' TO CA-RETURN-CODE
  When 100      ← データなし（NOT FOUND）
    MOVE '01' TO CA-RETURN-CODE
  When -913     ← デッドロック
    MOVE '01' TO CA-RETURN-CODE
  When Other    ← その他エラー
    MOVE '90' TO CA-RETURN-CODE
END-Evaluate.
```

### DB2データ型対応表

| DB2 | COBOL |
|-----|-------|
| SMALLINT | `PIC S9(4) COMP` |
| INTEGER | `PIC S9(9) COMP` |
| DATE | `PIC X(10)` |
| TIMESTAMP | `PIC X(26)` |
| VARCHAR(n) | `PIC X(n)` |

### 実例: lgacdb01.cbl のINSERT文

**ファイル**: `base/src/lgacdb01.cbl`（顧客追加DB2アクセス）

```cobol
* 222-244行目: カウンタ使用時のINSERT
      EXEC SQL
        INSERT INTO CUSTOMER
                  ( CUSTOMERNUMBER,       ← カラム名リスト
                    FIRSTNAME,
                    LASTNAME,
                    DATEOFBIRTH,
                    HOUSENAME,
                    HOUSENUMBER,
                    POSTCODE,
                    PHONEMOBILE,
                    PHONEHOME,
                    EMAILADDRESS )
           VALUES ( :DB2-CUSTOMERNUM-INT, ← ホスト変数（:で始まる）
                    :CA-FIRST-NAME,       ← COMMareaの値を挿入
                    :CA-LAST-NAME,
                    :CA-DOB,
                    :CA-HOUSE-NAME,
                    :CA-HOUSE-NUM,
                    :CA-POSTCODE,
                    :CA-PHONE-MOBILE,
                    :CA-PHONE-HOME,
                    :CA-EMAIL-ADDRESS )
      END-EXEC

* 245-249行目: エラーチェック
      IF SQLCODE NOT EQUAL 0              ← 0以外はエラー
        MOVE '90' TO CA-RETURN-CODE       ← エラーコード設定
        PERFORM WRITE-ERROR-MESSAGE       ← エラーログ出力
        EXEC CICS RETURN END-EXEC         ← 即時終了
      END-IF

* 251-282行目: DB2自動採番モード（IDENTITY使用）
      ELSE
        EXEC SQL
          INSERT INTO CUSTOMER
                    ( CUSTOMERNUMBER, ... )
             VALUES ( DEFAULT,            ← DB2が自動的に番号を割り当て
                      :CA-FIRST-NAME, ... )
        END-EXEC

* 280-282行目: 採番された値を取得
        EXEC SQL
          SET :DB2-CUSTOMERNUM-INT = IDENTITY_VAL_LOCAL()
        END-EXEC
      END-IF.

* 285行目: 結果をCOMMareaに格納
      MOVE DB2-CUSTOMERNUM-INT TO CA-CUSTOMER-NUM.
```

**ポイント**:
- `:変数名` でCOBOL変数をSQL文に埋め込む
- `DEFAULT` でDB2の自動採番を使用
- `IDENTITY_VAL_LOCAL()` で直前のINSERTで生成された値を取得

### 実例: lgupdb01.cbl のインジケータ変数

**ファイル**: `base/src/lgupdb01.cbl`（ポリシー更新DB2アクセス）

```cobol
* 106-111行目: NULL値を扱うためのインジケータ変数
      77  IND-BROKERID          PIC S9(4) COMP.  ← ブローカーID用
      77  IND-BROKERSREF        PIC S9(4) COMP.  ← ブローカー参照用
      77  IND-PAYMENT           PIC S9(4) COMP.  ← 支払額用
```

**インジケータ変数の使い方**:
```cobol
* SELECT文でNULL判定に使用
      EXEC SQL
        SELECT BROKERID, PAYMENT
        INTO :DB2-BROKERID-INT :IND-BROKERID,  ← 変数の後にインジケータ
             :DB2-PAYMENT-INT :IND-PAYMENT
        FROM POLICY
        WHERE ...
      END-EXEC.

* 判定: IND-BROKERID < 0 ならNULL値
      IF IND-BROKERID < 0
        MOVE ZEROS TO WS-BROKERID    ← NULL時のデフォルト値
      ELSE
        MOVE DB2-BROKERID-INT TO WS-BROKERID
      END-IF.
```

---

## 6. コピーブック（共通定義）

コピーブックは共通のデータ定義を格納したファイルです（C言語のヘッダファイルに相当）。

### インクルード方法

```cobol
COPY LGPOLICY.           ← lgpolicy.cpyをインクルード

EXEC SQL
    INCLUDE LGCMAREA     ← SQL経由でlgcmarea.cpyをインクルード
END-EXEC.
```

### 主要コピーブック

| ファイル | 用途 |
|----------|------|
| `lgcmarea.cpy` | COMMAREA構造（プログラム間通信用） |
| `lgpolicy.cpy` | ポリシー関連データ構造 |
| `soaic01.cpy` | SOAインターフェース定義 |

### lgcmarea.cpyの構造

```
CA-REQUEST-ID        : リクエスト識別子（6文字）
CA-RETURN-CODE       : 戻りコード（2桁）
CA-CUSTOMER-NUM      : 顧客番号（10桁）
CA-REQUEST-SPECIFIC  : リクエスト固有データ（32482バイト）
  └── REDEFINES で以下に分岐
      ├── CA-CUSTOMER-REQUEST  : 顧客データ
      ├── CA-POLICY-REQUEST    : ポリシーデータ
      │     └── REDEFINES で以下に分岐
      │         ├── CA-ENDOWMENT   : 養老保険
      │         ├── CA-HOUSE       : 住宅保険
      │         ├── CA-MOTOR       : 自動車保険
      │         └── CA-COMMERCIAL  : 企業保険
      └── CA-CUSTSECR-REQUEST  : セキュリティデータ
```

### 実例: コピーブックの使用方法

**lgicus01.cbl でのCOPY文使用（60-69行目）**:

```cobol
* 60行目: lgpolicy.cpyをインクルード（通常のCOPY文）
      COPY LGPOLICY.

* 使用可能になる変数:
*   WS-CUSTOMER-LEN, WS-POLICY-LEN 等の長さ定義
*   これらの変数はプログラム内で直接参照可能

* 68-69行目: lgcmarea.cpyをインクルード（LINKAGE SECTION内）
      LINKAGE SECTION.
      01  DFHCOMMAREA.
            COPY LGCMAREA.

* 使用可能になる変数:
*   CA-REQUEST-ID, CA-RETURN-CODE, CA-CUSTOMER-NUM 等
```

**lgicdb01.cbl でのSQL INCLUDEを使用（74行目、92-94行目）**:

```cobol
* 74行目: WORKING-STORAGE内でlgpolicy.cpyをSQL経由でインクルード
      COPY LGPOLICY.

* 92-94行目: LINKAGE SECTION内でlgcmarea.cpyをSQL経由でインクルード
      LINKAGE SECTION.
      01  DFHCOMMAREA.
          EXEC SQL
            INCLUDE LGCMAREA      ← SQL INCLUDEでコピーブックを読み込み
          END-EXEC.

* DB2プリコンパイラが処理するため、EXEC SQLで囲む
```

**コピーブック使い分けのルール**:
- `COPY xxx.`: 通常のCOBOLコピーブック読み込み
- `EXEC SQL INCLUDE xxx END-EXEC.`: DB2プリコンパイラがSQL文を含むコピーブックを処理

---

## 7. 処理フローの追い方

### 典型的な処理フロー

```
1. PROCEDURE DIVISION の最初のSECTIONから開始
2. PERFORM文で他のSECTION/PARAGRAPHを呼び出し
3. EXEC CICS LINKで他プログラムを呼び出し
4. EXEC CICS RETURNで呼び出し元に戻る
```

### PERFORM文の読み方

```cobol
PERFORM GET-CUSTOMER-INFO.        ← GET-CUSTOMER-INFOを1回実行
PERFORM PROCESS-DATA 5 TIMES.     ← PROCESS-DATAを5回実行
PERFORM UNTIL WS-EOF = 'Y'        ← WS-EOF='Y'になるまで繰り返し
    READ ...
END-PERFORM.
```

### 条件分岐

```cobol
IF EIBCALEN IS EQUAL TO ZERO
    MOVE ' NO COMMAREA RECEIVED' TO EM-VARIABLE
    PERFORM WRITE-ERROR-MESSAGE
    EXEC CICS ABEND ABCODE('LGCA') NODUMP END-EXEC
END-IF
```

### EVALUATE文（switch文相当）

```cobol
Evaluate SQLCODE
  When 0
    処理A
  When 100
    処理B
  When Other
    処理C
END-Evaluate.
```

### 実例: lgstsq.cbl の処理フロー完全解説

**ファイル**: `base/src/lgstsq.cbl`（一時記憶キュー処理プログラム）

このプログラムはエラーログを一時記憶キューに書き込む共通機能を提供します。

```cobol
* 55-56行目: プログラム開始
      PROCEDURE DIVISION.
      MAINLINE SECTION.

* 57-58行目: 変数初期化
         MOVE SPACES TO WRITE-MSG.
         MOVE SPACES TO WS-RECV.

* 60-66行目: システム情報取得
         EXEC CICS ASSIGN SYSID(WRITE-MSG-SYSID) ... END-EXEC.
         EXEC CICS ASSIGN INVOKINGPROG(WS-INVOKEPROG) ... END-EXEC.

* 68-80行目: 呼び出し方法による分岐
         IF WS-INVOKEPROG NOT = SPACES        ← 他プログラムからのLINK呼び出し
            MOVE 'C' To WS-FLAG
            MOVE COMMA-DATA TO WRITE-MSG-MSG  ← COMMareaからデータ取得
         ELSE                                  ← 端末から直接実行
            EXEC CICS RECEIVE INTO(WS-RECV) ... END-EXEC
            MOVE 'R' To WS-FLAG
            MOVE WS-RECV-DATA TO WRITE-MSG-MSG ← 端末入力からデータ取得
         END-IF.

* 82-88行目: キュー名の決定
         MOVE 'GENAERRS' TO STSQ-NAME.        ← デフォルトキュー名
         IF WRITE-MSG-MSG(1:2) = 'Q=' THEN    ← Q=nnnnパラメータがあれば
            MOVE WRITE-MSG-MSG(3:4) TO STSQ-EXT  ← キュー名を変更
            ...
         END-IF.

* 94-99行目: 一時データキュー(TDQ)へ書き込み
         EXEC CICS WRITEQ TD QUEUE(STDQ-NAME) ... END-EXEC.

* 105-111行目: 一時記憶キュー(TSQ)へ書き込み
         EXEC CICS WRITEQ TS QUEUE(STSQ-NAME) ... END-EXEC.

* 113-122行目: 端末呼び出し時のみ画面クリアして終了
         If WS-FLAG = 'R' Then
           EXEC CICS SEND TEXT ... END-EXEC.

         EXEC CICS RETURN END-EXEC.

* 124-126行目: 終了処理
      A-EXIT.
         EXIT.
         GOBACK.
```

**フロー図**:
```
開始
  ↓
変数初期化
  ↓
システム情報取得（ASSIGN）
  ↓
┌─ 呼び出し元の判定 ─┐
│                    │
↓                    ↓
LINK経由           端末直接
(COMMA-DATA使用)   (RECEIVE使用)
│                    │
└────────┬───────────┘
         ↓
キュー名決定（Q=パラメータ解析）
         ↓
TDQ書き込み（CSMT）
         ↓
TSQ書き込み（GENAERRS等）
         ↓
┌─ 呼び出し元の判定 ─┐
↓                    ↓
LINK経由           端末直接
(何もしない)       (SEND TEXT)
└────────┬───────────┘
         ↓
RETURN（呼び出し元へ戻る）
```

---

## 8. 頻出パターンと定型句

### プログラム開始時の初期化

```cobol
MAINLINE SECTION.
* 変数初期化
    INITIALIZE WS-HEADER.
* CICS実行情報の取得
    MOVE EIBTRNID TO WS-TRANSID.
    MOVE EIBTRMID TO WS-TERMID.
    MOVE EIBTASKN TO WS-TASKNUM.
```

### COMMareaチェック

```cobol
* COMMareaが渡されていない場合はエラー
IF EIBCALEN IS EQUAL TO ZERO
    EXEC CICS ABEND ABCODE('LGCA') NODUMP END-EXEC
END-IF
```

### 戻りコードの設定

```cobol
MOVE '00' TO CA-RETURN-CODE    ← 正常終了
MOVE '01' TO CA-RETURN-CODE    ← データなし
MOVE '90' TO CA-RETURN-CODE    ← エラー
MOVE '98' TO CA-RETURN-CODE    ← パラメータエラー
```

### エラーメッセージ出力

```cobol
WRITE-ERROR-MESSAGE.
    MOVE SQLCODE TO EM-SQLRC
    EXEC CICS ASKTIME ABSTIME(WS-ABSTIME) END-EXEC
    EXEC CICS FORMATTIME ... END-EXEC
    EXEC CICS LINK PROGRAM('LGSTSQ')
              COMMAREA(ERROR-MSG)
              LENGTH(LENGTH OF ERROR-MSG)
    END-EXEC.
```

### 実例: lgicdb01.cbl の定型パターン集

**ファイル**: `base/src/lgicdb01.cbl`（顧客照会DB2アクセス）

#### パターン1: プログラム開始時の初期化（102-112行目）

```cobol
      MAINLINE SECTION.
* 変数の初期化
         INITIALIZE WS-HEADER.

* CICS実行情報の取得（デバッグ・ログ用）
         MOVE EIBTRNID TO WS-TRANSID.    ← トランザクションID保存
         MOVE EIBTRMID TO WS-TERMID.     ← 端末ID保存
         MOVE EIBTASKN TO WS-TASKNUM.    ← タスク番号保存
```

#### パターン2: COMMareaの検証（118-128行目）

```cobol
* COMMareaが渡されていなければABEND
         IF EIBCALEN IS EQUAL TO ZERO
             MOVE ' NO COMMAREA RECEIVED' TO EM-VARIABLE
             PERFORM WRITE-ERROR-MESSAGE
             EXEC CICS ABEND ABCODE('LGCA') NODUMP END-EXEC
         END-IF

* 戻りコード初期化
         MOVE '00' TO CA-RETURN-CODE
         MOVE EIBCALEN TO WS-CALEN.
         SET WS-ADDR-DFHCOMMAREA TO ADDRESS OF DFHCOMMAREA.
```

#### パターン3: COMMarea長チェック（136-143行目）

```cobol
* 必要な長さを計算
         MOVE WS-CUSTOMER-LEN        TO WS-REQUIRED-CA-LEN
         ADD WS-CA-HEADERTRAILER-LEN TO WS-REQUIRED-CA-LEN

* 長さ不足ならエラー戻りコードを設定して終了
         IF EIBCALEN IS LESS THAN WS-REQUIRED-CA-LEN
           MOVE '98' TO CA-RETURN-CODE
           EXEC CICS RETURN END-EXEC
         END-IF
```

#### パターン4: SQLCODE判定（192-203行目）

```cobol
         Evaluate SQLCODE
           When 0                          ← 正常終了
             MOVE '00' TO CA-RETURN-CODE
           When 100                        ← データなし
             MOVE '01' TO CA-RETURN-CODE
           When -913                       ← デッドロック
             MOVE '01' TO CA-RETURN-CODE
           When Other                      ← その他のエラー
             MOVE '90' TO CA-RETURN-CODE
             PERFORM WRITE-ERROR-MESSAGE
             EXEC CICS RETURN END-EXEC
         END-Evaluate.
```

#### パターン5: エラーメッセージ書き込み（212-244行目）

```cobol
      WRITE-ERROR-MESSAGE.
* SQLCODEをメッセージに保存
         MOVE SQLCODE TO EM-SQLRC

* 日時を取得してフォーマット
         EXEC CICS ASKTIME ABSTIME(WS-ABSTIME) END-EXEC
         EXEC CICS FORMATTIME ABSTIME(WS-ABSTIME)
                   MMDDYYYY(WS-DATE)
                   TIME(WS-TIME)
         END-EXEC
         MOVE WS-DATE TO EM-DATE
         MOVE WS-TIME TO EM-TIME

* エラーログ書き込みプログラムを呼び出し
         EXEC CICS LINK PROGRAM('LGSTSQ')
                   COMMAREA(ERROR-MSG)
                   LENGTH(LENGTH OF ERROR-MSG)
         END-EXEC.

* COMMarea内容もログに出力（デバッグ用）
         IF EIBCALEN > 0 THEN
           IF EIBCALEN < 91 THEN
             MOVE DFHCOMMAREA(1:EIBCALEN) TO CA-DATA
           ELSE
             MOVE DFHCOMMAREA(1:90) TO CA-DATA
           END-IF
           EXEC CICS LINK PROGRAM('LGSTSQ')
                     COMMAREA(CA-ERROR-MSG)
                     LENGTH(LENGTH OF CA-ERROR-MSG)
           END-EXEC
         END-IF.
         EXIT.
```

**パターンの出現頻度**:
| パターン | 出現プログラム例 |
|---------|-----------------|
| 初期化パターン | 全31プログラム |
| COMMarea検証 | 全31プログラム |
| 長さチェック | DB/VSアクセスプログラム |
| SQLCODE判定 | DBアクセスプログラム（15個） |
| エラーログ出力 | 全31プログラム |

---

## 9. ファイル別リファレンス

### アプリケーション層構成

```
プレゼンテーション層（UI）
    LGICUS01, LGACUS01, LGUCUS01 ... （*US01）
        ↓ EXEC CICS LINK
ビジネスロジック層
    LGIPOL01, LGAPOL01, LGUPOL01 ... （*OL01）
        ↓ EXEC CICS LINK
データ管理層
    ├── DB2アクセス: LGICDB01, LGACDB01 ... （*DB01）
    └── VSAMアクセス: LGICVS01, LGACVS01 ... （*VS01）
```

### トランザクション対応表

| トランザクション | 機能 | 起点プログラム |
|-----------------|------|----------------|
| SSC1 | 顧客照会・追加 | LGICUS01 |
| SSP1 | 自動車保険ポリシー | LGIPOL01 |
| SSP2 | 養老保険ポリシー | LGIPOL01 |
| SSP3 | 住宅保険ポリシー | LGIPOL01 |
| SSP4 | 企業保険ポリシー | LGIPOL01 |

### データストア

| ストア | ファイル/テーブル | 用途 |
|--------|------------------|------|
| DB2 | CUSTOMER | 顧客マスタ |
| DB2 | POLICY | ポリシーマスタ |
| DB2 | MOTOR/HOUSE/ENDOWMENT/COMMERCIAL | 各保険種別データ |
| VSAM | KSDSCUST | 顧客データ（KSDS） |
| VSAM | KSDSPOLY | ポリシーデータ（KSDS） |
| TSQ | GENACNTL | 制御データ（一時記憶） |
| TSQ | GENAERRS | エラーログ（一時記憶） |

### 実例: 各層のプログラム比較

3層アーキテクチャの各層でプログラム構成がどう異なるかを比較します。

#### ビジネスロジック層（lgicus01.cbl）- 120行

```cobol
* シンプルな構成: 検証 → 下位層呼び出し → 戻る
      PROCEDURE DIVISION.
      MAINLINE SECTION.
         INITIALIZE WS-HEADER.
         ...（初期化・検証処理）...

* 109行目: DB2アクセス層を呼び出すだけ
         PERFORM GET-CUSTOMER-INFO.
         EXEC CICS RETURN END-EXEC.

      GET-CUSTOMER-INFO.
* 122-125行目: 単純な委譲
         EXEC CICS LINK Program(LGICDB01)
             Commarea(DFHCOMMAREA)
             LENGTH(32500)
         END-EXEC
         EXIT.
```

**特徴**: ビジネスロジック層は「薄い」設計。検証とルーティングのみ。

#### データアクセス層（lgicdb01.cbl）- 246行

```cobol
* 複雑な構成: 検証 → データ変換 → SQL実行 → 結果判定 → エラー処理
      PROCEDURE DIVISION.
      MAINLINE SECTION.
         ...（初期化・検証処理）...

* 130-131行目: DB2用ホスト変数を初期化
         INITIALIZE DB2-IN-INTEGERS.

* 146行目: COMMarea → DB2形式に変換
         MOVE CA-CUSTOMER-NUM TO DB2-CUSTOMERNUMBER-INT

* 154行目: SQL実行
         PERFORM GET-CUSTOMER-INFO.

      GET-CUSTOMER-INFO.
* 169-190行目: 実際のSELECT文
         EXEC SQL
             SELECT FIRSTNAME, LASTNAME, ...
             INTO :CA-FIRST-NAME, :CA-LAST-NAME, ...
             FROM CUSTOMER
             WHERE CUSTOMERNUMBER = :DB2-CUSTOMERNUMBER-INT
         END-EXEC.

* 192-203行目: SQLCODE判定
         Evaluate SQLCODE
           When 0 ...
           When 100 ...
           When Other ...
         END-Evaluate.
```

**特徴**: データアクセス層は「厚い」設計。データ変換、SQL、エラー処理を担当。

#### ユーティリティ層（lgstsq.cbl）- 127行

```cobol
* 共通機能: 呼び出し元判定 → パラメータ解析 → キュー書き込み
      PROCEDURE DIVISION.
      MAINLINE SECTION.
* 呼び出し元によって動作を変える
         IF WS-INVOKEPROG NOT = SPACES
            ...（LINK経由）
         ELSE
            ...（端末直接）
         END-IF.

* パラメータ解析（Q=nnnn形式）
         IF WRITE-MSG-MSG(1:2) = 'Q=' THEN
            MOVE WRITE-MSG-MSG(3:4) TO STSQ-EXT
         END-IF.

* 2種類のキューに書き込み
         EXEC CICS WRITEQ TD ... END-EXEC.
         EXEC CICS WRITEQ TS ... END-EXEC.
```

**特徴**: ユーティリティ層は複数の呼び出し方法に対応する柔軟な設計。

### プログラムサイズの目安

| 層 | 代表プログラム | 行数 | 複雑度 |
|----|---------------|------|--------|
| ビジネスロジック | lgicus01.cbl | ~120行 | 低 |
| データアクセス（DB2） | lgicdb01.cbl | ~246行 | 高 |
| データアクセス（VSAM） | lgicvs01.cbl | ~200行 | 中 |
| ユーティリティ | lgstsq.cbl | ~127行 | 中 |
| テスト | lgtestc1.cbl | ~300行 | 中 |

---

## 10. 業務フロー追跡の実践例

このセクションでは、実際の業務処理がソースコード上でどのように流れるかを、具体的なファイルと行番号を参照しながら追跡します。

---

### 例1: 顧客照会処理

**業務シナリオ**: ユーザーが顧客番号を入力し、顧客情報を画面に表示する

**処理フロー図**:
```
[端末] → SSC1トランザクション
           ↓
    ┌─────────────────┐
    │   LGICUS01      │  ビジネスロジック層
    │ (lgicus01.cbl)  │
    └────────┬────────┘
             │ EXEC CICS LINK
             ↓
    ┌─────────────────┐
    │   LGICDB01      │  データアクセス層
    │ (lgicdb01.cbl)  │
    └────────┬────────┘
             │ EXEC SQL SELECT
             ↓
    ┌─────────────────┐
    │   DB2 CUSTOMER  │  データベース
    │   テーブル      │
    └─────────────────┘
```

#### Step 1: ビジネスロジック層（lgicus01.cbl）

**ファイル**: `base/src/lgicus01.cbl`

```cobol
* 77-78行目: プログラム開始
MAINLINE SECTION.
    INITIALIZE WS-HEADER.

* 81-83行目: CICS実行情報の取得
    MOVE EIBTRNID TO WS-TRANSID.
    MOVE EIBTRMID TO WS-TERMID.
    MOVE EIBTASKN TO WS-TASKNUM.

* 87-91行目: パラメータチェック（COMMareaが渡されているか）
    IF EIBCALEN IS EQUAL TO ZERO
        MOVE ' NO COMMAREA RECEIVED' TO EM-VARIABLE
        PERFORM WRITE-ERROR-MESSAGE
        EXEC CICS ABEND ABCODE('LGCA') NODUMP END-EXEC
    END-IF

* 93-96行目: 戻りコード初期化
    MOVE '00' TO CA-RETURN-CODE
    MOVE '00' TO CA-NUM-POLICIES
    MOVE EIBCALEN TO WS-CALEN.
    SET WS-ADDR-DFHCOMMAREA TO ADDRESS OF DFHCOMMAREA.

* 109行目: データ取得処理を呼び出し
    PERFORM GET-CUSTOMER-INFO.
```

**ポイント**:
- `EIBCALEN`はCOMMareaの長さ。0なら呼び出し元からデータが渡されていない
- `CA-RETURN-CODE`に`'00'`を設定して正常状態で開始

#### Step 2: DB2アクセス層への呼び出し（lgicus01.cbl）

```cobol
* 120-128行目: GET-CUSTOMER-INFOパラグラフ
GET-CUSTOMER-INFO.
    EXEC CICS LINK Program(LGICDB01)    ← LGICDB01を呼び出し
        Commarea(DFHCOMMAREA)            ← 同じCOMMareaを渡す
        LENGTH(32500)                    ← COMMareaサイズ
    END-EXEC
    EXIT.
```

**ポイント**:
- `EXEC CICS LINK`で別プログラム（LGICDB01）を同期呼び出し
- `DFHCOMMAREA`を介してデータを受け渡し

#### Step 3: DB2からデータ取得（lgicdb01.cbl）

**ファイル**: `base/src/lgicdb01.cbl`

```cobol
* 145-148行目: 顧客番号をDB2形式に変換
    MOVE CA-CUSTOMER-NUM TO DB2-CUSTOMERNUMBER-INT
    MOVE CA-CUSTOMER-NUM TO EM-CUSNUM

* 154行目: SQL実行
    PERFORM GET-CUSTOMER-INFO.
```

```cobol
* 167-190行目: 実際のSQL SELECT文
GET-CUSTOMER-INFO.
    EXEC SQL
        SELECT FIRSTNAME,          ← 名
               LASTNAME,           ← 姓
               DATEOFBIRTH,        ← 生年月日
               HOUSENAME,          ← 住所1
               HOUSENUMBER,        ← 住所2
               POSTCODE,           ← 郵便番号
               PHONEMOBILE,        ← 携帯電話
               PHONEHOME,          ← 自宅電話
               EMAILADDRESS        ← メールアドレス
        INTO  :CA-FIRST-NAME,      ← 結果をCOMMareaに格納
              :CA-LAST-NAME,
              :CA-DOB,
              :CA-HOUSE-NAME,
              :CA-HOUSE-NUM,
              :CA-POSTCODE,
              :CA-PHONE-MOBILE,
              :CA-PHONE-HOME,
              :CA-EMAIL-ADDRESS
        FROM CUSTOMER
        WHERE CUSTOMERNUMBER = :DB2-CUSTOMERNUMBER-INT  ← 検索条件
    END-EXEC.
```

#### Step 4: 結果判定と戻りコード設定（lgicdb01.cbl）

```cobol
* 192-203行目: SQLCODEによる結果判定
    Evaluate SQLCODE
      When 0                            ← 正常取得
        MOVE '00' TO CA-RETURN-CODE
      When 100                          ← データなし（顧客が存在しない）
        MOVE '01' TO CA-RETURN-CODE
      When -913                         ← デッドロック発生
        MOVE '01' TO CA-RETURN-CODE
      When Other                        ← その他のエラー
        MOVE '90' TO CA-RETURN-CODE
        PERFORM WRITE-ERROR-MESSAGE     ← エラーログ出力
        EXEC CICS RETURN END-EXEC       ← 即時終了
    END-Evaluate.
```

**戻りコードの意味**:
| コード | 意味 |
|--------|------|
| 00 | 正常終了（顧客データ取得成功） |
| 01 | 該当顧客なし |
| 90 | DB2エラー |

---

### 例2: 顧客登録処理

**業務シナリオ**: 新規顧客の情報を入力し、データベースに登録する

**処理フロー図**:
```
[端末] → SSC1トランザクション（追加モード）
           ↓
    ┌─────────────────┐
    │   LGACUS01      │  ビジネスロジック層
    │ (lgacus01.cbl)  │
    └────────┬────────┘
             │ EXEC CICS LINK
             ↓
    ┌─────────────────┐
    │   LGACDB01      │  データアクセス層（メイン）
    │ (lgacdb01.cbl)  │
    └────────┬────────┘
             │
    ┌────────┴────────┬─────────────────┐
    ↓                 ↓                 ↓
┌────────┐     ┌────────┐       ┌────────┐
│名前付き │     │  DB2   │       │ VSAM   │
│カウンタ │     │INSERT  │       │書込み  │
└────────┘     └────────┘       └────────┘
```

#### Step 1: ビジネスロジック層（lgacus01.cbl）

**ファイル**: `base/src/lgacus01.cbl`

```cobol
* 94-99行目: パラメータチェック
    IF EIBCALEN IS EQUAL TO ZERO
        MOVE ' NO COMMAREA RECEIVED' TO EM-VARIABLE
        PERFORM WRITE-ERROR-MESSAGE
        EXEC CICS ABEND ABCODE('LGCA') NODUMP END-EXEC
    END-IF

* 119行目: 顧客登録処理を呼び出し
    PERFORM INSERT-CUSTOMER.

* 132-137行目: DB2アクセス層への呼び出し
INSERT-CUSTOMER.
    EXEC CICS LINK Program(LGACDB01)
         Commarea(DFHCOMMAREA)
         LENGTH(32500)
    END-EXEC.
```

#### Step 2: 顧客番号の採番（lgacdb01.cbl）

**ファイル**: `base/src/lgacdb01.cbl`

```cobol
* 199-211行目: 顧客番号を名前付きカウンタから取得
Obtain-CUSTOMER-Number.
    Exec CICS Get Counter(GENAcount)    ← カウンタ名: GENACUSTNUM
                  Pool(GENApool)         ← プール名: GENA
                  Value(LastCustNum)     ← 取得した番号を格納
                  Resp(WS-RESP)          ← 応答コード
    End-Exec.

    If WS-RESP Not = DFHRESP(NORMAL)     ← カウンタ取得失敗時
      MOVE 'NO' TO LGAC-NCS              ← フラグをOFFに
      Initialize DB2-CUSTOMERNUM-INT     ← DB2自動採番モードへ
    ELSE
      Move LastCustNum To DB2-CUSTOMERNUM-INT  ← 取得した番号を使用
    End-If.
```

**ポイント**:
- `EXEC CICS GET COUNTER`で一意な顧客番号を取得
- カウンタが使えない場合はDB2の`IDENTITY`機能で採番

#### Step 3: DB2へのINSERT（lgacdb01.cbl）

```cobol
* 215-248行目: 顧客データの登録（カウンタ使用時）
INSERT-CUSTOMER.
    IF LGAC-NCS = 'ON'                  ← 名前付きカウンタが有効な場合
      EXEC SQL
        INSERT INTO CUSTOMER
                  ( CUSTOMERNUMBER,      ← 顧客番号（カウンタから取得）
                    FIRSTNAME,           ← 名
                    LASTNAME,            ← 姓
                    DATEOFBIRTH,         ← 生年月日
                    HOUSENAME,           ← 住所1
                    HOUSENUMBER,         ← 住所2
                    POSTCODE,            ← 郵便番号
                    PHONEMOBILE,         ← 携帯電話
                    PHONEHOME,           ← 自宅電話
                    EMAILADDRESS )       ← メールアドレス
           VALUES ( :DB2-CUSTOMERNUM-INT,
                    :CA-FIRST-NAME,      ← COMMareaから取得
                    :CA-LAST-NAME,
                    :CA-DOB,
                    :CA-HOUSE-NAME,
                    :CA-HOUSE-NUM,
                    :CA-POSTCODE,
                    :CA-PHONE-MOBILE,
                    :CA-PHONE-HOME,
                    :CA-EMAIL-ADDRESS )
      END-EXEC
```

```cobol
* 251-283行目: カウンタが使えない場合（DB2自動採番）
    ELSE
      EXEC SQL
        INSERT INTO CUSTOMER
                  ( CUSTOMERNUMBER,
                    ...
                  )
           VALUES ( DEFAULT,              ← DB2が自動的に番号を割り当て
                    :CA-FIRST-NAME,
                    ...
                  )
      END-EXEC

      * 採番された番号を取得
      EXEC SQL
        SET :DB2-CUSTOMERNUM-INT = IDENTITY_VAL_LOCAL()
      END-EXEC
    END-IF.

* 285行目: 採番した顧客番号をCOMMareaに設定（呼び出し元に返す）
    MOVE DB2-CUSTOMERNUM-INT TO CA-CUSTOMER-NUM.
```

#### Step 4: VSAMへの書き込み（lgacdb01.cbl）

```cobol
* 174-177行目: VSAM書き込みプログラムを呼び出し
    EXEC CICS LINK Program(LGACVS01)    ← VSAM書き込みプログラム
         Commarea(DFHCOMMAREA)
         LENGTH(225)
    END-EXEC.
```

**ポイント**:
- DB2とVSAMの両方にデータを保存（二重化）
- 2フェーズコミットにより整合性を保証

#### Step 5: セキュリティ情報の登録（lgacdb01.cbl）

```cobol
* 179-189行目: 顧客セキュリティ情報の登録
    MOVE DB2-CUSTOMERNUM-INT TO D2-CUSTOMER-NUM.
    Move '02ACUS'     To  D2-REQUEST-ID.
    move '5732fec825535eeafb8fac50fee3a8aa'  ← 初期パスワード（ハッシュ）
                      To  D2-CUSTSECR-PASS.
    Move '0000'       To  D2-CUSTSECR-COUNT.  ← ログイン失敗回数
    Move 'N'          To  D2-CUSTSECR-STATE.  ← アカウント状態

    EXEC CICS LINK Program(LGACDB02)         ← セキュリティ情報登録
         Commarea(CDB2AREA)
         LENGTH(32500)
    END-EXEC.
```

---

### 例3: ポリシー照会処理

**業務シナリオ**: 保険ポリシー番号を入力し、ポリシー詳細を表示する

**処理フロー図**:
```
[端末] → SSP1〜4トランザクション
           ↓
    ┌─────────────────┐
    │   LGIPOL01      │  ビジネスロジック層
    │ (lgipol01.cbl)  │
    └────────┬────────┘
             │ EXEC CICS LINK
             ↓
    ┌─────────────────┐
    │   LGIPDB01      │  データアクセス層
    │ (lgipdb01.cbl)  │
    └────────┬────────┘
             │ EXEC SQL（カーソル使用）
             ↓
    ┌─────────────────┐
    │ POLICY + 各種   │  DB2テーブル群
    │ ポリシーテーブル│
    └─────────────────┘
```

#### Step 1: ビジネスロジック層（lgipol01.cbl）

**ファイル**: `base/src/lgipol01.cbl`

```cobol
* 70-76行目: 初期化処理
MAINLINE SECTION.
    INITIALIZE WS-HEADER.
    MOVE EIBTRNID TO WS-TRANSID.
    MOVE EIBTRMID TO WS-TERMID.
    MOVE EIBTASKN TO WS-TASKNUM.

* 91-94行目: DB2アクセス層呼び出し
    EXEC CICS LINK Program(LGIPDB01)
        Commarea(DFHCOMMAREA)
        Length(32500)
    END-EXEC.

* 96行目: 呼び出し元に戻る
    EXEC CICS RETURN END-EXEC.
```

**ポイント**:
- ビジネスロジック層は薄い（シンプルな中継）
- 複雑なロジックはデータアクセス層に委譲

#### Step 2: カーソルによるデータ取得（lgipdb01.cbl）

**ファイル**: `base/src/lgipdb01.cbl`

```cobol
* 89-118行目: カーソル定義（顧客番号で検索）
    EXEC SQL
      DECLARE Cust_Cursor Insensitive Scroll Cursor For
      SELECT
            CustomerNumber,           ← 顧客番号
            Policy.PolicyNumber,      ← ポリシー番号
            RequestDate,              ← 申請日
            StartDate,                ← 開始日
            RenewalDate,              ← 更新日
            Address,                  ← 住所
            Zipcode,                  ← 郵便番号
            ...
            Status,                   ← ステータス
            RejectionReason           ← 拒否理由
      FROM  POLICY,COMMERCIAL         ← POLICYとCOMMERCIALをJOIN
      WHERE ( POLICY.POLICYNUMBER =
                 Commercial.POLICYNUMBER AND
              Policy.CustomerNumber =
                 :DB2-CUSTOMERNUM-INT  )   ← 顧客番号で絞り込み
    END-EXEC.
```

**ポイント**:
- `DECLARE CURSOR`でカーソルを定義（複数行取得用）
- `INSENSITIVE SCROLL`で前後移動可能なカーソル
- POLICYテーブルと各ポリシー種別テーブルをJOIN

---

### 例4: 処理フロー追跡のチェックリスト

ソースコードを追跡する際の確認ポイント：

| 確認項目 | 確認方法 |
|----------|----------|
| 1. 起点プログラム | トランザクションIDから特定（SSC1→LGICUS01等） |
| 2. 呼び出し先 | `EXEC CICS LINK Program(...)` を検索 |
| 3. パラメータ | `Commarea(...)` で渡されるデータ構造を確認 |
| 4. SQL操作 | `EXEC SQL` ブロックを検索 |
| 5. エラー処理 | `SQLCODE` 判定と `CA-RETURN-CODE` 設定を確認 |
| 6. 戻り先 | `EXEC CICS RETURN` の位置を確認 |

---

### 例5: データの流れを追う

COMMarea（`lgcmarea.cpy`）を通じたデータフロー：

```
【入力時】
端末入力 → CA-CUSTOMER-NUM に顧客番号セット
        → CA-FIRST-NAME, CA-LAST-NAME 等に顧客情報セット

【処理中】
LGACUS01: COMMareaを受け取り、LGACDB01へ転送
LGACDB01: COMMareaから顧客情報を取り出し、DB2にINSERT
        → 採番した顧客番号を CA-CUSTOMER-NUM に格納

【出力時】
LGACDB01: CA-RETURN-CODE に結果コードをセット
        → LGACUS01に戻る
LGACUS01: 端末に結果を表示
```

**変数名から処理を推測するコツ**:
- `CA-`で始まる: プログラム間で受け渡されるデータ
- `DB2-`で始まる: DB2との入出力に使用
- `WS-`で始まる: そのプログラム内でのみ使用

---

## 11. 設計書の参照順序ガイド

`docs/code-to-docs/`ディレクトリには、このプロジェクトに関する各種設計書が格納されています。プロジェクトを効率的に理解するための推奨参照順序を示します。

---

### 11.1 学習目的別ロードマップ

#### 目的A: プロジェクトの全体像を把握したい

```
[1] README.md                  → プロジェクト概要
[2] 業務要件一覧/              → このシステムが解決する業務課題
[3] 機能一覧/                  → 提供される機能の全体像
[4] 画面一覧/                  → ユーザーインターフェースの概要
```

**所要時間目安**: 1〜2時間

#### 目的B: システムの技術構成を理解したい

```
[1] インフラ設計書/            → システム構成・環境
[2] データベース設計書/        → データ構造・ER図
[3] API設計書/                 → 外部インターフェース
[4] セキュリティ設計書/        → セキュリティ対策
```

**所要時間目安**: 2〜3時間

#### 目的C: 特定の画面・機能を開発/修正したい

```
[1] 画面機能マッピング/        → 画面と機能の対応関係
[2] 画面設計書/[対象画面].md   → 該当画面の詳細仕様
[3] 機能設計書/[対象機能].md   → 該当機能の詳細仕様
[4] 単体テストケース一覧/      → テスト観点の確認
```

**所要時間目安**: 画面/機能あたり1〜2時間

#### 目的D: バッチ処理を理解したい

```
[1] バッチ一覧/                → バッチジョブの全体像
[2] バッチ設計書/[対象].md     → 該当バッチの詳細仕様
[3] 帳票一覧/                  → 出力帳票の確認
[4] 帳票設計書/[対象].md       → 帳票レイアウト
```

**所要時間目安**: バッチあたり30分〜1時間

---

### 11.2 段階的学習ステップ

プロジェクトに新規参画した際の推奨学習ステップです。

#### Phase 1: 概要把握（1日目）

| 順序 | ドキュメント | 得られる知識 | 優先度 |
|------|-------------|-------------|--------|
| 1 | `README.md` | プロジェクト概要、技術スタック | 必須 |
| 2 | `業務要件一覧/` | 業務の目的、ユースケース | 必須 |
| 3 | `機能一覧/` | 提供機能の全体像 | 必須 |
| 4 | `画面一覧/` | 画面構成、画面遷移 | 必須 |

#### Phase 2: アーキテクチャ理解（2〜3日目）

| 順序 | ドキュメント | 得られる知識 | 優先度 |
|------|-------------|-------------|--------|
| 5 | `インフラ設計書/` | システム構成、環境情報 | 必須 |
| 6 | `データベース設計書/` | テーブル構造、リレーション | 必須 |
| 7 | `API設計書/` | 外部I/F、Webサービス仕様 | 推奨 |
| 8 | `セキュリティ設計書/` | 認証・認可、データ保護 | 推奨 |

#### Phase 3: 詳細仕様理解（4〜5日目）

| 順序 | ドキュメント | 得られる知識 | 優先度 |
|------|-------------|-------------|--------|
| 9 | `画面機能マッピング/` | 画面と機能の対応関係 | 必須 |
| 10 | `画面設計書/` | 画面項目、バリデーション | 担当分 |
| 11 | `機能設計書/` | 処理ロジック、業務ルール | 担当分 |
| 12 | `通知一覧/` `通知設計書/` | 通知機能の仕様 | 任意 |

#### Phase 4: テスト・品質理解（6〜7日目）

| 順序 | ドキュメント | 得られる知識 | 優先度 |
|------|-------------|-------------|--------|
| 13 | `テスト方針書/` | テスト戦略、品質基準 | 推奨 |
| 14 | `非機能要件定義書/` | 性能、可用性要件 | 推奨 |
| 15 | `単体テストケース一覧/` | テスト観点、期待結果 | 担当分 |
| 16 | `結合テストケース一覧/` | 連携テストの観点 | 任意 |

#### Phase 5: 運用・改善理解（以降）

| 順序 | ドキュメント | 得られる知識 | 優先度 |
|------|-------------|-------------|--------|
| 17 | `運用マニュアル/` | 運用手順、障害対応 | 任意 |
| 18 | `セキュリティ診断レポート/` | 脆弱性、対策状況 | 任意 |
| 19 | `改善提案書/` | 既知の課題、改善案 | 任意 |

---

### 11.3 設計書一覧と概要

#### 要件・概要系

| ドキュメント | 内容 | ファイル形式 |
|-------------|------|-------------|
| `README.md` | プロジェクト全体の概要説明 | Markdown |
| `業務要件一覧/` | 業務要件の一覧と詳細 | CSV/Markdown |
| `非機能要件定義書/` | 性能・可用性・セキュリティ要件 | Markdown |

#### 設計系

| ドキュメント | 内容 | ファイル形式 |
|-------------|------|-------------|
| `インフラ設計書/` | システム構成図、環境定義 | Markdown |
| `データベース設計書/` | ER図、テーブル定義 | Markdown |
| `API設計書/` | API仕様、リクエスト/レスポンス | Markdown |
| `セキュリティ設計書/` | 認証・認可、暗号化方針 | Markdown |

#### 画面系

| ドキュメント | 内容 | ファイル形式 |
|-------------|------|-------------|
| `画面一覧/` | 画面ID、画面名、遷移図 | CSV/Markdown |
| `画面機能マッピング/` | 画面と機能の対応表 | CSV |
| `画面設計書/` | 各画面の詳細仕様（12画面分） | Markdown |

#### 機能系

| ドキュメント | 内容 | ファイル形式 |
|-------------|------|-------------|
| `機能一覧/` | 機能ID、機能名の一覧 | CSV |
| `機能設計書/` | 各機能の詳細仕様（60機能分） | Markdown |
| `バッチ一覧/` | バッチID、実行スケジュール | CSV |
| `バッチ設計書/` | 各バッチの詳細仕様（58バッチ分） | Markdown |

#### 帳票・通知系

| ドキュメント | 内容 | ファイル形式 |
|-------------|------|-------------|
| `帳票一覧/` | 帳票ID、帳票名の一覧 | CSV |
| `帳票設計書/` | 各帳票のレイアウト・項目定義（6帳票分） | Markdown |
| `通知一覧/` | 通知ID、通知種別の一覧 | CSV |
| `通知設計書/` | 各通知の詳細仕様（34通知分） | Markdown |

#### テスト系

| ドキュメント | 内容 | ファイル形式 |
|-------------|------|-------------|
| `テスト方針書/` | テスト戦略、品質目標 | Markdown |
| `単体テストケース一覧/` | 単体テストケース（15モジュール分） | CSV/Markdown |
| `結合テストケース一覧/` | 結合テストケース | CSV/Markdown |
| `受入テストケース一覧/` | 受入テストケース | CSV/Markdown |
| `非機能要件テストケース一覧/` | 性能・負荷テストケース | CSV/Markdown |

#### 運用・診断系

| ドキュメント | 内容 | ファイル形式 |
|-------------|------|-------------|
| `運用マニュアル/` | 運用手順、障害対応 | Markdown |
| `セキュリティ診断レポート/` | 脆弱性診断結果 | Markdown |
| `改善提案書/` | 技術的負債、改善案 | Markdown |

---

### 11.4 ソースコードと設計書の対応

ソースコードを読む際に参照すべき設計書の対応関係：

| ソースファイル | 参照すべき設計書 |
|---------------|-----------------|
| `lgicus01.cbl`（顧客照会UI） | `画面設計書/顧客照会画面.md` |
| `lgacus01.cbl`（顧客追加UI） | `画面設計書/顧客追加画面.md` |
| `lgicdb01.cbl`（顧客DB照会） | `機能設計書/顧客照会機能.md`、`データベース設計書/` |
| `lgacdb01.cbl`（顧客DB追加） | `機能設計書/顧客追加機能.md`、`データベース設計書/` |
| `lgipol01.cbl`（ポリシー照会） | `画面設計書/ポリシー照会画面.md` |
| `lgapol01.cbl`（ポリシー追加） | `機能設計書/ポリシー追加機能.md` |
| `lgcmarea.cpy`（COMMAREA定義） | `API設計書/`、`データベース設計書/` |

---

### 11.5 よくある質問への回答フロー

| 質問 | 参照すべきドキュメント |
|------|----------------------|
| 「この画面は何をする画面？」 | `画面一覧/` → `画面設計書/[該当画面].md` |
| 「この機能のビジネスルールは？」 | `機能設計書/[該当機能].md` |
| 「このテーブルの項目定義は？」 | `データベース設計書/` |
| 「このバッチはいつ動く？」 | `バッチ一覧/` → `バッチ設計書/[該当バッチ].md` |
| 「テストで何を確認すべき？」 | `単体テストケース一覧/[該当モジュール].md` |
| 「セキュリティ要件は？」 | `セキュリティ設計書/` → `セキュリティ診断レポート/` |
| 「性能要件は？」 | `非機能要件定義書/` |

---

## 付録A: COBOLキーワード対応表

| COBOL | 現代言語 |
|-------|----------|
| MOVE A TO B | B = A |
| ADD A TO B | B = B + A |
| COMPUTE B = A + C | B = A + C |
| IF ... END-IF | if ... |
| EVALUATE ... END-EVALUATE | switch ... |
| PERFORM X | X() を呼び出し |
| PERFORM X UNTIL Y | while (!Y) { X(); } |
| INITIALIZE X | memset(X, 0, sizeof(X)) |
| SPACES | 空白文字で埋める |
| ZEROS | ゼロで埋める |

---

## 付録B: よくある疑問

### Q: `FILLER`とは何ですか？
A: 名前のない埋め草フィールドです。構造体のパディングや未使用領域を表します。

### Q: `VALUE`句は何ですか？
A: 変数の初期値を設定します。
```cobol
03 WS-EYECATCHER PIC X(16) VALUE 'LGICDB01------WS'.
```

### Q: なぜ`03`, `05`のように奇数なのですか？
A: 後からレベル間に項目を追加しやすくするための慣習です。

### Q: `COMP`とは何ですか？
A: COMPUTATIONALの略で、バイナリ形式で格納することを意味します。

---

## 付録C: デバッグのヒント

1. **エラーログの確認**: `GENAERRS`キューにエラーメッセージが出力される
2. **戻りコードの確認**: `CA-RETURN-CODE`の値でエラー種別を判定
3. **SQLCODEの確認**: DB2エラーはSQLCODEで特定
4. **ABENDコードの確認**: 'LGCA'はCOMMAREAエラーを示す

---

#1
