# セキュリティ設計書

## 1. 概要

### 1.1 ドキュメントの目的
本ドキュメントは、IBM CICS Transaction Server上で稼働する一般保険アプリケーション（GenApp）のセキュリティ設計について記述する。

### 1.2 対象システム
- **システム名**: General Insurance Application (GenApp)
- **プラットフォーム**: IBM CICS Transaction Server for z/OS (CICS TS)
- **データベース**: IBM Db2
- **データストレージ**: VSAM, Db2, Temporary Storage Queues

### 1.3 アプリケーション概要
GenAppは保険業務を模擬したアプリケーションで、以下の機能を提供する：
- 顧客情報の登録・照会・更新
- 保険契約（自動車、養老、住宅、商業物件）の管理
- 3270端末インターフェースおよびWebサービスインターフェース

---

## 2. セキュリティアーキテクチャ

### 2.1 アーキテクチャ概要

```
+------------------+     +------------------+     +------------------+
|   3270 Terminal  |     |   Web Services   |     |  Workload Sim    |
|    Interface     |     |    Interface     |     |    Interface     |
+--------+---------+     +--------+---------+     +--------+---------+
         |                        |                        |
         +------------------------+------------------------+
                                  |
                    +-------------+--------------+
                    |      CICS TS Region        |
                    |  +----------------------+  |
                    |  | Presentation Logic   |  |
                    |  | (LGTESTC1, LGTESTP*) |  |
                    |  +----------+-----------+  |
                    |             |              |
                    |  +----------+-----------+  |
                    |  |   Business Logic     |  |
                    |  | (LG*CUS01, LG*POL01) |  |
                    |  +----------+-----------+  |
                    |             |              |
                    |  +----------+-----------+  |
                    |  |   Data Access Layer  |  |
                    |  | (LG*DB01, LG*VS01)   |  |
                    |  +----------+-----------+  |
                    +-------------+--------------+
                                  |
         +------------------------+------------------------+
         |                        |                        |
+--------+---------+     +--------+---------+     +--------+---------+
|      VSAM        |     |       Db2        |     |   Coupling       |
|      Files       |     |    Database      |     |   Facility       |
+------------------+     +------------------+     +------------------+
```

### 2.2 レイヤー構成とセキュリティ境界

| レイヤー | コンポーネント | セキュリティ責務 |
|---------|---------------|-----------------|
| プレゼンテーション層 | LGTESTC1, LGTESTP1-4, SSMAP | 入力検証、画面制御 |
| ビジネスロジック層 | LGACUS01, LGICUS01, LGUCUS01, LG*POL01 | リクエスト検証、ビジネスルール適用 |
| データアクセス層 | LG*DB01, LG*VS01 | データ整合性、アクセス制御 |

---

## 3. 認証・認可設計

### 3.1 顧客認証メカニズム

#### 3.1.1 CUSTOMER_SECURE テーブル構造

```sql
CREATE TABLE CUSTOMER_SECURE (
    customerNumber INTEGER NOT NULL,
    customerPass   CHAR(32),       -- MD5ハッシュ形式
    state_indicator CHAR(1),       -- 'N':通常, 他:ロック等
    pass_changes   INTEGER,        -- パスワード変更回数
    PRIMARY KEY(customerNumber),
    FOREIGN KEY(customerNumber)
        REFERENCES customer (customerNumber)
        ON DELETE CASCADE
)
```

#### 3.1.2 パスワード保存方式
- **ハッシュアルゴリズム**: MD5（32文字16進表記）
- **初期パスワード**: `5732fec825535eeafb8fac50fee3a8aa`
- **保存形式**: CHAR(32)固定長フィールド

```cobol
* LGACDB01.cbl - パスワード設定処理
move '5732fec825535eeafb8fac50fee3a8aa'
                    To  D2-CUSTSECR-PASS.
Move '0000'       To  D2-CUSTSECR-COUNT.
Move 'N'          To  D2-CUSTSECR-STATE.
```

#### 3.1.3 認証状態管理
| フィールド | 説明 |
|-----------|------|
| state_indicator | 'N': 通常状態, その他: ロック等の状態制御 |
| pass_changes | パスワード変更回数のカウンター |

### 3.2 CICSセキュリティ

#### 3.2.1 トランザクションセキュリティ
CICSリソース定義により以下のトランザクションが定義される：

| トランザクションID | 機能 | セキュリティレベル |
|-------------------|------|-------------------|
| SSC1 | 顧客メニュー | 端末認証 |
| SSP1-4 | 保険契約メニュー | 端末認証 |
| LGSE | 初期化処理 | システム権限 |
| LGST | イベントトリガー | 内部処理 |

#### 3.2.2 プログラムセキュリティ
```
EXEC CICS LINK Program(LGACDB01)
    Commarea(DFHCOMMAREA)
    LENGTH(32500)
END-EXEC.
```
- プログラム間通信はCICS LINK/RETURNメカニズムを使用
- COMMAPREAを介したデータ受け渡し

### 3.3 データベースアクセス制御

#### 3.3.1 Db2権限設定
```sql
-- db2cre.jcl より
GRANT DBADM ON DATABASE <DB2DBID>      TO PUBLIC;
GRANT ALL PRIVILEGES ON TABLE <DB2DBID>.customer TO PUBLIC;
GRANT ALL PRIVILEGES ON TABLE <DB2DBID>.policy   TO PUBLIC;
-- ... 全テーブルにPUBLICアクセス
```

**注意**: 現在の設定では全テーブルにPUBLIC権限が付与されている。

#### 3.3.2 推奨アクセス制御設定
```sql
-- 推奨: ロールベースアクセス制御
CREATE ROLE GENAPP_READ;
CREATE ROLE GENAPP_WRITE;
CREATE ROLE GENAPP_ADMIN;

GRANT SELECT ON <DB2DBID>.customer TO GENAPP_READ;
GRANT SELECT, INSERT, UPDATE ON <DB2DBID>.customer TO GENAPP_WRITE;
GRANT ALL PRIVILEGES ON <DB2DBID>.customer TO GENAPP_ADMIN;
```

---

## 4. データ保護設計

### 4.1 機密データの分類

| データ種別 | 機密レベル | 保存場所 | 保護方式 |
|-----------|-----------|----------|----------|
| 顧客個人情報 | 高 | CUSTOMER テーブル, KSDSCUST VSAM | アクセス制御 |
| 認証情報 | 最高 | CUSTOMER_SECURE テーブル | ハッシュ化 |
| 保険契約情報 | 高 | POLICY/各種テーブル, KSDSPOLY VSAM | アクセス制御 |
| 連絡先情報 | 中 | CUSTOMER テーブル | アクセス制御 |

### 4.2 個人情報フィールド

```cobol
* lgcmarea.cpy - 顧客データ構造
03 CA-CUSTOMER-REQUEST REDEFINES CA-REQUEST-SPECIFIC.
   05 CA-FIRST-NAME         PIC X(10).
   05 CA-LAST-NAME          PIC X(20).
   05 CA-DOB                PIC X(10).    -- 生年月日
   05 CA-HOUSE-NAME         PIC X(20).
   05 CA-HOUSE-NUM          PIC X(4).
   05 CA-POSTCODE           PIC X(8).
   05 CA-NUM-POLICIES       PIC 9(3).
   05 CA-PHONE-MOBILE       PIC X(20).    -- 携帯電話
   05 CA-PHONE-HOME         PIC X(20).    -- 自宅電話
   05 CA-EMAIL-ADDRESS      PIC X(100).   -- メールアドレス
```

### 4.3 データ暗号化

#### 4.3.1 現状
- **転送中**: 3270プロトコル（平文）
- **保存時**: 平文保存（Db2/VSAM）
- **パスワード**: MD5ハッシュ

#### 4.3.2 推奨暗号化方式
| 対象 | 推奨方式 |
|------|---------|
| 通信 | TLS 1.3 |
| 保存データ | Db2 Native Encryption |
| パスワード | bcrypt/Argon2 |

### 4.4 データ整合性

#### 4.4.1 参照整合性制約
```sql
-- 顧客-セキュリティ関連
FOREIGN KEY(customerNumber)
    REFERENCES customer (customerNumber)
    ON DELETE CASCADE

-- 保険契約-顧客関連
FOREIGN KEY(customerNumber)
    REFERENCES customer (customerNumber)
    ON DELETE CASCADE

-- 保険契約-詳細関連
FOREIGN KEY(policyNumber)
    REFERENCES policy (policyNumber)
    ON DELETE CASCADE
```

#### 4.4.2 楽観的ロック
```cobol
* lgupdb01.cbl - タイムスタンプによる排他制御
IF CA-LASTCHANGED EQUAL TO DB2-LASTCHANGED
    PERFORM UPDATE-ENDOW-DB2-INFO
ELSE
    MOVE '02' TO CA-RETURN-CODE  -- 競合検出
END-IF.
```

### 4.5 二相コミット処理
```cobol
* Db2とVSAMの両方を更新する際の整合性確保
EXEC CICS LINK Program(LGACVS01)
    Commarea(DFHCOMMAREA)
    LENGTH(225)
END-EXEC.

EXEC CICS LINK Program(LGACDB02)
    Commarea(CDB2AREA)
    LENGTH(32500)
END-EXEC.
```

---

## 5. 入力検証設計

### 5.1 COMMAREA検証

#### 5.1.1 長さ検証
```cobol
* lgacus01.cbl - COMMAREA長検証
IF EIBCALEN IS EQUAL TO ZERO
    MOVE ' NO COMMAREA RECEIVED' TO EM-VARIABLE
    PERFORM WRITE-ERROR-MESSAGE
    EXEC CICS ABEND ABCODE('LGCA') NODUMP END-EXEC
END-IF

ADD WS-CA-HEADER-LEN TO WS-REQUIRED-CA-LEN
ADD WS-CUSTOMER-LEN  TO WS-REQUIRED-CA-LEN

IF EIBCALEN IS LESS THAN WS-REQUIRED-CA-LEN
    MOVE '98' TO CA-RETURN-CODE
    EXEC CICS RETURN END-EXEC
END-IF
```

#### 5.1.2 リクエストID検証
```cobol
* lgucus01.cbl - リクエストID検証
If CA-REQUEST-ID NOT = '01UCUS'
    MOVE '99' TO CA-RETURN-CODE
    GO TO END-PROGRAM
END-IF.
```

```cobol
* lgdpdb01.cbl - 複数リクエストID検証
IF ( CA-REQUEST-ID NOT EQUAL TO '01DEND' AND
     CA-REQUEST-ID NOT EQUAL TO '01DHOU' AND
     CA-REQUEST-ID NOT EQUAL TO '01DCOM' AND
     CA-REQUEST-ID NOT EQUAL TO '01DMOT' ) Then
    MOVE '99' TO CA-RETURN-CODE
END-IF
```

### 5.2 リクエストIDマッピング

| リクエストID | 機能 | プログラム |
|-------------|------|-----------|
| 01ICUS | 顧客照会 | LGICDB01 |
| 01ACUS | 顧客追加 | LGACDB01 |
| 02ACUS | 顧客セキュリティ追加 | LGACDB02 |
| 01UCUS | 顧客更新 | LGUCDB01 |
| 01IMOT/01IEND/01IHOU/01ICOM | 保険照会 | LGIPDB01 |
| 01AMOT/01AEND/01AHOU/01ACOM | 保険追加 | LGAPDB01 |
| 01UMOT/01UEND/01UHOU | 保険更新 | LGUPDB01 |
| 01DMOT/01DEND/01DHOU/01DCOM | 保険削除 | LGDPDB01 |

### 5.3 SQLインジェクション対策

#### 5.3.1 プリペアドステートメントの使用
```cobol
* lgicdb01.cbl - パラメータ化クエリ
EXEC SQL
    SELECT FIRSTNAME,
           LASTNAME,
           DATEOFBIRTH,
           ...
    INTO  :CA-FIRST-NAME,
          :CA-LAST-NAME,
          :CA-DOB,
          ...
    FROM CUSTOMER
    WHERE CUSTOMERNUMBER = :DB2-CUSTOMERNUMBER-INT
END-EXEC.
```

全てのSQL文でホスト変数（:変数名形式）を使用し、SQLインジェクションを防止。

---

## 6. エラー処理・ログ設計

### 6.1 エラーメッセージ構造

```cobol
01  ERROR-MSG.
    03 EM-DATE                  PIC X(8)  VALUE SPACES.
    03 FILLER                   PIC X     VALUE SPACES.
    03 EM-TIME                  PIC X(6)  VALUE SPACES.
    03 FILLER                   PIC X(9)  VALUE ' LGACDB01'.
    03 EM-VARIABLE.
      05 FILLER                 PIC X(6)  VALUE ' CNUM='.
      05 EM-CUSNUM              PIC X(10) VALUE SPACES.
      05 EM-SQLREQ              PIC X(16) VALUE SPACES.
      05 FILLER                 PIC X(9)  VALUE ' SQLCODE='.
      05 EM-SQLRC               PIC +9(5) USAGE DISPLAY.
```

### 6.2 エラーログ出力処理

```cobol
WRITE-ERROR-MESSAGE.
* Save SQLCODE in message
    MOVE SQLCODE TO EM-SQLRC
* Obtain and format current time and date
    EXEC CICS ASKTIME ABSTIME(WS-ABSTIME)
    END-EXEC
    EXEC CICS FORMATTIME ABSTIME(WS-ABSTIME)
              MMDDYYYY(WS-DATE)
              TIME(WS-TIME)
    END-EXEC
    MOVE WS-DATE TO EM-DATE
    MOVE WS-TIME TO EM-TIME
* Write output message to TSQ
    EXEC CICS LINK PROGRAM('LGSTSQ')
              COMMAREA(ERROR-MSG)
              LENGTH(LENGTH OF ERROR-MSG)
    END-EXEC.
```

### 6.3 Temporary Storage Queue

| キュー名 | 用途 | セキュリティ考慮 |
|---------|------|-----------------|
| GENACNTL | カウンター範囲管理 | 内部処理用 |
| GENAERRS | エラーログ | 診断情報含む |
| GENASTAT | 統計情報 | 運用監視用 |

### 6.4 エラーコード体系

| コード | 意味 | 対応 |
|-------|------|------|
| 00 | 正常終了 | - |
| 01 | レコード未検出 | 情報メッセージ |
| 02 | タイムスタンプ不一致（競合） | 再試行要求 |
| 90 | データベースエラー | エラーログ出力 |
| 98 | COMMAREA長不正 | 処理中断 |
| 99 | リクエストID不正 | 処理拒否 |

---

## 7. セッション管理

### 7.1 CICSセッション管理

- **端末セッション**: CICS Terminal Control
- **トランザクション分離**: 各トランザクションは独立実行
- **状態管理**: COMMAREA経由

### 7.2 COMMAREA構造

```cobol
01  DFHCOMMAREA.
    03 CA-REQUEST-ID            PIC X(6).
    03 CA-RETURN-CODE           PIC 9(2).
    03 CA-CUSTOMER-NUM          PIC 9(10).
    03 CA-REQUEST-SPECIFIC      PIC X(32482).
```

### 7.3 セキュリティ考慮事項

- COMMAPREAはメモリ上に存在し、トランザクション終了時に解放
- 機密データはCOMMAPREAに必要最小限のみ格納
- 認証情報（パスワード）はCOMMAPREA経由で受け渡し

---

## 8. 監査・モニタリング設計

### 8.1 トランザクションカウンター

```cobol
* lgastat1.cbl - カウンター更新
Exec CICS Get Counter(GENAcount)
             Pool(GENApool)
             Value(LastCustNum)
             Resp(WS-RESP)
End-Exec.
```

### 8.2 カウンター定義

| カウンター | 用途 |
|-----------|------|
| GENACUSTNUM | 顧客番号採番 |
| GENA01ICUS00-99 | 顧客照会成功/エラー |
| GENA01ACUS00-99 | 顧客追加成功/エラー |
| GENA01IMOT00-99 | 自動車保険照会成功/エラー |
| 等 | 各操作の成功/エラーカウント |

### 8.3 イベントバインディング
- `Transaction_Counters.evbind`: トランザクション統計収集用イベント定義

---

## 9. セキュリティリスク評価

### 9.1 識別されたリスク

| リスクID | リスク | 重大度 | 現状 | 推奨対策 |
|---------|-------|--------|------|---------|
| R001 | MD5ハッシュの脆弱性 | 高 | MD5使用 | bcrypt/Argon2へ移行 |
| R002 | Db2 PUBLIC権限 | 高 | 全権限付与 | ロールベース権限へ変更 |
| R003 | 平文通信 | 中 | 3270プロトコル | AT-TLS適用 |
| R004 | ハードコード認証情報 | 中 | 初期パスワード固定 | 初期設定プロセス改善 |
| R005 | エラーメッセージ情報漏洩 | 低 | SQLCODE出力 | エラーログの保護強化 |

### 9.2 リスク対応優先度

```
高優先度:
1. R001 - パスワードハッシュアルゴリズム更新
2. R002 - データベース権限の最小化

中優先度:
3. R003 - 通信暗号化
4. R004 - 認証情報管理改善

低優先度:
5. R005 - ログセキュリティ強化
```

---

## 10. コンプライアンス考慮事項

### 10.1 個人情報保護

保存される個人情報：
- 氏名（firstName, lastName）
- 生年月日（dateOfBirth）
- 住所（houseName, houseNumber, postcode）
- 連絡先（phoneMobile, phoneHome, emailAddress）

### 10.2 推奨コンプライアンス対応

| 要件 | 現状 | 推奨対応 |
|------|------|---------|
| データ最小化 | 必要データのみ収集 | 適合 |
| 暗号化 | パスワードのみ | 全機密データの暗号化 |
| アクセス制御 | PUBLIC権限 | 最小権限原則の適用 |
| 監査ログ | 基本的なエラーログ | 詳細アクセスログの実装 |

---

## 11. 付録

### 11.1 セキュリティ関連プログラム一覧

| プログラム | 機能 | セキュリティ関連処理 |
|-----------|------|---------------------|
| LGACDB01 | 顧客追加（Db2） | セキュリティレコード作成呼出 |
| LGACDB02 | 顧客セキュリティ追加 | パスワードハッシュ保存 |
| LGICDB01 | 顧客照会（Db2） | データアクセス制御 |
| LGUCDB01 | 顧客更新（Db2） | 更新権限チェック |
| LGSTSQ | TSQメッセージ出力 | エラーログ記録 |

### 11.2 データベーステーブル一覧

| テーブル | セキュリティレベル | 主キー |
|---------|-------------------|--------|
| CUSTOMER | 高（個人情報） | customerNumber |
| CUSTOMER_SECURE | 最高（認証情報） | customerNumber |
| POLICY | 高（契約情報） | policyNumber |
| MOTOR | 高（契約詳細） | policyNumber |
| ENDOWMENT | 高（契約詳細） | policyNumber |
| HOUSE | 高（契約詳細） | policyNumber |
| COMMERCIAL | 高（契約詳細） | policyNumber |
| CLAIM | 高（請求情報） | claimNumber |

### 11.3 参照ドキュメント

- IBM CICS Transaction Server セキュリティガイド
- IBM Db2 セキュリティガイド
- COBOL プログラミングガイド

---

## 改訂履歴

| バージョン | 日付 | 変更内容 | 作成者 |
|-----------|------|---------|--------|
| 1.0 | 2026-01-06 | 初版作成 | Claude Code |
