# API設計書

## 概要

本ドキュメントは、.NET Runtime WebAssembly/Browser向けJavaScript APIの設計書です。このAPIは、ブラウザおよびNode.js環境で.NETランタイムを実行するためのJavaScriptインターフェースを提供します。

## 共通仕様

### ベースURL

```
N/A (JavaScriptモジュールとして提供)
```

本APIはHTTP REST APIではなく、JavaScriptモジュールとして提供されます。`dotnet.js`モジュールをインポートして使用します。

### 認証方式

N/A（ローカルJavaScript API）

### 共通ヘッダー

N/A（HTTP APIではない）

### 共通エラーレスポンス

| エラー種別 | 説明 |
| --- | --- |
| Error | 一般的なエラー |
| OperationCanceledException | 操作がキャンセルされた |
| AbortError | HTTP/WebSocketリクエストが中止された |
| InvalidState | 不正な状態での操作試行 |

## API一覧

| カテゴリ | API名 | 種別 | 説明 |
| --- | --- | --- | --- |
| Runtime Builder | dotnet.withConfig() | Method | ランタイム設定を構成 |
| Runtime Builder | dotnet.withApplicationArguments() | Method | Main()メソッドの引数を設定 |
| Runtime Builder | dotnet.withEnvironmentVariable() | Method | 環境変数を設定 |
| Runtime Builder | dotnet.withEnvironmentVariables() | Method | 複数の環境変数を設定 |
| Runtime Builder | dotnet.withVirtualWorkingDirectory() | Method | 仮想作業ディレクトリを設定 |
| Runtime Builder | dotnet.withDiagnosticTracing() | Method | 診断トレースを有効化 |
| Runtime Builder | dotnet.withDebugging() | Method | デバッグレベルを設定 |
| Runtime Builder | dotnet.withMainAssembly() | Method | メインアセンブリを指定 |
| Runtime Builder | dotnet.withApplicationArgumentsFromQuery() | Method | URLクエリから引数を取得 |
| Runtime Builder | dotnet.withApplicationEnvironment() | Method | アプリケーション環境を設定 |
| Runtime Builder | dotnet.withApplicationCulture() | Method | カルチャを設定 |
| Runtime Builder | dotnet.withResourceLoader() | Method | リソースローダーをオーバーライド |
| Runtime Builder | dotnet.download() | Method | アセットをダウンロード |
| Runtime Builder | dotnet.create() | Method | ランタイムを作成 |
| Runtime Builder | dotnet.runMain() | Method | Main()を実行（ランタイム継続） |
| Runtime Builder | dotnet.runMainAndExit() | Method | Main()を実行（終了） |
| Runtime API | runMain() | Function | Main()メソッドを実行 |
| Runtime API | runMainAndExit() | Function | Main()メソッドを実行して終了 |
| Runtime API | exit() | Function | ランタイムを終了 |
| Runtime API | setEnvironmentVariable() | Function | 環境変数を設定 |
| Runtime API | getAssemblyExports() | Function | JSExportメソッドを取得 |
| Runtime API | setModuleImports() | Function | JSImport用関数を提供 |
| Runtime API | getConfig() | Function | 設定を取得 |
| Runtime API | invokeLibraryInitializers() | Function | ライブラリ初期化子を実行 |
| Memory API | setHeapB32/B8/U8/U16/U32/I8/I16/I32/I52/U52/I64Big/F32/F64 | Function | WASMメモリに書き込み |
| Memory API | getHeapB32/B8/U8/U16/U32/I8/I16/I32/I52/U52/I64Big/F32/F64 | Function | WASMメモリから読み取り |
| Memory API | localHeapViewI8/I16/I32/I64Big/U8/U16/U32/F32/F64 | Function | WASMメモリビューを取得 |
| Diagnostics API | collectCpuSamples() | Function | CPUサンプルを収集 |
| Diagnostics API | collectMetrics() | Function | メトリクスを収集 |
| Diagnostics API | collectGcDump() | Function | GCダンプを収集 |
| Diagnostics API | connectDSRouter() | Function | 診断ルーターに接続 |
| HTTP Interop | httpCreateController() | Function | HTTPコントローラーを作成 |
| HTTP Interop | httpFetch() | Function | HTTPリクエストを実行 |
| HTTP Interop | httpFetchBytes() | Function | バイト配列でHTTPリクエスト |
| HTTP Interop | httpFetchStream() | Function | ストリーミングHTTPリクエスト |
| HTTP Interop | httpAbort() | Function | HTTPリクエストを中止 |
| HTTP Interop | httpGetResponseStatus() | Function | レスポンスステータスを取得 |
| HTTP Interop | httpGetResponseHeaderNames() | Function | レスポンスヘッダー名を取得 |
| HTTP Interop | httpGetResponseHeaderValues() | Function | レスポンスヘッダー値を取得 |
| HTTP Interop | httpGetResponseLength() | Function | レスポンス長を取得 |
| HTTP Interop | httpGetResponseBytes() | Function | レスポンスバイトを取得 |
| HTTP Interop | httpGetStreamedResponseBytes() | Function | ストリーミングレスポンスを取得 |
| HTTP Interop | httpSupportsStreamingRequest() | Function | ストリーミングリクエスト対応確認 |
| HTTP Interop | httpSupportsStreamingResponse() | Function | ストリーミングレスポンス対応確認 |
| WebSocket Interop | wsCreate() | Function | WebSocketを作成 |
| WebSocket Interop | wsOpen() | Function | WebSocket接続を開く |
| WebSocket Interop | wsSend() | Function | WebSocketでデータ送信 |
| WebSocket Interop | wsReceive() | Function | WebSocketでデータ受信 |
| WebSocket Interop | wsClose() | Function | WebSocket接続を閉じる |
| WebSocket Interop | wsAbort() | Function | WebSocket接続を中止 |
| WebSocket Interop | wsGetState() | Function | WebSocket状態を取得 |

## 各APIエンドポイント定義

### DotnetHostBuilder（ランタイムビルダー）

#### 1. withConfig

ランタイムの設定オブジェクトを指定します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `withConfig(config: LoaderConfig): DotnetHostBuilder` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| config | LoaderConfig | ○ | ランタイム設定オブジェクト |

**LoaderConfig型**

```typescript
{
  ignorePdbLoadErrors?: boolean;           // PDBロードエラーを無視
  maxParallelDownloads?: number;           // 並列ダウンロード数（デフォルト: 16）
  enableDownloadRetry?: boolean;           // ダウンロードリトライ有効化
  mainAssemblyName?: string;               // メインアセンブリ名
  globalizationMode?: GlobalizationMode;   // グローバリゼーションモード
  debugLevel?: number;                     // デバッグレベル
  disableIntegrityCheck?: boolean;         // 整合性チェック無効化
  disableNoCacheFetch?: boolean;           // no-cacheディレクティブ無効化
  diagnosticTracing?: boolean;             // 診断トレース有効化
  environmentVariables?: { [i: string]: string };  // 環境変数
  pthreadPoolInitialSize?: number;         // pthreadプール初期サイズ
  pthreadPoolUnusedSize?: number;          // 未使用pthreadプールサイズ
  interpreterPgo?: boolean;                // インタプリタPGO有効化
  interpreterPgoSaveDelay?: number;        // PGO保存遅延
  applicationEnvironment?: string;         // アプリケーション環境
  applicationCulture?: string;             // アプリケーションカルチャ（BCP 47形式）
  resources?: Assets;                      // アセット定義
  appsettings?: string[];                  // appsettingsファイル
  virtualWorkingDirectory?: string;        // 仮想作業ディレクトリ
  applicationArguments?: string[];         // アプリケーション引数
}
```

**レスポンス**

戻り値: `DotnetHostBuilder`（メソッドチェーン用）

---

#### 2. withApplicationArguments

Main()メソッドに渡すコマンドライン引数を設定します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `withApplicationArguments(...args: string[]): DotnetHostBuilder` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| args | string[] | ○ | コマンドライン引数（可変長） |

**使用例**

```javascript
dotnet.withApplicationArguments("--verbose", "--config", "app.json")
```

---

#### 3. withEnvironmentVariable

単一の環境変数を設定します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `withEnvironmentVariable(name: string, value: string): DotnetHostBuilder` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| name | string | ○ | 環境変数名 |
| value | string | ○ | 環境変数値 |

---

#### 4. withEnvironmentVariables

複数の環境変数を一括設定します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `withEnvironmentVariables(variables: { [i: string]: string }): DotnetHostBuilder` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| variables | object | ○ | 環境変数の辞書オブジェクト |

---

#### 5. withVirtualWorkingDirectory

仮想ファイルシステム上の作業ディレクトリを設定します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `withVirtualWorkingDirectory(vfsPath: string): DotnetHostBuilder` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| vfsPath | string | ○ | 仮想ファイルシステムパス |

---

#### 6. withDiagnosticTracing

診断トレースメッセージの出力を有効化します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `withDiagnosticTracing(enabled: boolean): DotnetHostBuilder` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| enabled | boolean | ○ | トレース有効フラグ |

---

#### 7. withDebugging

デバッグレベルを設定します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `withDebugging(level: number): DotnetHostBuilder` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| level | number | ○ | デバッグレベル（> 0: デバッグ有効、= 0: 最適化有効、< 0: デバッグ有効・ログ無効） |

---

#### 8. withMainAssembly

Main()メソッドを含むアセンブリ名を指定します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `withMainAssembly(mainAssemblyName: string): DotnetHostBuilder` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| mainAssemblyName | string | ○ | アセンブリ名（デフォルトは.csproj名と同じ） |

---

#### 9. withApplicationArgumentsFromQuery

ブラウザURLのクエリパラメータから引数を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `withApplicationArgumentsFromQuery(): DotnetHostBuilder` |
| 認証 | 不要 |
| 権限 | なし |

**使用例**

```
// URL: index.html?arg=A&arg=B&arg=C
// Main()に ["A", "B", "C"] が渡される
dotnet.withApplicationArgumentsFromQuery()
```

---

#### 10. withApplicationEnvironment

アプリケーション環境名を設定します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `withApplicationEnvironment(applicationEnvironment?: string): DotnetHostBuilder` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| applicationEnvironment | string | - | 環境名（"Development", "Staging", "Production"等） |

---

#### 11. withApplicationCulture

アプリケーションカルチャを設定します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `withApplicationCulture(applicationCulture?: string): DotnetHostBuilder` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| applicationCulture | string | - | BCP 47形式のカルチャコード（例: "ja-JP", "en-US"） |

---

#### 12. withResourceLoader

ブートリソースのロード方法をオーバーライドします。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `withResourceLoader(loadBootResource?: LoadBootResourceCallback): DotnetHostBuilder` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| loadBootResource | LoadBootResourceCallback | - | カスタムリソースローダー |

**LoadBootResourceCallback型**

```typescript
type LoadBootResourceCallback = (
  type: WebAssemblyBootResourceType,
  name: string,
  defaultUri: string,
  integrity: string,
  behavior: AssetBehaviors
) => string | Promise<Response> | Promise<BootModule> | null | undefined;
```

---

#### 13. download

全てのアセットをダウンロードしますが、ランタイムインスタンスは作成しません。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `download(): Promise<void>` |
| 認証 | 不要 |
| 権限 | なし |

**レスポンス**

戻り値: `Promise<void>`

---

#### 14. create

ランタイムを起動してAPIオブジェクトを返します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `create(): Promise<RuntimeAPI>` |
| 認証 | 不要 |
| 権限 | なし |

**レスポンス**

戻り値: `Promise<RuntimeAPI>`

**使用例**

```javascript
const runtime = await dotnet
  .withConfig({ mainAssemblyName: "MyApp" })
  .create();
```

---

#### 15. runMain

Main()メソッドを実行し、ランタイムを維持します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `runMain(): Promise<number>` |
| 認証 | 不要 |
| 権限 | なし |

**レスポンス**

戻り値: `Promise<number>` - Main()の終了コード

---

#### 16. runMainAndExit

Main()メソッドを実行し、ランタイムを終了します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `runMainAndExit(): Promise<number>` |
| 認証 | 不要 |
| 権限 | なし |

**レスポンス**

戻り値: `Promise<number>` - Main()の終了コード

**注意**: ランタイム終了後は全てのAPI呼び出しが拒否されます。

---

### RuntimeAPI（ランタイムAPI）

#### 1. runMain

Main()メソッドを実行します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `runMain(mainAssemblyName?: string, args?: string[]): Promise<number>` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| mainAssemblyName | string | - | アセンブリ名（デフォルトは設定値） |
| args | string[] | - | コマンドライン引数 |

**レスポンス**

戻り値: `Promise<number>` - 終了コード

---

#### 2. runMainAndExit

Main()メソッドを実行してランタイムを終了します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `runMainAndExit(mainAssemblyName?: string, args?: string[]): Promise<number>` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| mainAssemblyName | string | - | アセンブリ名 |
| args | string[] | - | コマンドライン引数 |

---

#### 3. exit

ランタイムを終了します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `exit(code: number, reason?: any): void` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| code | number | ○ | 終了コード |
| reason | any | - | 終了理由（文字列またはErrorオブジェクト） |

---

#### 4. setEnvironmentVariable

環境変数を設定します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `setEnvironmentVariable(name: string, value: string): void` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| name | string | ○ | 環境変数名 |
| value | string | ○ | 環境変数値 |

---

#### 5. getAssemblyExports

指定アセンブリの[JSExport]メソッドを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `getAssemblyExports(assemblyName: string): Promise<any>` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| assemblyName | string | ○ | アセンブリ名 |

**レスポンス**

戻り値: `Promise<any>` - エクスポートされた関数オブジェクト

**使用例**

```javascript
const exports = await runtime.getAssemblyExports("MyApp");
const result = exports.MyNamespace.MyClass.MyMethod("arg");
```

---

#### 6. setModuleImports

マネージドコードから[JSImport]でインポート可能な関数を提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `setModuleImports(moduleName: string, moduleImports: any): void` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| moduleName | string | ○ | モジュール名（[JSImport]の第2引数に対応） |
| moduleImports | any | ○ | インポート可能な関数オブジェクト |

**使用例**

```javascript
runtime.setModuleImports("myModule", {
  alert: (message) => window.alert(message),
  log: (message) => console.log(message)
});
```

---

#### 7. getConfig

ランタイム起動時の設定オブジェクトを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `getConfig(): LoaderConfig` |
| 認証 | 不要 |
| 権限 | なし |

**レスポンス**

戻り値: `LoaderConfig` - 設定オブジェクト

---

#### 8. invokeLibraryInitializers

ランタイムブートストラップ中にロードされたスクリプトを実行します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `invokeLibraryInitializers(functionName: string, args: any[]): Promise<void>` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| functionName | string | ○ | 関数名 |
| args | any[] | ○ | 引数配列 |

---

### MemoryAPI（メモリAPI）

WASMリニアメモリへの読み書きを行うAPIです。

#### Heap書き込み関数群

| 関数名 | シグネチャ | 説明 |
| --- | --- | --- |
| setHeapB32 | `(offset: NativePointer, value: number \| boolean): void` | 32ビットブール値を書き込み |
| setHeapB8 | `(offset: NativePointer, value: number \| boolean): void` | 8ビットブール値を書き込み |
| setHeapU8 | `(offset: NativePointer, value: number): void` | 符号なし8ビット整数を書き込み |
| setHeapU16 | `(offset: NativePointer, value: number): void` | 符号なし16ビット整数を書き込み |
| setHeapU32 | `(offset: NativePointer, value: NativePointer \| number): void` | 符号なし32ビット整数を書き込み |
| setHeapI8 | `(offset: NativePointer, value: number): void` | 符号付き8ビット整数を書き込み |
| setHeapI16 | `(offset: NativePointer, value: number): void` | 符号付き16ビット整数を書き込み |
| setHeapI32 | `(offset: NativePointer, value: number): void` | 符号付き32ビット整数を書き込み |
| setHeapI52 | `(offset: NativePointer, value: number): void` | 52ビット整数を書き込み |
| setHeapU52 | `(offset: NativePointer, value: number): void` | 符号なし52ビット整数を書き込み |
| setHeapI64Big | `(offset: NativePointer, value: bigint): void` | 64ビット整数（BigInt）を書き込み |
| setHeapF32 | `(offset: NativePointer, value: number): void` | 32ビット浮動小数点数を書き込み |
| setHeapF64 | `(offset: NativePointer, value: number): void` | 64ビット浮動小数点数を書き込み |

#### Heap読み取り関数群

| 関数名 | シグネチャ | 説明 |
| --- | --- | --- |
| getHeapB32 | `(offset: NativePointer): boolean` | 32ビットブール値を読み取り |
| getHeapB8 | `(offset: NativePointer): boolean` | 8ビットブール値を読み取り |
| getHeapU8 | `(offset: NativePointer): number` | 符号なし8ビット整数を読み取り |
| getHeapU16 | `(offset: NativePointer): number` | 符号なし16ビット整数を読み取り |
| getHeapU32 | `(offset: NativePointer): number` | 符号なし32ビット整数を読み取り |
| getHeapI8 | `(offset: NativePointer): number` | 符号付き8ビット整数を読み取り |
| getHeapI16 | `(offset: NativePointer): number` | 符号付き16ビット整数を読み取り |
| getHeapI32 | `(offset: NativePointer): number` | 符号付き32ビット整数を読み取り |
| getHeapI52 | `(offset: NativePointer): number` | 52ビット整数を読み取り |
| getHeapU52 | `(offset: NativePointer): number` | 符号なし52ビット整数を読み取り |
| getHeapI64Big | `(offset: NativePointer): bigint` | 64ビット整数（BigInt）を読み取り |
| getHeapF32 | `(offset: NativePointer): number` | 32ビット浮動小数点数を読み取り |
| getHeapF64 | `(offset: NativePointer): number` | 64ビット浮動小数点数を読み取り |

#### Heapビュー関数群

**注意**: これらの関数が返すビューは短期間のみ有効です。参照を保持したり、awaitの後に使用しないでください。

| 関数名 | シグネチャ | 説明 |
| --- | --- | --- |
| localHeapViewI8 | `(): Int8Array` | Int8Arrayビューを取得 |
| localHeapViewI16 | `(): Int16Array` | Int16Arrayビューを取得 |
| localHeapViewI32 | `(): Int32Array` | Int32Arrayビューを取得 |
| localHeapViewI64Big | `(): BigInt64Array` | BigInt64Arrayビューを取得 |
| localHeapViewU8 | `(): Uint8Array` | Uint8Arrayビューを取得 |
| localHeapViewU16 | `(): Uint16Array` | Uint16Arrayビューを取得 |
| localHeapViewU32 | `(): Uint32Array` | Uint32Arrayビューを取得 |
| localHeapViewF32 | `(): Float32Array` | Float32Arrayビューを取得 |
| localHeapViewF64 | `(): Float64Array` | Float64Arrayビューを取得 |

---

### DiagnosticsAPI（診断API）

#### 1. collectCpuSamples

CPUサンプルを収集して診断トレースファイルを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `collectCpuSamples(options?: DiagnosticCommandOptions): Promise<Uint8Array[]>` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| options | DiagnosticCommandOptions | - | 診断オプション |

**DiagnosticCommandOptions型**

```typescript
{
  durationSeconds?: number;              // 収集期間（デフォルト: 60秒）
  intervalSeconds?: number;              // インターバル
  skipDownload?: boolean;                // ダウンロードスキップ
  circularBufferMB?: number;             // 循環バッファサイズ（MB）
  extraProviders?: DiagnosticCommandProviderV2[];  // 追加プロバイダー
}
```

**レスポンス**

戻り値: `Promise<Uint8Array[]>` - .nettraceファイル（PerfViewまたはVisual Studioで開けます）

---

#### 2. collectMetrics

メトリクスを収集して診断トレースファイルを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `collectMetrics(options?: DiagnosticCommandOptions): Promise<Uint8Array[]>` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| options | DiagnosticCommandOptions | - | 診断オプション |

**レスポンス**

戻り値: `Promise<Uint8Array[]>` - .nettraceファイル

**注意**: `dotnet-trace report xxx.nettrace topN -n 10` で要約可能

---

#### 3. collectGcDump

GCダンプを収集します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `collectGcDump(options?: DiagnosticCommandOptions): Promise<Uint8Array[]>` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| options | DiagnosticCommandOptions | - | 診断オプション |

**レスポンス**

戻り値: `Promise<Uint8Array[]>` - GCダンプファイル

**注意**: PerfViewでそのまま開けます。Visual Studio用には `dotnet-gcdump convert` で変換が必要です。

---

#### 4. connectDSRouter

診断サーバールーターに接続します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `connectDSRouter(url: string): void` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| url | string | ○ | WebSocket URL（ws://またはwss://） |

**使用方法**

```bash
# サーバー側
dotnet-dsrouter server-websocket -v trace
```

---

### HTTP Interop API

マネージドコードから使用されるHTTPクライアントインターフェースです。

#### 1. httpCreateController

HTTPリクエスト制御用のコントローラーを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `httpCreateController(): HttpController` |
| 認証 | 不要 |
| 権限 | なし |

**前提条件**

- ブラウザ: Fetch APIおよびAbortControllerのサポート
- Node.js: `node-fetch`および`node-abort-controller`パッケージが必要

---

#### 2. httpFetch

HTTPリクエストを実行します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `httpFetch(controller, url, headerNames, headerValues, optionNames, optionValues, body): Promise<void>` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| controller | HttpController | ○ | HTTPコントローラー |
| url | string | ○ | リクエストURL |
| headerNames | string[] | ○ | ヘッダー名配列 |
| headerValues | string[] | ○ | ヘッダー値配列 |
| optionNames | string[] | ○ | オプション名配列 |
| optionValues | any[] | ○ | オプション値配列 |
| body | Uint8Array \| ReadableStream \| null | ○ | リクエストボディ |

---

#### 3. httpFetchBytes

バイト配列ボディでHTTPリクエストを実行します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `httpFetchBytes(controller, url, headerNames, headerValues, optionNames, optionValues, bodyPtr, bodyLength): Promise<void>` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 4. httpFetchStream

ストリーミングボディでHTTPリクエストを実行します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `httpFetchStream(controller, url, headerNames, headerValues, optionNames, optionValues): Promise<void>` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 5. httpAbort

HTTPリクエストを中止します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `httpAbort(controller: HttpController): void` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 6. httpGetResponseStatus

レスポンスのHTTPステータスコードを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `httpGetResponseStatus(controller: HttpController): number` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 7. httpGetResponseHeaderNames / httpGetResponseHeaderValues

レスポンスヘッダーを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `httpGetResponseHeaderNames(controller): string[]` / `httpGetResponseHeaderValues(controller): string[]` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 8. httpSupportsStreamingRequest / httpSupportsStreamingResponse

ストリーミングのサポート状況を確認します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `httpSupportsStreamingRequest(): boolean` / `httpSupportsStreamingResponse(): boolean` |
| 認証 | 不要 |
| 権限 | なし |

---

### WebSocket Interop API

マネージドコードから使用されるWebSocketインターフェースです。

#### 1. wsCreate

WebSocket接続を作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `wsCreate(uri: string, subProtocols: string[] \| null, receiveStatusPtr: VoidPtr): WebSocketExtension` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| uri | string | ○ | WebSocket URI (ws://またはwss://) |
| subProtocols | string[] \| null | - | サブプロトコル配列 |
| receiveStatusPtr | VoidPtr | ○ | 受信ステータス格納ポインタ |

**前提条件**

- ブラウザ: WebSocket APIのサポート
- Node.js: `ws`パッケージが必要

---

#### 2. wsOpen

WebSocket接続の完了を待機します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `wsOpen(ws: WebSocketExtension): Promise<WebSocketExtension> \| null` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 3. wsSend

WebSocketでデータを送信します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `wsSend(ws, bufferPtr, bufferLength, messageType, endOfMessage): Promise<void> \| null` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| ws | WebSocketExtension | ○ | WebSocketインスタンス |
| bufferPtr | VoidPtr | ○ | バッファポインタ |
| bufferLength | number | ○ | バッファ長 |
| messageType | number | ○ | メッセージタイプ（0: Text, 1: Binary） |
| endOfMessage | boolean | ○ | メッセージ終端フラグ |

---

#### 4. wsReceive

WebSocketからデータを受信します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `wsReceive(ws, bufferPtr, bufferLength): Promise<void> \| null` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 5. wsClose

WebSocket接続を閉じます。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `wsClose(ws, code, reason, waitForCloseReceived): Promise<void> \| null` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| ws | WebSocketExtension | ○ | WebSocketインスタンス |
| code | number | ○ | クローズコード |
| reason | string \| null | - | クローズ理由 |
| waitForCloseReceived | boolean | ○ | クローズ受信待機フラグ |

---

#### 6. wsAbort

WebSocket接続を中止します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `wsAbort(ws: WebSocketExtension): void` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 7. wsGetState

WebSocketの状態を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `wsGetState(ws: WebSocketExtension): number` |
| 認証 | 不要 |
| 権限 | なし |

**レスポンス**

| 値 | 状態 |
| --- | --- |
| 0 | CONNECTING |
| 1 | OPEN |
| 2 | CLOSING |
| 3 | CLOSED |

---

## 型定義

### GlobalizationMode

```typescript
enum GlobalizationMode {
  Sharded = "sharded",   // シャード化されたICUデータをロード
  All = "all",           // 全ICUデータをロード
  Invariant = "invariant", // 不変グローバリゼーションモード
  Custom = "custom"      // ユーザー定義ICUファイル使用
}
```

### RuntimeAPI

```typescript
type RuntimeAPI = {
  INTERNAL: any;
  Module: EmscriptenModule;
  runtimeId: number;
  runtimeBuildInfo: {
    productVersion: string;
    gitHash: string;
    buildConfiguration: string;
    wasmEnableThreads: boolean;
    wasmEnableSIMD: boolean;
    wasmEnableExceptionHandling: boolean;
  };
} & RunAPIType & MemoryAPIType & DiagnosticsAPIType;
```

### IMemoryView

```typescript
interface IMemoryView extends IDisposable {
  set(source: TypedArray, targetOffset?: number): void;
  copyTo(target: TypedArray, sourceOffset?: number): void;
  slice(start?: number, end?: number): TypedArray;
  readonly length: number;
  readonly byteLength: number;
}
```

### IDisposable

```typescript
interface IDisposable {
  dispose(): void;
  readonly isDisposed: boolean;
}
```

## 備考

### 環境要件

- **ブラウザ**:
  - Fetch API、AbortController、WebSocket APIをサポートするモダンブラウザ
  - ストリーミングリクエストにはduplex: "half"のサポートが必要（Safari非対応）

- **Node.js**:
  - HTTPサポート: `node-fetch`、`node-abort-controller`パッケージ
  - WebSocketサポート: `ws`パッケージ

### 参考リンク

- [.NET WebAssembly Features](https://aka.ms/dotnet-wasm-features)
- [BCP 47言語タグ仕様](https://tools.ietf.org/html/bcp47)
- [Fetch Streaming Requests](https://developer.chrome.com/articles/fetch-streaming-requests/)

### ライセンス

MIT License - .NET Foundation
