# アーキテクチャ設計書

## 概要

本ドキュメントは、.NET Runtime（dotnet/runtime）のアーキテクチャ設計書です。.NET Runtimeは、.NETアプリケーションを実行するための包括的なランタイム環境であり、複数のランタイムエンジン（CoreCLR、Mono）、基盤ライブラリ、およびホストインストーラーから構成されます。

本システムは、ガベージコレクション、JITコンパイル、型システム、スレッド管理などの機能を提供し、Windows、Linux、macOS、WebAssemblyなど多様なプラットフォームでマネージドコードの実行を可能にします。

## システム全体構成

### アーキテクチャ概要図

[アーキテクチャ構成図.md](./アーキテクチャ構成図.md) を参照してください。

### システム境界と外部連携

| 外部システム | 連携方式 | 用途 |
| --- | --- | --- |
| オペレーティングシステム（Windows/Linux/macOS/FreeBSD） | ネイティブAPI呼び出し（P/Invoke、syscall） | メモリ管理、スレッド管理、ファイルI/O、ネットワーク通信 |
| ネイティブライブラリ | P/Invoke、COM相互運用 | ネイティブコードとの相互運用 |
| NuGetパッケージリポジトリ | HTTPS/REST API | パッケージ取得・配布 |
| Azure DevOps | CI/CDパイプライン | ビルド、テスト、リリース自動化 |
| Helix（テスト基盤） | REST API | 分散テスト実行 |

## レイヤー構成

### アーキテクチャスタイル

本システムは**コンポーネントベースアーキテクチャ**を採用しています。複数の独立したコンポーネント（ランタイム、ライブラリ、ホスト）が明確な責務分離のもとで連携し、クロスプラットフォーム実行環境を提供します。

各コンポーネントは以下の設計原則に基づいています：

- **プラットフォーム抽象化層（PAL）**: OS固有の実装を抽象化
- **世代別ガベージコレクション**: 効率的なメモリ管理
- **JITコンパイル**: プラットフォーム非依存の中間言語から最適化されたネイティブコードへの変換

### レイヤー定義

| レイヤー | 責務 | 主なコンポーネント |
| --- | --- | --- |
| Host | .NETアプリケーションの起動、ランタイム選択、依存関係解決 | dotnet host, apphost, hostfxr, hostpolicy |
| Runtime (CoreCLR/Mono) | マネージドコードの実行、メモリ管理、型システム、JIT/AOTコンパイル | VM, JIT (RyuJIT), GC, Type System, Interop |
| Libraries | 基盤クラスライブラリ、プラットフォームAPIの抽象化 | System.*, Microsoft.Extensions.* |
| Native | プラットフォーム固有のネイティブコード実装 | corehost, eventpipe, containers |

### レイヤー間の依存関係ルール

```
Host → Runtime → Libraries → Native
         ↓
    Platform (OS)
```

**依存の方向性:**
- Hostレイヤーはランタイムのロードと初期化を担当
- Runtimeレイヤーはライブラリおよびネイティブコードを利用
- ライブラリはランタイム機能（GC、型システム）を利用
- すべてのレイヤーはプラットフォーム抽象化層を通じてOSと連携

**許可される参照:**
- 上位レイヤーから下位レイヤーへの参照
- 同一レイヤー内のコンポーネント間参照

**禁止される参照:**
- 下位レイヤーから上位レイヤーへの直接参照（コールバックインターフェースを除く）

## モジュール構成

### ドメイン分割

| ドメイン | 責務 | 関連モジュール |
| --- | --- | --- |
| CoreCLR | 高性能.NETランタイム実行エンジン | vm/, jit/, gc/, interpreter/ |
| Mono | 軽量・クロスプラットフォームランタイム | mono/, browser/, wasm/, wasi/ |
| Libraries | 基盤クラスライブラリ | System.*, Microsoft.* |
| Installer | パッケージング・配布 | pkg/, managed/ |
| Tasks | ビルドタスク・ツール | AppleAppBuilder, WasmAppBuilder, MonoAOTCompiler |

### パッケージ構造

```
src/
├── coreclr/                    # CoreCLRランタイム
│   ├── System.Private.CoreLib/ # CoreCLR用CoreLib
│   ├── vm/                     # 仮想マシン実装
│   ├── jit/                    # RyuJIT コンパイラ
│   ├── gc/                     # ガベージコレクター
│   ├── interpreter/            # インタープリター
│   ├── nativeaot/              # Native AOT
│   ├── pal/                    # プラットフォーム抽象化層
│   └── tools/                  # 開発ツール
├── mono/                       # Monoランタイム
│   ├── System.Private.CoreLib/ # Mono用CoreLib
│   ├── mono/                   # Monoコアエンジン
│   ├── browser/                # ブラウザ(WebAssembly)サポート
│   ├── wasm/                   # WebAssemblyサポート
│   └── wasi/                   # WASIサポート
├── libraries/                  # 基盤クラスライブラリ
│   ├── System.*/               # システムライブラリ
│   ├── Microsoft.Extensions.*/ # 拡張ライブラリ
│   └── Common/                 # 共通コード
├── native/                     # ネイティブコンポーネント
│   ├── corehost/               # ホストネイティブ実装
│   ├── eventpipe/              # イベントパイプライン
│   └── libs/                   # ネイティブライブラリ
├── installer/                  # インストーラー
│   └── pkg/                    # パッケージ定義
├── tasks/                      # MSBuildタスク
└── tools/                      # 開発ツール
```

### コンポーネント依存関係

**CoreCLRコンポーネント:**
- VM → JIT、GC、Type System、Interop
- JIT → VM（型情報取得）、GC（GC参照追跡）
- GC → VM（実行エンジン連携）

**ライブラリコンポーネント:**
- System.Private.CoreLib → Runtime（CoreCLR/Mono）
- 各System.* → System.Private.CoreLib
- Microsoft.Extensions.* → System.*

## ミドルウェア構成

### データベース

本プロジェクトはランタイム環境であり、データベースミドルウェアは含みません。ただし、以下のライブラリでデータベース接続をサポートします：

| 種類 | ミドルウェア | バージョン | 用途 |
| --- | --- | --- | --- |
| ADO.NET | System.Data.Common | .NET 11.0 | データアクセス抽象化 |
| ODBC | System.Data.Odbc | .NET 11.0 | ODBC接続 |

### キャッシュ

| ミドルウェア | バージョン | 用途 | TTL |
| --- | --- | --- | --- |
| Microsoft.Extensions.Caching.Memory | .NET 11.0 | インメモリキャッシュ | 設定可能 |
| Microsoft.Extensions.Caching.Abstractions | .NET 11.0 | キャッシュ抽象化 | N/A |

### メッセージキュー

本プロジェクトはメッセージキューミドルウェアを直接含みませんが、以下の非同期処理機構を提供します：

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| System.Threading.Channels | .NET 11.0 | インプロセスメッセージング |
| System.Threading.Tasks.Dataflow | .NET 11.0 | データフローパイプライン |

### 検索エンジン

該当なし（ランタイム環境として検索エンジンは含まない）

### その他ミドルウェア

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| EventPipe | .NET 11.0 | 診断・トレーシング |
| System.Diagnostics.DiagnosticSource | .NET 11.0 | 診断イベント |
| Microsoft.Extensions.Logging | .NET 11.0 | ロギング抽象化 |

## データフロー

### リクエスト処理の流れ

1. **アプリケーション起動**: dotnet hostがアプリケーションを起動
2. **ランタイム初期化**: hostfxrがランタイム（CoreCLR/Mono）をロード・初期化
3. **アセンブリロード**: CLRがアセンブリをロードし、型システムを構築
4. **JITコンパイル**: メソッド呼び出し時にILをネイティブコードにコンパイル
5. **マネージドコード実行**: ネイティブコードを実行
6. **GC実行**: 必要に応じてガベージコレクションを実行
7. **結果返却**: 実行結果をアプリケーションに返却

### 非同期処理の流れ

1. **タスク生成**: async/awaitまたはTask.Runでタスクを生成
2. **スレッドプールキューイング**: タスクをスレッドプールにエンキュー
3. **ワーカースレッド実行**: スレッドプールワーカーがタスクを実行
4. **継続処理**: await後の処理を継続として登録
5. **完了通知**: タスク完了時に継続処理を実行

### データ永続化の流れ

本システムはランタイム環境であり、直接的なデータ永続化機能は提供しません。アプリケーションはSystem.IO等のライブラリを通じてファイルシステムやデータベースにアクセスします。

## 横断的関心事

### 認証・認可

| 方式 | 実装箇所 | 対象 |
| --- | --- | --- |
| コード署名検証 | CLR Assembly Loader | アセンブリの信頼性検証 |
| Strong Name検証 | Type System | アセンブリ参照の整合性 |
| CAS (非推奨) | Security System | コードアクセスセキュリティ |

### ロギング・監査

| 種類 | 実装方式 | 保存先 |
| --- | --- | --- |
| アプリケーションログ | Microsoft.Extensions.Logging | 設定可能（Console、File、EventLog等） |
| 監査ログ | EventPipe | ETW/LTTng（OS診断システム） |
| アクセスログ | DiagnosticSource | アプリケーション定義 |
| GC/JITログ | EventPipe | ETW/LTTng |

### エラーハンドリング

マネージドコードでは例外ベースのエラーハンドリングを採用：

| エラー種別 | ハンドリング方式 | レスポンス |
| --- | --- | --- |
| マネージド例外 | try-catch-finally | 例外オブジェクト |
| ネイティブ例外 | SEH/シグナルハンドラ | 例外に変換 |
| スタックオーバーフロー | 専用ハンドラ | StackOverflowException |
| アクセス違反 | 専用ハンドラ | AccessViolationException（一部） |
| アウトオブメモリ | GC/アロケータ | OutOfMemoryException |

### トランザクション管理

| 範囲 | 管理方式 | 分離レベル |
| --- | --- | --- |
| 分散トランザクション | System.Transactions | アプリケーション定義 |
| ローカルトランザクション | ADO.NET | DB依存 |

## 設計原則・コーディング規約

### 適用している設計原則

| 原則 | 適用箇所 | 実装例 |
| --- | --- | --- |
| プラットフォーム抽象化 | PAL (Platform Abstraction Layer) | OS固有コードの分離（src/coreclr/pal/） |
| 世代別GC | Garbage Collector | Gen0/Gen1/Gen2による効率的メモリ管理 |
| 遅延JITコンパイル | JIT Compiler | 呼び出し時コンパイルによる起動時間短縮 |
| 階層型コンパイル | Tiered Compilation | 実行頻度に応じた最適化レベル調整 |
| 不変文字列 | System.String | 文字列の不変性によるAPI簡素化・スレッド安全性 |
| 値型最適化 | Type System | Span<T>、ValueTask等による割り当て削減 |

### コーディング規約

本プロジェクトでは以下の規約を適用：

**C#コード:**
- 警告をエラーとして扱う（TreatWarningsAsErrors=true）
- LangVersion=preview（最新言語機能を使用）
- Nullable annotations有効
- 一貫した命名規則（docs/coding-guidelines/coding-style.md参照）

**C/C++コード:**
- .clang-formatによるフォーマット統一
- .clang-tidyによる静的解析

**ファイルヘッダー:**
```
// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
```

## 備考

### サポートプラットフォーム

| OS | アーキテクチャ | ビルドサポート | ターゲットサポート |
| --- | --- | --- | --- |
| Windows | x64, x86, ARM64 | Yes | Yes |
| Linux | x64, ARM32, ARM64 | Yes | Yes |
| macOS | x64, ARM64 | Yes | Yes |
| FreeBSD | x64 | Yes | Yes |
| iOS/tvOS | ARM64 | Cross-compile | Yes |
| Android | ARM64, x64 | Cross-compile | Yes |
| WebAssembly | wasm32 | Cross-compile | Yes |

### バージョン情報

- .NET Version: 11.0 (Preview)
- SDK Version: 11.0.100-alpha
- Microsoft.DotNet.Arcade.Sdk: 11.0.0-beta

### 関連ドキュメント

- [Book of the Runtime (BOTR)](https://github.com/dotnet/runtime/tree/main/docs/design/coreclr/botr)
- [Workflow Guide](https://github.com/dotnet/runtime/blob/main/docs/workflow/README.md)
- [Contributing Guide](https://github.com/dotnet/runtime/blob/main/CONTRIBUTING.md)
