# .NET Runtime コードリーディングガイドライン

## はじめに

このガイドラインは、.NET Runtime（dotnet/runtime）のコードベースを効率的に理解するための手引きです。
C#やC++に精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者
- コードレビューを行う担当者

---

## 1. 言語基礎

> このセクションでは、本プロジェクトで使用される主要言語の基本構文と概念を解説します。

### 1.1 使用言語一覧

本プロジェクトでは以下の言語が使用されています：

| 言語 | 用途 | 主要ディレクトリ |
|-----|------|-----------------|
| C# | マネージドライブラリ、CoreLib | `src/libraries/`, `src/coreclr/System.Private.CoreLib/` |
| C/C++ | ランタイム（CoreCLR/Mono）、JIT、GC | `src/coreclr/`, `src/mono/` |
| IL (Intermediate Language) | 低レベルアセンブリ | `src/coreclr/ilasm/` |
| CMake | ネイティブビルドシステム | プロジェクト全体 |
| MSBuild/XML | マネージドビルドシステム | `.csproj`, `.props`, `.targets` |

### 1.2 C# プログラム構造

C#はこのプロジェクトのマネージドコードで使用される主要言語です。

**基本構造:**
```csharp
// ファイル: src/libraries/System.Collections/src/System/Collections/Generic/Stack.cs:1-10
// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System.Diagnostics;
using System.Diagnostics.CodeAnalysis;
using System.Runtime.CompilerServices;

namespace System.Collections.Generic
{
    // クラス定義がここに続く
}
```

**ポイント:**
- ファイル先頭には必ずライセンスヘッダーが付与される
- `using`ディレクティブは名前空間宣言の外側に配置
- `System.*`名前空間を先頭に、その他はアルファベット順

### 1.3 データ型と変数

**フィールド命名規則（C#）:**
```csharp
// ファイル: src/libraries/System.Collections/src/System/Collections/Generic/Stack.cs:29-33
private T[] _array; // プライベートフィールドは `_` プレフィックス
private int _size;
private int _version;

private const int DefaultCapacity = 4; // 定数はPascalCase
```

**静的フィールドの命名:**
```csharp
// ファイル: src/libraries/System.Private.CoreLib/src/System/Collections/Generic/List.cs:30
private static readonly T[] s_emptyArray = new T[0]; // 静的フィールドは `s_` プレフィックス
```

### 1.4 制御構造

**条件分岐の例:**
```csharp
// ファイル: src/libraries/System.Collections/src/System/Collections/Generic/Stack.cs:192-203
public T Peek()
{
    int size = _size - 1;
    T[] array = _array;

    if ((uint)size >= (uint)array.Length)
    {
        ThrowForEmptyStack();
    }

    return array[size];
}
```

**ポイント:**
- Allmanスタイル（波括弧は新しい行に配置）
- 4スペースインデント（タブ不使用）
- 単一文でも波括弧を推奨

### 1.5 関数/メソッド定義

**メソッド定義の例:**
```csharp
// ファイル: src/libraries/System.Collections/src/System/Collections/Generic/Stack.cs:266-281
public void Push(T item)
{
    int size = _size;
    T[] array = _array;

    if ((uint)size < (uint)array.Length)
    {
        array[size] = item;
        _version++;
        _size = size + 1;
    }
    else
    {
        PushWithResize(item);
    }
}
```

### 1.6 モジュール/インポート

**using ディレクティブ:**
```csharp
// ファイル: src/libraries/System.Private.CoreLib/src/System/Collections/Generic/List.cs:4-7
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.Diagnostics.CodeAnalysis;
using System.Runtime.CompilerServices;
```

---

## 2. プロジェクト固有の概念

> このセクションでは、.NET Runtime特有の概念を解説します。

### 2.1 ランタイムの二重構成

.NET Runtimeには2つの実行エンジンが存在します：

| ランタイム | 特徴 | 主な用途 |
|-----------|------|---------|
| **CoreCLR** | 高性能、フル機能のJIT | クラウド（ASP.NET）、デスクトップ（WPF, WinForms） |
| **Mono** | 軽量、クロスプラットフォーム | モバイル（Xamarin）、ブラウザ（Blazor） |

### 2.2 主要コンポーネント

```
.NET Runtime
├── Runtime (実行エンジン)
│   ├── CoreCLR - src/coreclr/
│   └── Mono - src/mono/
├── CoreLib (System.Private.CoreLib)
│   ├── 共通コード - src/libraries/System.Private.CoreLib/
│   ├── CoreCLR固有 - src/coreclr/System.Private.CoreLib/
│   └── Mono固有 - src/mono/System.Private.CoreLib/
├── Libraries (ライブラリ群) - src/libraries/
└── Host & Installer - src/installer/
```

### 2.3 ビルド構成

| 構成 | 最適化 | アサート | 用途 |
|-----|--------|---------|------|
| Debug | なし | 有効 | デバッグ、開発 |
| Checked | あり | 有効 | CoreCLR専用、パフォーマンステスト |
| Release | あり | 無効 | 本番、パフォーマンスプロファイリング |

### 2.4 CONTRACTL マクロ（C++）

CoreCLRのC++コードでは、関数の契約を明示するCONTRACTLマクロが使用されます：

```cpp
// ファイル: src/coreclr/vm/class.cpp:40-46
void *EEClass::operator new(
    size_t size,
    LoaderHeap *pHeap,
    AllocMemTracker *pamTracker)
{
    CONTRACTL
    {
        THROWS;          // 例外をスローする可能性
        GC_NOTRIGGER;    // GCをトリガーしない
        INJECT_FAULT(COMPlusThrowOM());  // メモリ不足時の動作
    }
    CONTRACTL_END;
    // 実装...
}
```

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `System.{名前}` | 標準ライブラリ名前空間 | `System.Collections`, `System.IO` |
| `{クラス名}.cs` | 単一クラスファイル | `Stack.cs`, `List.cs` |
| `{クラス名}.{部分名}.cs` | 部分クラスファイル | `SortedSet.TreeSubSet.cs` |
| `ref/` | 参照アセンブリ | `src/libraries/*/ref/` |
| `src/` | ソースコード | `src/libraries/*/src/` |
| `tests/` | テストコード | `src/libraries/*/tests/` |

### 3.2 クラス・関数・変数命名

| 要素 | 命名規則 | 例 |
|-----|---------|-----|
| クラス・構造体 | PascalCase | `Stack<T>`, `LinkedList<T>` |
| パブリックメソッド | PascalCase | `Push()`, `Pop()`, `GetEnumerator()` |
| プライベートメソッド | PascalCase | `PushWithResize()`, `ThrowForEmptyStack()` |
| パブリックプロパティ | PascalCase | `Count`, `Capacity` |
| プライベートフィールド | `_camelCase` | `_array`, `_size`, `_version` |
| 静的フィールド | `s_camelCase` | `s_emptyArray` |
| スレッドスタティック | `t_camelCase` | `t_threadLocal` |
| 定数 | PascalCase | `DefaultCapacity` |
| ローカル変数 | camelCase | `size`, `array`, `item` |
| パラメータ | camelCase | `capacity`, `collection` |

### 3.3 プレフィックス/サフィックス一覧

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `I` | インターフェース | `IEnumerable`, `ICollection` |
| `T` | ジェネリック型パラメータ | `T`, `TKey`, `TValue` |
| `_` | プライベートインスタンスフィールド | `_array`, `_count` |
| `s_` | プライベート静的フィールド | `s_emptyArray` |
| `t_` | スレッドスタティックフィールド | `t_current` |
| `Async` | 非同期メソッド | `ReadAsync()` |
| `Internal` | 内部使用のみ | `InternalMethod()` |
| `Try` | 失敗時にfalseを返す | `TryPeek()`, `TryPop()` |

### 3.4 C++ 命名規則（CoreCLR）

| 要素 | 命名規則 | 例 |
|-----|---------|-----|
| クラス | PascalCase | `EEClass`, `MethodTable` |
| メソッド | PascalCase | `Destruct()`, `GetComClassFactory()` |
| メンバー変数 | `m_` プレフィックス | `m_pccwTemplate` |
| ポインタ | `p` プレフィックス | `pHeap`, `pamTracker` |
| マクロ | UPPER_SNAKE_CASE | `CONTRACTL`, `GC_NOTRIGGER` |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
runtime-main/
├── src/                          # ソースコード
│   ├── coreclr/                  # CoreCLRランタイム
│   │   ├── System.Private.CoreLib/  # CoreCLR用CoreLib
│   │   ├── vm/                   # 仮想マシン実装
│   │   ├── jit/                  # JITコンパイラ（RyuJIT）
│   │   ├── gc/                   # ガベージコレクタ
│   │   ├── debug/                # デバッガサポート
│   │   └── tools/                # 開発ツール
│   ├── mono/                     # Monoランタイム
│   │   ├── System.Private.CoreLib/  # Mono用CoreLib
│   │   ├── mono/                 # Monoコア実装
│   │   ├── wasm/                 # WebAssemblyサポート
│   │   └── browser/              # ブラウザサポート
│   ├── libraries/                # マネージドライブラリ
│   │   ├── System.Collections/   # コレクションライブラリ
│   │   ├── System.IO/            # I/Oライブラリ
│   │   ├── System.Net.*/         # ネットワークライブラリ
│   │   └── ...                   # その他多数
│   ├── installer/                # インストーラー
│   ├── native/                   # 共有ネイティブコード
│   ├── tasks/                    # MSBuildタスク
│   └── tests/                    # 共通テストインフラ
├── docs/                         # ドキュメント
│   ├── workflow/                 # ビルド・テストワークフロー
│   ├── coding-guidelines/        # コーディングガイドライン
│   ├── design/                   # 設計ドキュメント
│   └── project/                  # プロジェクト情報
├── eng/                          # エンジニアリングインフラ
└── build.sh / build.cmd          # ビルドスクリプト
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `src/coreclr/` | CoreCLRランタイムのソースコード | `vm/*.cpp`, `jit/*.cpp` |
| `src/mono/` | Monoランタイムのソースコード | `mono/**/*.c` |
| `src/libraries/` | マネージドライブラリのソースコード | `*.cs`, `*.csproj` |
| `src/libraries/*/ref/` | 参照アセンブリ（APIシグネチャ） | `*.cs` |
| `src/libraries/*/src/` | 実装コード | `*.cs` |
| `src/libraries/*/tests/` | ユニットテスト | `*Tests.cs` |
| `docs/` | ドキュメント | `*.md` |
| `eng/` | ビルドインフラストラクチャ | `*.props`, `*.targets` |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

.NET Runtimeは、マネージドコードとネイティブコードが階層的に構成されたレイヤードアーキテクチャを採用しています。

```mermaid
graph TB
    subgraph "アプリケーション層"
        APP[.NET アプリケーション]
    end

    subgraph "ライブラリ層"
        BCL[Base Class Library<br/>System.*, Microsoft.*]
    end

    subgraph "CoreLib層"
        SPCL[System.Private.CoreLib<br/>String, Object, Array, etc.]
    end

    subgraph "ランタイム層"
        CLR[CoreCLR / Mono]
        JIT[JIT Compiler]
        GC[Garbage Collector]
    end

    subgraph "PAL層"
        PAL[Platform Adaptation Layer]
    end

    subgraph "OS層"
        OS[Windows / Linux / macOS / etc.]
    end

    APP --> BCL
    BCL --> SPCL
    SPCL --> CLR
    CLR --> JIT
    CLR --> GC
    CLR --> PAL
    PAL --> OS
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| アプリケーション | ユーザーアプリケーション | - |
| BCL | 高レベルAPI（コレクション、I/O、ネットワーク等） | `src/libraries/System.Collections/src/**/*.cs` |
| CoreLib | 最低レベルマネージド型（Object, String等） | `src/libraries/System.Private.CoreLib/src/**/*.cs` |
| VM | 仮想マシン実装 | `src/coreclr/vm/*.cpp` |
| JIT | Just-In-Timeコンパイラ | `src/coreclr/jit/*.cpp` |
| GC | ガベージコレクタ | `src/coreclr/gc/*.cpp` |
| PAL | プラットフォーム抽象化 | `src/coreclr/pal/**/*.cpp` |

### 5.3 データフロー

**マネージドコード実行フロー:**

```
1. C# ソースコード
   ↓ (Roslyn コンパイラ - 別リポジトリ)
2. IL (Intermediate Language)
   ↓ (JIT コンパイラ - src/coreclr/jit/)
3. ネイティブマシンコード
   ↓ (実行エンジン - src/coreclr/vm/)
4. 実行結果
```

**ライブラリの依存関係:**

```
アプリケーション
    ↓ 参照
System.Collections (高レベルコレクション)
    ↓ 依存
System.Private.CoreLib (基本型)
    ↓ 内部呼び出し
CoreCLR/Mono VM (ネイティブ実装)
```

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

**ライブラリのエントリーポイント:**

各ライブラリのエントリーポイントは、そのライブラリの主要パブリッククラスです。

```csharp
// ファイル: src/libraries/System.Collections/src/System/Collections/Generic/Stack.cs:25-28
public class Stack<T> : IEnumerable<T>,
    System.Collections.ICollection,
    IReadOnlyCollection<T>
{
    // Stackクラスの実装
}
```

**ランタイムのエントリーポイント:**

CoreCLRのホスト起動は `src/coreclr/hosts/` ディレクトリに実装されています。

### 6.2 ビジネスロジック

ライブラリの主要なビジネスロジックは `src/` サブディレクトリに配置されています。

```csharp
// ファイル: src/libraries/System.Collections/src/System/Collections/Generic/Stack.cs:219-242
public T Pop()
{
    int size = _size - 1;
    T[] array = _array;

    // バウンドチェック最適化
    if ((uint)size >= (uint)array.Length)
    {
        ThrowForEmptyStack();
    }

    _version++;
    _size = size;
    T item = array[size];
    if (RuntimeHelpers.IsReferenceOrContainsReferences<T>())
    {
        array[size] = default!;     // メモリ解放を早める
    }
    return item;
}
```

### 6.3 データアクセス

.NET Runtimeでは、データアクセスは主にメモリ操作として実装されています。

**配列操作の例:**
```csharp
// ファイル: src/libraries/System.Collections/src/System/Collections/Generic/Stack.cs:313-328
private void Grow(int capacity)
{
    Debug.Assert(_array.Length < capacity);

    int newcapacity = _array.Length == 0 ? DefaultCapacity : 2 * _array.Length;

    // 最大容量チェック
    if ((uint)newcapacity > Array.MaxLength) newcapacity = Array.MaxLength;

    if (newcapacity < capacity) newcapacity = capacity;

    Array.Resize(ref _array, newcapacity);
}
```

### 6.4 ユーティリティ/共通機能

**ThrowHelper パターン:**

例外スローを専用メソッドに委譲することで、インライン化を改善します。

```csharp
// ファイル: src/libraries/System.Private.CoreLib/src/System/Collections/Generic/List.cs:48-50
if (capacity < 0)
    ThrowHelper.ThrowArgumentOutOfRangeException(ExceptionArgument.capacity, ExceptionResource.ArgumentOutOfRange_NeedNonNegNum);
```

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| TryGet パターン | 失敗時にfalseを返す | 高 | `Stack.cs`, `Dictionary.cs` |
| Enumerator パターン | コレクションの列挙 | 高 | 全コレクションクラス |
| バウンドチェック最適化 | uint キャストによる最適化 | 高 | `Stack.cs`, `List.cs` |
| ThrowHelper | 例外スローの分離 | 高 | `ThrowHelper.cs` |
| CONTRACTL | 関数契約（C++） | 高 | `src/coreclr/vm/*.cpp` |

### 各パターンの詳細

#### パターン1: TryGet パターン

**目的:** 操作が失敗する可能性がある場合に、例外をスローせずにboolで結果を返す

**実装例:**
```csharp
// ファイル: src/libraries/System.Collections/src/System/Collections/Generic/Stack.cs:205-217
public bool TryPeek([MaybeNullWhen(false)] out T result)
{
    int size = _size - 1;
    T[] array = _array;

    if ((uint)size >= (uint)array.Length)
    {
        result = default!;
        return false;
    }
    result = array[size];
    return true;
}
```

**解説:** `[MaybeNullWhen(false)]` 属性により、戻り値がfalseの場合に`result`がnullになり得ることをコンパイラに伝えます。

#### パターン2: バウンドチェック最適化

**目的:** 配列アクセス時のバウンドチェックを最適化する

**実装例:**
```csharp
// ファイル: src/libraries/System.Collections/src/System/Collections/Generic/Stack.cs:226-232
// if (_size == 0) is equivalent to if (size == -1), and this case
// is covered with (uint)size, thus allowing bounds check elimination
if ((uint)size >= (uint)array.Length)
{
    ThrowForEmptyStack();
}
```

**解説:** `size`を`uint`にキャストすることで、負の値が大きな正の値として扱われ、単一の比較でゼロチェックとバウンドチェックを同時に行えます。

#### パターン3: Enumerator パターン

**目的:** コレクションの要素を順番に取得するためのイテレータを提供

**実装例:**
```csharp
// ファイル: src/libraries/System.Collections/src/System/Collections/Generic/Stack.cs:352-407
public struct Enumerator : IEnumerator<T>, System.Collections.IEnumerator
{
    private readonly Stack<T> _stack;
    private readonly int _version;
    private int _index;
    private T? _currentElement;

    internal Enumerator(Stack<T> stack)
    {
        _stack = stack;
        _version = stack._version;
        _index = stack.Count;
        _currentElement = default;
    }

    public bool MoveNext()
    {
        if (_version != _stack._version)
        {
            ThrowInvalidVersion();
        }
        // ...
    }

    public T Current => _currentElement!;
}
```

**解説:** `struct`として実装することでヒープアロケーションを回避。`_version`フィールドでコレクションの変更を検出します。

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定
2. 処理の流れを追跡（呼び出し関係を追う）
3. データの変換を確認
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: Stack.Push() の処理フロー

**概要:** 要素をスタックにプッシュする処理の流れ

**処理フロー:**
```
Push(item) → 容量チェック → [容量不足] → PushWithResize() → Grow() → Array.Resize()
                         → [容量十分] → 配列に追加
```

**詳細な追跡:**

1. **Push メソッド** (`src/libraries/System.Collections/src/System/Collections/Generic/Stack.cs:266-281`)
   ```csharp
   public void Push(T item)
   {
       int size = _size;
       T[] array = _array;

       if ((uint)size < (uint)array.Length)
       {
           array[size] = item;
           _version++;
           _size = size + 1;
       }
       else
       {
           PushWithResize(item);
       }
   }
   ```

2. **PushWithResize メソッド** (`src/libraries/System.Collections/src/System/Collections/Generic/Stack.cs:284-292`)
   ```csharp
   [MethodImpl(MethodImplOptions.NoInlining)]
   private void PushWithResize(T item)
   {
       Debug.Assert(_size == _array.Length);
       Grow(_size + 1);
       _array[_size] = item;
       _version++;
       _size++;
   }
   ```

3. **Grow メソッド** (`src/libraries/System.Collections/src/System/Collections/Generic/Stack.cs:313-328`)
   ```csharp
   private void Grow(int capacity)
   {
       Debug.Assert(_array.Length < capacity);

       int newcapacity = _array.Length == 0 ? DefaultCapacity : 2 * _array.Length;

       if ((uint)newcapacity > Array.MaxLength) newcapacity = Array.MaxLength;

       if (newcapacity < capacity) newcapacity = capacity;

       Array.Resize(ref _array, newcapacity);
   }
   ```

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイントを特定したか
- [ ] 呼び出し関係を把握したか
- [ ] データの変換ポイントを確認したか
- [ ] エラーハンドリングを確認したか
- [ ] 最終的な出力を確認したか

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. `README.md` - プロジェクト概要
2. `docs/workflow/README.md` - ビルド・テストワークフロー
3. `docs/project/glossary.md` - 用語集

#### 特定機能を理解したい場合
1. `src/libraries/{ライブラリ名}/README.md` - ライブラリ概要
2. `src/libraries/{ライブラリ名}/src/` - ソースコード
3. `src/libraries/{ライブラリ名}/tests/` - テストコード

#### 改修作業を行う場合
1. `CONTRIBUTING.md` - コントリビューションガイド
2. `docs/coding-guidelines/coding-style.md` - コーディングスタイル
3. 該当コンポーネントのドキュメント

#### ランタイム内部を理解したい場合
1. `docs/design/coreclr/botr/` - Book of the Runtime
2. `src/coreclr/` - CoreCLRソースコード
3. `src/mono/` - Monoソースコード

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| `README.md` | プロジェクト概要 | 最初に読む |
| `CONTRIBUTING.md` | コントリビューションガイド | PRを作成する前 |
| `docs/workflow/README.md` | ビルド・テスト手順 | 開発を始める前 |
| `docs/coding-guidelines/coding-style.md` | コーディング規約 | コードを書く前 |
| `docs/project/glossary.md` | 用語集 | 不明な用語に遭遇した時 |
| `docs/area-owners.md` | 担当者一覧 | 質問やレビュー依頼時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: CoreCLRとMonoの違いは何ですか？
A: CoreCLRはクラウド/デスクトップ向けの高性能ランタイム、Monoはモバイル/ブラウザ向けの軽量ランタイムです。両方とも同じマネージドライブラリ（BCL）を共有しますが、System.Private.CoreLibの実装は異なります。

#### Q: `ref/` と `src/` ディレクトリの違いは何ですか？
A: `ref/` には参照アセンブリ（APIシグネチャのみ）が含まれ、`src/` には実際の実装が含まれます。参照アセンブリはコンパイル時に使用され、実装はランタイムに使用されます。

#### Q: `_version` フィールドの目的は何ですか？
A: コレクションが変更されたときに更新されるバージョンカウンターです。Enumeratorはこの値をチェックして、列挙中にコレクションが変更された場合に例外をスローします。

#### Q: `ThrowHelper` を使う理由は何ですか？
A: 例外スローコードを別メソッドに分離することで、呼び出し元メソッドのインライン化が容易になり、パフォーマンスが向上します。

### デバッグのヒント

1. **コードナビゲーション**: Visual StudioやRiderの「定義へ移動」機能を活用
2. **テストを読む**: `tests/` ディレクトリのテストコードは使用例として参考になる
3. **ソースリンク**: https://source.dot.net で.NETのソースコードをオンラインで閲覧可能
4. **GitHub検索**: リポジトリ内検索で関連コードを見つける

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| AOT | Ahead-of-Time コンパイル。事前にネイティブコードにコンパイルする |
| BCL | Base Class Library。基本クラスライブラリ |
| CLR | Common Language Runtime。共通言語ランタイム |
| CoreLib | System.Private.CoreLib。最低レベルのマネージドライブラリ |
| GC | Garbage Collector。ガベージコレクタ |
| IL | Intermediate Language。中間言語 |
| JIT | Just-in-Time コンパイル。実行時にネイティブコードにコンパイルする |
| PAL | Platform Adaptation Layer。プラットフォーム抽象化層 |
| TFM | Target Framework Moniker。ターゲットフレームワーク識別子（例: net8.0） |

### B. ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `build.sh` / `build.cmd` | ビルドスクリプト | プロジェクトのビルドを実行 |
| `global.json` | SDKバージョン設定 | .NET SDKのバージョン指定 |
| `Directory.Build.props` | 共通ビルドプロパティ | プロジェクト全体の設定 |
| `NuGet.config` | NuGet設定 | パッケージソースの設定 |
| `.editorconfig` | エディタ設定 | コーディングスタイル設定 |

### C. 参考資料

- [.NET ドキュメント](https://learn.microsoft.com/dotnet)
- [.NET Runtime GitHub](https://github.com/dotnet/runtime)
- [.NET ソースブラウザ](https://source.dot.net)
- [Book of the Runtime (BOTR)](https://github.com/dotnet/runtime/tree/main/docs/design/coreclr/botr)
- [C# 言語仕様](https://learn.microsoft.com/dotnet/csharp/language-reference)
