# セキュリティ設計書

## 概要

本ドキュメントは、.NET Runtime プロジェクトにおけるセキュリティ設計を記述したものです。.NET Runtimeは、暗号化、SSL/TLS通信、認証、乱数生成などのセキュリティ機能を提供するプラットフォーム抽象化層（PAL: Platform Abstraction Layer）を通じて、クロスプラットフォームでのセキュリティ機能を実現しています。

## 認証設計

### 認証方式

本プロジェクトは、GSS-API（Generic Security Service Application Program Interface）を通じて、以下の認証方式をサポートしています。

| 認証方式 | 説明 | 実装ファイル |
| --- | --- | --- |
| SPNEGO (Negotiate) | Simple and Protected GSSAPI Negotiation Mechanism | `pal_gssapi.c/h` |
| Kerberos | ネットワーク認証プロトコル | `pal_gssapi.c/h` |
| NTLM | NT LAN Manager認証 | `pal_gssapi.c/h` |

#### GSS-API機能

- `NetSecurityNative_InitSecContext`: セキュリティコンテキストの初期化
- `NetSecurityNative_AcceptSecContext`: セキュリティコンテキストの受け入れ
- `NetSecurityNative_Wrap` / `NetSecurityNative_Unwrap`: メッセージの暗号化・復号化
- `NetSecurityNative_GetMic` / `NetSecurityNative_VerifyMic`: メッセージ完全性コードの生成・検証

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション有効期限 | OpenSSL設定に依存 | `SSL_CTX_set_timeout`で設定可能 |
| セッション固定化対策 | SSL_SESSION_dup使用 | `g_ssl_sess_cert_index`で証明書参照を管理 |
| セッションキャッシング | 設定可能 | `CryptoNative_SslCtxSetCaching`で制御 |

## 認可設計

### 権限体系

本プロジェクトはランタイムライブラリであるため、アプリケーションレベルの権限体系は上位層で実装されます。ただし、以下の機能を提供しています。

| 機能 | 説明 | 実装ファイル |
| --- | --- | --- |
| X.509証明書検証 | 証明書チェーンの検証 | `pal_x509.c/h` |
| 証明書目的チェック | 証明書の使用目的検証 | `CryptoNative_X509CheckPurpose` |
| GSS委任フラグ | 認証情報の委任制御 | `PAL_GSS_C_DELEG_FLAG` |

### アクセス制御

X.509証明書チェーン検証を通じて、以下のアクセス制御機能を提供しています。

- **証明書失効確認**: `CryptoNative_X509StoreSetRevocationFlag`で失効フラグを設定
  - `EndCertificateOnly`: エンドエンティティ証明書のみ検証
  - `EntireChain`: 証明書チェーン全体を検証
  - `ExcludeRoot`: ルート証明書を除外して検証
- **ホスト名検証**: `CryptoNative_CheckX509Hostname`でホスト名をチェック
- **IPアドレス検証**: `CryptoNative_CheckX509IpAddress`でIPアドレスをチェック
- **OCSP検証**: `CryptoNative_X509ChainVerifyOcsp`でOCSPレスポンスを検証

## 通信セキュリティ

| 項目 | 対策 | 実装詳細 |
| --- | --- | --- |
| SSL/TLS | OpenSSLベースの実装 | `pal_ssl.c/h`で提供 |
| プロトコルバージョン | TLS 1.0 / 1.1 / 1.2 / 1.3 サポート | `SslProtocols` enum で定義 |
| 暗号スイート設定 | アプリケーション設定可能 | `CryptoNative_SslCtxSetCiphers` |
| ALPN | アプリケーション層プロトコルネゴシエーション対応 | `CryptoNative_SslSetAlpnProtos` |
| OCSP Stapling | サーバー証明書ステータス確認 | `CryptoNative_SslStapleOcsp` |

### サポートされるTLSプロトコル

```c
typedef enum
{
    PAL_SSL_NONE = 0,
    PAL_SSL_SSL2 = 12,    // 非推奨
    PAL_SSL_SSL3 = 48,    // 非推奨
    PAL_SSL_TLS = 192,    // TLS 1.0
    PAL_SSL_TLS11 = 768,  // TLS 1.1
    PAL_SSL_TLS12 = 3072, // TLS 1.2
    PAL_SSL_TLS13 = 12288,// TLS 1.3
} SslProtocols;
```

### 暗号化ポリシー

```c
typedef enum
{
    RequireEncryption = 0,  // 暗号化必須
    AllowNoEncryption,      // 暗号化なしを許可
    NoEncryption            // 暗号化なし
} EncryptionPolicy;
```

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 | 実装詳細 |
| --- | --- | --- |
| 通信 | TLS 1.0-1.3 (OpenSSL) | `pal_ssl.c/h` |
| 共通鍵暗号 | AES-128/192/256 (ECB, CBC, GCM, CCM, CFB) | `pal_evp_cipher.c` |
| 共通鍵暗号 | 3DES (ECB, CBC, CFB) | `pal_evp_cipher.c` |
| 共通鍵暗号 | ChaCha20-Poly1305 | `pal_evp_cipher.c` |
| 公開鍵暗号 | RSA, DSA, ECC (ECDSA, ECDH) | `pal_evp_pkey.h` |
| 公開鍵暗号 | ML-KEM, SLH-DSA, ML-DSA (PQC) | `pal_evp_pkey.h` |

#### サポートされる暗号アルゴリズム

| 種別 | アルゴリズム | 関数 |
| --- | --- | --- |
| AES-128 | ECB, CBC, GCM, CFB128, CFB8, CCM | `CryptoNative_EvpAes128*` |
| AES-192 | ECB, CBC, GCM, CFB128, CFB8, CCM | `CryptoNative_EvpAes192*` |
| AES-256 | ECB, CBC, GCM, CFB128, CFB8, CCM | `CryptoNative_EvpAes256*` |
| DES | ECB, CBC, CFB8 | `CryptoNative_EvpDes*` |
| 3DES | ECB, CBC, CFB8, CFB64 | `CryptoNative_EvpDes3*` |
| RC2 | ECB, CBC | `CryptoNative_EvpRC2*` |
| ChaCha20-Poly1305 | AEAD | `CryptoNative_EvpChaCha20Poly1305` |

### ハッシュアルゴリズム

| アルゴリズム | 関数 | 用途 |
| --- | --- | --- |
| MD5 | `CryptoNative_EvpMd5` | レガシー互換性 |
| SHA-1 | `CryptoNative_EvpSha1` | レガシー互換性 |
| SHA-256 | `CryptoNative_EvpSha256` | 推奨 |
| SHA-384 | `CryptoNative_EvpSha384` | 推奨 |
| SHA-512 | `CryptoNative_EvpSha512` | 推奨 |
| SHA3-256/384/512 | `CryptoNative_EvpSha3_*` | 次世代 |
| SHAKE128/256 | `CryptoNative_EvpShake*` | 可変長出力 |

### 鍵導出関数

| 関数 | 説明 |
| --- | --- |
| PBKDF2 | `CryptoNative_Pbkdf2` - パスワードベース鍵導出 |
| HMAC | `CryptoNative_HmacOneShot` - ハッシュベースメッセージ認証コード |

### 機密情報管理

乱数生成機能を通じて、安全な鍵生成をサポートしています。

| プラットフォーム | 実装 | 関数 |
| --- | --- | --- |
| Linux/Unix | OpenSSL RAND | `CryptoNative_GetRandomBytes` |
| macOS/iOS | CommonCrypto CCRandomGenerateBytes | `AppleCryptoNative_GetRandomBytes` |
| 汎用 | minipal | `SystemNative_GetCryptographicallySecureRandomBytes` |

#### 乱数生成の種別

```c
// 暗号学的に安全な乱数
int32_t SystemNative_GetCryptographicallySecureRandomBytes(uint8_t* buffer, int32_t bufferLength);

// 暗号学的に安全でない乱数（パフォーマンス用途）
void SystemNative_GetNonCryptographicallySecureRandomBytes(uint8_t* buffer, int32_t bufferLength);
```

## 入出力対策

| 脅威 | 対策 | 備考 |
| --- | --- | --- |
| バッファオーバーフロー | 境界チェック、安全なメモリ操作 | assert文による入力検証 |
| メモリリーク | コンテキスト適切な解放 | `*_Destroy`関数の一貫した提供 |
| エラー情報漏洩 | エラーキュークリア | `ERR_clear_error()`の使用 |
| Null参照 | Null安全な実装 | 入力パラメータの検証 |

### 入力検証例

```c
int32_t CryptoNative_HmacUpdate(HMAC_CTX* ctx, const uint8_t* data, int32_t len)
{
    assert(ctx != NULL);
    assert(data != NULL || len == 0);
    assert(len >= 0);

    ERR_clear_error();

    if (len < 0)
    {
        return 0;
    }
    // ...
}
```

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| TLSキーログ | セッションキー情報（デバッグ用） | 設定による |

### TLSキーログ機能

```c
// TLSセッションキーをログに記録するコールバック設定
typedef void (*SslCtxSetKeylogCallback)(const SSL* ssl, const char *line);
PALEXPORT void CryptoNative_SslCtxSetKeylogCallback(SSL_CTX* ctx, SslCtxSetKeylogCallback callback);
```

**注意**: この機能はデバッグ目的であり、本番環境での使用は推奨されません。

## X.509証明書エラーコード

以下のエラーコードが証明書検証時に返される可能性があります。

| エラーコード | 値 | 説明 |
| --- | --- | --- |
| PAL_X509_V_OK | 0 | 検証成功 |
| PAL_X509_V_ERR_CERT_SIGNATURE_FAILURE | 7 | 署名検証失敗 |
| PAL_X509_V_ERR_CERT_NOT_YET_VALID | 9 | 証明書がまだ有効でない |
| PAL_X509_V_ERR_CERT_HAS_EXPIRED | 10 | 証明書が期限切れ |
| PAL_X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT | 18 | 自己署名証明書 |
| PAL_X509_V_ERR_CERT_REVOKED | 23 | 証明書が失効済み |
| PAL_X509_V_ERR_HOSTNAME_MISMATCH | 62 | ホスト名不一致 |

## プラットフォーム別実装

### OpenSSL (Linux/Unix)

- `System.Security.Cryptography.Native`: OpenSSLラッパー
- バージョン1.0.x、1.1.x、3.xをサポート

### Apple Security Framework (macOS/iOS)

- `System.Security.Cryptography.Native.Apple`: CommonCryptoラッパー
- Security.frameworkとの統合

### Android

- `System.Security.Cryptography.Native.Android`: Android暗号化APIラッパー

## 備考

### ポスト量子暗号（PQC）対応

`PalPKeyFamilyId` enumに以下のPQCアルゴリズムファミリーが定義されています。

- `PalPKeyFamilyId_MLKem`: ML-KEM（旧CRYSTALS-Kyber）
- `PalPKeyFamilyId_SlhDsa`: SLH-DSA（旧SPHINCS+）
- `PalPKeyFamilyId_MLDsa`: ML-DSA（旧CRYSTALS-Dilithium）

これらは将来の量子コンピュータへの耐性を提供することを目的としています。

### セキュリティ上の注意事項

1. **SSL 2.0/3.0は非推奨**: セキュリティ上の脆弱性が知られているため使用を避けてください
2. **MD5/SHA-1は非推奨**: 衝突攻撃への脆弱性があるため、新規開発ではSHA-256以上を使用してください
3. **DES/RC2は非推奨**: 鍵長が短いため、AESの使用を推奨します
