# 機能設計書 10-System.Text.Json

## 概要

本ドキュメントは、.NET runtimeプロジェクトにおけるSystem.Text.Jsonライブラリの機能設計について記述する。System.Text.JsonはJSONのシリアライズ/デシリアライズを高パフォーマンスで提供する標準ライブラリである。

### 本機能の処理概要

**業務上の目的・背景**：System.Text.Jsonは、.NETオブジェクトとJSON形式データの相互変換を行う。UTF-8ネイティブ実装により、文字列変換オーバーヘッドを削減し、高いパフォーマンスを実現する。Web API、設定ファイル読み込み、データ交換など幅広いシナリオで使用される。

**機能の利用シーン**：
- RESTful APIでのJSONリクエスト/レスポンス処理
- 設定ファイル（appsettings.json）の読み込み
- NoSQLデータベースとのデータ交換
- マイクロサービス間のデータ通信

**主要な処理内容**：
1. JsonSerializer：オブジェクト⇔JSON変換の中核クラス
2. JsonSerializerOptions：シリアライズ動作の設定
3. JsonConverter<T>：カスタム型変換ロジック
4. Utf8JsonReader/Utf8JsonWriter：低レベルJSONパース/生成
5. JsonDocument/JsonElement：JSONドキュメントのDOM操作
6. JsonNode：変更可能なJSON DOM
7. Source Generator：AOT互換のコンパイル時コード生成

**関連システム・外部連携**：
- ASP.NET Core（HTTPリクエスト/レスポンス処理）
- System.IO.Pipelines（ストリーミング処理）
- System.Text.Encodings.Web（エンコーディング）

**権限による制御**：
- 特になし（汎用ライブラリ）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 直接関連する画面なし |

## 機能種別

データ変換 / シリアライゼーション

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| json | string/ReadOnlySpan<char>/ReadOnlySpan<byte> | Yes | JSON文字列 | 有効なJSON形式 |
| value | T | Yes | シリアライズ対象 | null許容 |
| options | JsonSerializerOptions | No | 動作オプション | null許容 |
| jsonTypeInfo | JsonTypeInfo<T> | No | 型メタデータ | Source Generator使用時 |

### 入力データソース

- 文字列（string）
- UTF-8バイト配列（ReadOnlySpan<byte>）
- ストリーム（Stream）
- Utf8JsonReader

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| オブジェクト | T | デシリアライズされたオブジェクト |
| JSON文字列 | string | シリアライズされたJSON |
| UTF-8バイト | byte[] | UTF-8エンコードされたJSON |

### 出力先

- 文字列返却
- ストリーム書き込み
- Utf8JsonWriter

## 処理フロー

### 処理シーケンス

```
1. Serialize呼び出し
   └─ オプション解決（デフォルト/指定）
2. JsonTypeInfo取得
   └─ リフレクション or Source Generator
3. JsonConverter選択
   └─ 型に対応するコンバーター決定
4. Utf8JsonWriter生成
   └─ バッファ確保
5. 値書き込み
   └─ コンバーターによるJSON生成
6. 結果返却
   └─ string/byte[]/Stream
```

### フローチャート

```mermaid
flowchart TD
    A[Serialize呼び出し] --> B[JsonSerializerOptions解決]
    B --> C{TypeInfoResolver?}
    C -->|リフレクション| D[DefaultJsonTypeInfoResolver]
    C -->|Source Generator| E[JsonSerializerContext]
    D --> F[JsonTypeInfo取得]
    E --> F
    F --> G[JsonConverter選択]
    G --> H[Utf8JsonWriter生成]
    H --> I[OnTryWrite実行]
    I --> J[UTF-8バイト出力]
    J --> K{出力形式}
    K -->|string| L[UTF-8→string変換]
    K -->|byte[]| M[直接返却]
    K -->|Stream| N[ストリーム書き込み]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | UTF-8ネイティブ | 内部処理はUTF-8で実行 | 全操作 |
| BR-02 | オプション不変 | 使用開始後はオプション変更不可 | MakeReadOnly後 |
| BR-03 | 最大深度制限 | デフォルト64レベル | ネスト構造 |
| BR-04 | Source Generator優先 | AOT環境ではリフレクション不可 | NativeAOT |

### 計算ロジック

- デフォルトバッファサイズ：16KB（BufferSizeDefault = 16 * 1024）
- デフォルト最大深度：64（DefaultMaxDepth = JsonReaderOptions.DefaultMaxDepth）
- インデントサイズ：2（JsonConstants.DefaultIndentSize）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | シリアライゼーションライブラリはデータベースを操作しない |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| JsonException | JSONパースエラー | 不正なJSON形式 | JSON形式確認 |
| NotSupportedException | 未対応型 | JsonConverter未定義 | カスタムコンバーター追加 |
| ArgumentNullException | null引数 | json/typeがnull | null検証 |
| InvalidOperationException | 不正操作 | ReadOnly後のオプション変更 | 事前設定 |

### リトライ仕様

- 該当なし（同期的変換操作）

## トランザクション仕様

シリアライゼーションライブラリはトランザクション管理を行わない。

## パフォーマンス要件

- UTF-8ネイティブ処理による高速変換
- バッファプール再利用によるGC削減
- Source Generator使用時のゼロリフレクション

## セキュリティ考慮事項

- 最大深度制限による再帰攻撃防止
- 型情報漏洩防止（ポリモーフィズム制御）
- JSON Injection対策（エンコーディング）

## 備考

- .NET Core 3.0で導入
- Newtonsoft.Json代替として設計
- .NET 6以降でSource Generator本格サポート
- .NET 8でStrict/Webプリセット追加

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: オプション設定を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | JsonSerializerOptions.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Serialization/JsonSerializerOptions.cs` | シリアライズ設定 |

**読解のコツ**:
- **JsonSerializerOptions.cs 18-22行目**: クラス定義 - `JsonSerializerOptions`はデバッガ表示属性付き
- **JsonSerializerOptions.cs 24行目**: `BufferSizeDefault = 16 * 1024` - デフォルトバッファサイズ16KB
- **JsonSerializerOptions.cs 28行目**: `DefaultMaxDepth = JsonReaderOptions.DefaultMaxDepth` - 最大深度64
- **JsonSerializerOptions.cs 38-43行目**: `Default`プロパティ - シングルトンインスタンス
- **JsonSerializerOptions.cs 53-58行目**: `Web`プロパティ - Web用プリセット
- **JsonSerializerOptions.cs 68-73行目**: `Strict`プロパティ - 厳格モードプリセット
- **JsonSerializerOptions.cs 77-88行目**: 各種ポリシーフィールド定義
- **JsonSerializerOptions.cs 90-105行目**: オプションフィールド（バッファサイズ、最大深度、インデント等）

#### Step 2: シリアライザAPIを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | JsonSerializer.Helpers.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Serialization/JsonSerializer.Helpers.cs` | ヘルパーメソッド |
| 2-2 | JsonSerializer.Write.String.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Serialization/JsonSerializer.Write.String.cs` | 文字列シリアライズ |
| 2-3 | JsonSerializer.Read.String.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Serialization/JsonSerializer.Read.String.cs` | 文字列デシリアライズ |

**主要処理フロー**:
- **JsonSerializer.Helpers.cs 13行目**: `public static partial class JsonSerializer` - パーシャルクラス宣言
- **JsonSerializer.Helpers.cs 15行目**: `SerializationUnreferencedCodeMessage` - トリミング警告メッセージ
- **JsonSerializer.Helpers.cs 16行目**: `SerializationRequiresDynamicCodeMessage` - AOT警告メッセージ
- **JsonSerializer.Helpers.cs 27行目**: `IsReflectionEnabledByDefault` - リフレクション有効フラグ
- **JsonSerializer.Write.String.cs 30-34行目**: `Serialize<TValue>` - ジェネリックシリアライズ
- **JsonSerializer.Read.String.cs 47-53行目**: `Deserialize<TValue>` - ジェネリックデシリアライズ

#### Step 3: コンバーターシステムを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | JsonConverter.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Serialization/JsonConverter.cs` | 基底コンバーター |
| 3-2 | JsonConverterOfT.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Serialization/JsonConverterOfT.cs` | ジェネリックコンバーター |

**主要処理フロー**:
- **JsonConverter.cs 16行目**: `public abstract partial class JsonConverter` - 抽象基底クラス
- **JsonConverter.cs 31行目**: `public abstract Type? Type { get; }` - 変換対象型
- **JsonConverter.cs 38行目**: `public abstract bool CanConvert(Type typeToConvert)` - 変換可否判定
- **JsonConverter.cs 40-49行目**: `ConverterStrategy` - 変換戦略プロパティ
- **JsonConverterOfT.cs 15行目**: `public abstract partial class JsonConverter<T>` - ジェネリック基底
- **JsonConverterOfT.cs 48-51行目**: `CanConvert` - デフォルト実装（typeof(T)と比較）
- **JsonConverterOfT.cs 67-74行目**: `HandleNull` - null処理制御

#### Step 4: Source Generatorコンテキストを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | JsonSerializerContext.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Serialization/JsonSerializerContext.cs` | Source Generatorコンテキスト |

**主要処理フロー**:
- **JsonSerializerContext.cs 12行目**: `public abstract partial class JsonSerializerContext` - 抽象コンテキスト
- **JsonSerializerContext.cs 14行目**: `private JsonSerializerOptions? _options` - オプション保持
- **JsonSerializerContext.cs 23-38行目**: `Options`プロパティ - 遅延初期化パターン
- **JsonSerializerContext.cs 40-46行目**: `AssociateWithOptions` - オプションとの関連付け
- **JsonSerializerContext.cs 52-79行目**: `IsCompatibleWithOptions` - 互換性チェック
- **JsonSerializerContext.cs 84行目**: `GeneratedSerializerOptions` - 生成時オプション

### プログラム呼び出し階層図

```
JsonSerializer
    │
    ├─ Serialize<T>(value, options)
    │      ├─ GetTypeInfo<T>(options)
    │      │      └─ options.GetTypeInfoForRootType()
    │      └─ WriteString(value, jsonTypeInfo)
    │             ├─ Utf8JsonWriter生成
    │             └─ JsonConverter<T>.Write()
    │
    ├─ Deserialize<T>(json, options)
    │      ├─ GetTypeInfo<T>(options)
    │      └─ ReadFromSpan(json, jsonTypeInfo)
    │             ├─ Utf8JsonReader生成
    │             └─ JsonConverter<T>.Read()
    │
    └─ SerializeAsync / DeserializeAsync
           └─ ストリーミング処理

JsonSerializerOptions
    │
    ├─ Default / Web / Strict (シングルトン)
    ├─ TypeInfoResolver → JsonTypeInfo取得
    ├─ Converters → カスタムコンバーター
    └─ MakeReadOnly() → 不変化

JsonConverter<T>
    │
    ├─ Read(ref Utf8JsonReader, Type, JsonSerializerOptions)
    ├─ Write(Utf8JsonWriter, T, JsonSerializerOptions)
    └─ CanConvert(Type)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

Serialize:
Object ──────────────▶ JsonTypeInfo取得 ──────▶ 型メタデータ
     │                       │
     │                       ▼
     │               JsonConverter選択
     │                       │
     ▼                       ▼
型情報 ──────────────▶ Utf8JsonWriter
     │                       │
     │                       ▼
     │               Write処理（再帰）
     │                       │
     ▼                       ▼
UTF-8バイト ◀──────── バッファ出力 ──────▶ string変換

Deserialize:
JSON string ─────────▶ UTF-8変換 ──────────▶ ReadOnlySpan<byte>
     │                       │
     │                       ▼
     │               Utf8JsonReader生成
     │                       │
     ▼                       ▼
JsonTypeInfo ────────▶ JsonConverter選択
     │                       │
     │                       ▼
     │               Read処理（再帰）
     │                       │
     ▼                       ▼
Object ◀─────────────── オブジェクト構築
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| JsonSerializerOptions.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Serialization/JsonSerializerOptions.cs` | ソース | シリアライズ設定 |
| JsonSerializer.Helpers.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Serialization/JsonSerializer.Helpers.cs` | ソース | ヘルパーメソッド |
| JsonSerializer.Write.String.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Serialization/JsonSerializer.Write.String.cs` | ソース | 文字列シリアライズ |
| JsonSerializer.Read.String.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Serialization/JsonSerializer.Read.String.cs` | ソース | 文字列デシリアライズ |
| JsonSerializer.Write.Stream.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Serialization/JsonSerializer.Write.Stream.cs` | ソース | ストリームシリアライズ |
| JsonSerializer.Read.Stream.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Serialization/JsonSerializer.Read.Stream.cs` | ソース | ストリームデシリアライズ |
| JsonConverter.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Serialization/JsonConverter.cs` | ソース | 基底コンバーター |
| JsonConverterOfT.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Serialization/JsonConverterOfT.cs` | ソース | ジェネリックコンバーター |
| JsonConverterFactory.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Serialization/JsonConverterFactory.cs` | ソース | コンバーターファクトリ |
| JsonSerializerContext.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Serialization/JsonSerializerContext.cs` | ソース | Source Generatorコンテキスト |
| Utf8JsonReader.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Reader/Utf8JsonReader.cs` | ソース | UTF-8 JSONリーダー |
| Utf8JsonWriter.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Writer/Utf8JsonWriter.cs` | ソース | UTF-8 JSONライター |
| JsonDocument.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Document/JsonDocument.cs` | ソース | JSONドキュメント |
| JsonNode.cs | `src/libraries/System.Text.Json/src/System/Text/Json/Nodes/JsonNode.cs` | ソース | 変更可能DOM |
