# 機能設計書: System.Security.Cryptography（暗号化）

## 1. 機能概要

### 1.1 機能名
System.Security.Cryptography（暗号化）

### 1.2 機能ID
FUNC-016

### 1.3 機能説明
.NET ランタイムにおける暗号化機能を提供するライブラリ。対称暗号化（AES, DES等）、非対称暗号化（RSA, ECDsa等）、ハッシュアルゴリズム（SHA, MD5等）、鍵交換、デジタル署名、乱数生成、X.509証明書操作など、包括的な暗号化サービスを提供する。FIPS準拠のアルゴリズムをサポートし、セキュアなアプリケーション開発を可能にする。

### 1.4 関連画面
- なし（基盤ライブラリのため直接的な関連画面は存在しない）

## 2. 機能要件

### 2.1 入力仕様

| 項目名 | データ型 | 必須 | 説明 |
|--------|----------|------|------|
| plaintext | byte[]/ReadOnlySpan<byte> | Yes | 暗号化対象の平文 |
| ciphertext | byte[]/ReadOnlySpan<byte> | Yes | 復号対象の暗号文 |
| key | byte[] | Yes | 暗号化キー |
| iv | byte[] | No | 初期化ベクトル |
| data | byte[] | Yes | ハッシュ対象データ |

### 2.2 出力仕様

| 項目名 | データ型 | 説明 |
|--------|----------|------|
| ciphertext | byte[] | 暗号化結果 |
| plaintext | byte[] | 復号結果 |
| hash | byte[] | ハッシュ値 |
| signature | byte[] | デジタル署名 |
| bool | bool | 署名検証結果 |

### 2.3 処理フロー（AES暗号化）

```
平文データ
    │
    ▼
┌─────────────────────┐
│ Aes.Create()        │
│ アルゴリズムインスタンス│
└─────────────────────┘
    │
    ▼
┌─────────────────────┐
│ Key, IV 設定         │
│ Mode, Padding 設定   │
└─────────────────────┘
    │
    ▼
┌─────────────────────┐
│ CreateEncryptor()   │
│ 暗号化トランスフォーム │
└─────────────────────┘
    │
    ▼
┌─────────────────────┐
│ TransformFinalBlock()│
│ データ変換           │
└─────────────────────┘
    │
    ▼
暗号文データ
```

## 3. 詳細設計

### 3.1 クラス構成

#### 3.1.1 対称暗号化

| クラス名 | 責務 |
|----------|------|
| SymmetricAlgorithm | 対称暗号化の抽象基底クラス |
| Aes | AES 暗号化（128/192/256ビット） |
| AesGcm | AES-GCM 認証付き暗号化 |
| AesCcm | AES-CCM 認証付き暗号化 |
| ChaCha20Poly1305 | ChaCha20-Poly1305 認証付き暗号化 |
| TripleDES | Triple DES 暗号化（レガシー） |

#### 3.1.2 非対称暗号化

| クラス名 | 責務 |
|----------|------|
| AsymmetricAlgorithm | 非対称暗号化の抽象基底クラス |
| RSA | RSA 暗号化・署名 |
| ECDsa | 楕円曲線デジタル署名 |
| ECDiffieHellman | 楕円曲線鍵交換 |
| DSA | DSA 署名（レガシー） |

#### 3.1.3 ハッシュアルゴリズム

| クラス名 | 責務 |
|----------|------|
| HashAlgorithm | ハッシュの抽象基底クラス |
| SHA256 | SHA-256 ハッシュ |
| SHA384 | SHA-384 ハッシュ |
| SHA512 | SHA-512 ハッシュ |
| SHA3_256 | SHA3-256 ハッシュ |
| MD5 | MD5 ハッシュ（レガシー） |
| HMACSHA256 | HMAC-SHA256 |

### 3.2 Aes クラス（Aes.cs）

#### 3.2.1 コンストラクタと初期化

```csharp
// Aes.cs: 14-23行目
protected Aes()
{
    LegalBlockSizesValue = s_legalBlockSizes.CloneKeySizesArray();
    LegalKeySizesValue = s_legalKeySizes.CloneKeySizesArray();

    BlockSizeValue = 128;
    FeedbackSizeValue = 8;
    KeySizeValue = 256;
    ModeValue = CipherMode.CBC;
}

// ファクトリメソッド（Aes.cs: 25-29行目）
[UnsupportedOSPlatform("browser")]
public static new Aes Create()
{
    return new AesImplementation();
}
```

#### 3.2.2 サポートされるキーサイズ

```csharp
// Aes.cs: 514-515行目
private static readonly KeySizes[] s_legalBlockSizes = { new KeySizes(128, 128, 0) };
private static readonly KeySizes[] s_legalKeySizes = { new KeySizes(128, 256, 64) };
```

- ブロックサイズ: 128ビット固定
- キーサイズ: 128, 192, 256ビット

### 3.3 AES Key Wrap（Aes.cs: 59-512行目）

#### 3.3.1 RFC 5649 Key Wrap with Padding

```csharp
// パディング長計算（Aes.cs: 59-77行目）
public static int GetKeyWrapPaddedLength(int plaintextLengthInBytes)
{
    ArgumentOutOfRangeException.ThrowIfNegativeOrZero(plaintextLengthInBytes);

    int blocks = (plaintextLengthInBytes + 7) / 8;
    return (blocks + 1) * 8;
}

// 暗号化（Aes.cs: 86-149行目）
public byte[] EncryptKeyWrapPadded(byte[] plaintext);
public byte[] EncryptKeyWrapPadded(ReadOnlySpan<byte> plaintext);
public void EncryptKeyWrapPadded(ReadOnlySpan<byte> plaintext, Span<byte> destination);

// 復号（Aes.cs: 163-227行目）
public byte[] DecryptKeyWrapPadded(byte[] ciphertext);
public byte[] DecryptKeyWrapPadded(ReadOnlySpan<byte> ciphertext);
public int DecryptKeyWrapPadded(ReadOnlySpan<byte> ciphertext, Span<byte> destination);
```

### 3.4 CipherMode 列挙型

| モード | 説明 | 推奨度 |
|--------|------|--------|
| CBC | Cipher Block Chaining | 推奨（IVが必要） |
| ECB | Electronic Code Book | 非推奨 |
| CFB | Cipher Feedback | 条件付き |
| CTS | Cipher Text Stealing | 特殊用途 |

### 3.5 PaddingMode 列挙型

| モード | 説明 |
|--------|------|
| None | パディングなし |
| PKCS7 | PKCS#7 パディング（推奨） |
| Zeros | ゼロパディング |
| ANSIX923 | ANSI X.923 パディング |
| ISO10126 | ISO 10126 パディング |

### 3.6 RSA クラス

#### 3.6.1 主要メソッド

```csharp
// 暗号化
public byte[] Encrypt(byte[] data, RSAEncryptionPadding padding);
public byte[] Decrypt(byte[] data, RSAEncryptionPadding padding);

// 署名
public byte[] SignData(byte[] data, HashAlgorithmName hashAlgorithm, RSASignaturePadding padding);
public bool VerifyData(byte[] data, byte[] signature, HashAlgorithmName hashAlgorithm, RSASignaturePadding padding);
public byte[] SignHash(byte[] hash, HashAlgorithmName hashAlgorithm, RSASignaturePadding padding);
public bool VerifyHash(byte[] hash, byte[] signature, HashAlgorithmName hashAlgorithm, RSASignaturePadding padding);
```

#### 3.6.2 パディングモード

| 暗号化パディング | 説明 |
|------------------|------|
| Pkcs1 | PKCS#1 v1.5 |
| OaepSHA256 | OAEP with SHA-256 |
| OaepSHA384 | OAEP with SHA-384 |
| OaepSHA512 | OAEP with SHA-512 |

| 署名パディング | 説明 |
|----------------|------|
| Pkcs1 | PKCS#1 v1.5 |
| Pss | PSS (Probabilistic Signature Scheme) |

### 3.7 ハッシュアルゴリズム

```csharp
// ワンショット静的メソッド
byte[] hash = SHA256.HashData(data);

// インスタンスメソッド
using var sha256 = SHA256.Create();
byte[] hash = sha256.ComputeHash(data);

// インクリメンタルハッシュ
using var incremental = IncrementalHash.CreateHash(HashAlgorithmName.SHA256);
incremental.AppendData(chunk1);
incremental.AppendData(chunk2);
byte[] hash = incremental.GetHashAndReset();
```

### 3.8 乱数生成

```csharp
// 暗号論的乱数
byte[] randomBytes = RandomNumberGenerator.GetBytes(32);
RandomNumberGenerator.Fill(buffer);

// 整数乱数
int randomInt = RandomNumberGenerator.GetInt32(min, max);
```

## 4. エラー処理

### 4.1 例外一覧

| 例外 | 発生条件 |
|------|----------|
| CryptographicException | 暗号操作エラー |
| ArgumentNullException | 必須引数が null |
| ArgumentOutOfRangeException | 引数が範囲外 |
| ArgumentException | 引数が不正 |
| PlatformNotSupportedException | プラットフォーム未サポート |

### 4.2 CryptographicException の発生ケース
- 不正なキーサイズ
- 不正なブロックサイズ
- 復号エラー（パディング不正など）
- 署名検証失敗
- Key Wrap 復号失敗

## 5. セキュリティ考慮事項

### 5.1 アルゴリズム選択
- **推奨**: AES-256, RSA-2048以上, SHA-256以上, ECDSA P-256以上
- **非推奨**: DES, 3DES, MD5, SHA-1, RSA-1024

### 5.2 鍵管理
- 鍵の安全な生成（RandomNumberGenerator使用）
- 鍵の安全な保管（DPAPI, Azure Key Vault等）
- 使用後のメモリクリア（CryptographicOperations.ZeroMemory）

### 5.3 IV/ノンス管理
- CBC モードでは毎回異なる IV 使用
- AES-GCM では同一鍵でノンス再利用禁止
- IV は暗号文と共に保存可能（秘密不要）

## 6. コードリーディングガイド

### 6.1 推奨読解順序

1. **データ構造の理解**
   - `KeySizes.cs`: サポートされるキーサイズ
   - `CipherMode.cs`: 暗号化モード
   - `PaddingMode.cs`: パディングモード

2. **エントリーポイント**
   - `Aes.cs`: AES 暗号化
   - **14-23行目**: コンストラクタ、デフォルト設定
   - **25-29行目**: Create() ファクトリメソッド
   - **59-77行目**: GetKeyWrapPaddedLength
   - **409-448行目**: EncryptKeyWrapPaddedCore 実装

3. **内部実装**
   - `AesImplementation.cs`: AES の具体的実装
   - `BasicSymmetricCipher.cs`: 対称暗号の基本処理

4. **非対称暗号**
   - `RSA.cs`: RSA アルゴリズム
   - `ECDsa.cs`: 楕円曲線署名

### 6.2 プログラム呼び出し階層図

```
Aes.Create()
    │
    └── new AesImplementation()
            │
            └── AesImplementation : Aes
                    │
                    └── プラットフォーム固有実装
                            │
                            ├── Windows: BCrypt
                            └── Unix: OpenSSL

aes.EncryptCbc(plaintext, iv)
    │
    ├── CreateEncryptor(Key, iv)
    │       │
    │       └── ICryptoTransform 作成
    │
    └── TransformFinalBlock(plaintext)
            │
            ├── ブロック分割
            │
            ├── パディング適用
            │
            └── 暗号化変換
                    │
                    └── 暗号文返却
```

### 6.3 データフロー図

```
┌──────────────────┐
│ 平文              │
│ byte[]           │
└────────┬─────────┘
         │
         ▼
┌──────────────────┐
│ SymmetricAlgorithm│
│ Key, IV, Mode    │
└────────┬─────────┘
         │
         ▼
┌──────────────────┐
│ ICryptoTransform │
│ Encryptor        │
└────────┬─────────┘
         │
    ┌────┴────┐
    ▼         ▼
┌────────┐ ┌────────┐
│ブロック1│ │ブロック2│ ...
└────┬───┘ └────┬───┘
     │          │
     ▼          ▼
┌──────────────────┐
│ 暗号化処理        │
│ (CBC/GCM等)      │
└────────┬─────────┘
         │
         ▼
┌──────────────────┐
│ 暗号文            │
│ byte[]           │
└──────────────────┘
```

### 6.4 読解のコツ

#### 6.4.1 RFC 3394/5649 Key Wrap
Aes.cs の Key Wrap 実装（450-512行目）:
- `Rfc3394Wrap`: 基本的な Key Wrap
- `Rfc3394Unwrap`: 基本的な Key Unwrap
- AIV (Alternative IV) の検証: `0xA65959A6`

#### 6.4.2 CryptoPoolLease パターン
メモリプールを使用した効率的なバッファ管理:
```csharp
using (CryptoPoolLease lease = CryptoPoolLease.Rent(size))
{
    // lease.Span でバッファアクセス
}
```

#### 6.4.3 プラットフォーム抽象化
- Windows: BCrypt API
- Unix/Linux: OpenSSL
- macOS: CommonCrypto

### 6.5 関連ファイル一覧

| パス | 種別 | 役割 |
|------|------|------|
| `src/libraries/System.Security.Cryptography/src/System/Security/Cryptography/Aes.cs` | ソース | AES 暗号化 |
| `src/libraries/System.Security.Cryptography/src/System/Security/Cryptography/AesGcm.cs` | ソース | AES-GCM |
| `src/libraries/System.Security.Cryptography/src/System/Security/Cryptography/RSA.cs` | ソース | RSA 暗号化 |
| `src/libraries/System.Security.Cryptography/src/System/Security/Cryptography/ECDsa.cs` | ソース | ECDSA |
| `src/libraries/System.Security.Cryptography/src/System/Security/Cryptography/SHA256.cs` | ソース | SHA-256 |
| `src/libraries/System.Security.Cryptography/src/System/Security/Cryptography/HMACSHA256.cs` | ソース | HMAC-SHA256 |
| `src/libraries/System.Security.Cryptography/src/System/Security/Cryptography/RandomNumberGenerator.cs` | ソース | 乱数生成 |
| `src/libraries/System.Security.Cryptography/src/System/Security/Cryptography/SymmetricAlgorithm.cs` | ソース | 対称暗号基底 |
| `src/libraries/System.Security.Cryptography/src/System/Security/Cryptography/AsymmetricAlgorithm.cs` | ソース | 非対称暗号基底 |
| `src/libraries/System.Security.Cryptography/src/System/Security/Cryptography/HashAlgorithm.cs` | ソース | ハッシュ基底 |

## 7. 使用例

### 7.1 AES 暗号化

```csharp
using var aes = Aes.Create();
aes.Key = RandomNumberGenerator.GetBytes(32); // 256-bit key
aes.GenerateIV();

// 暗号化
byte[] encrypted = aes.EncryptCbc(plaintext, aes.IV, PaddingMode.PKCS7);

// 復号
byte[] decrypted = aes.DecryptCbc(encrypted, aes.IV, PaddingMode.PKCS7);
```

### 7.2 RSA 署名

```csharp
using var rsa = RSA.Create(2048);

// 署名
byte[] signature = rsa.SignData(data, HashAlgorithmName.SHA256, RSASignaturePadding.Pss);

// 検証
bool isValid = rsa.VerifyData(data, signature, HashAlgorithmName.SHA256, RSASignaturePadding.Pss);
```

### 7.3 ハッシュ計算

```csharp
// ワンショット
byte[] hash = SHA256.HashData(data);

// HMAC
byte[] key = RandomNumberGenerator.GetBytes(32);
byte[] mac = HMACSHA256.HashData(key, data);
```

### 7.4 AES-GCM 認証付き暗号化

```csharp
using var aesGcm = new AesGcm(key, tagSizeInBytes: 16);

byte[] nonce = RandomNumberGenerator.GetBytes(12);
byte[] ciphertext = new byte[plaintext.Length];
byte[] tag = new byte[16];

aesGcm.Encrypt(nonce, plaintext, ciphertext, tag);
aesGcm.Decrypt(nonce, ciphertext, tag, decrypted);
```

## 8. テスト観点

### 8.1 単体テスト観点
- 各アルゴリズムの暗号化・復号の往復
- 異なるキーサイズでの動作
- パディングモードの動作
- 不正入力時のエラー処理

### 8.2 相互運用テスト観点
- 他プラットフォームとの互換性
- 標準テストベクトルでの検証

### 8.3 性能テスト観点
- 大量データの処理速度
- 鍵生成速度
- メモリ使用量

## 9. 変更履歴

| バージョン | 日付 | 変更内容 |
|------------|------|----------|
| 1.0 | 2026-01-30 | 初版作成 |
