# 機能設計書 23-System.Numerics.Tensors

## 概要

本ドキュメントは、.NETランタイムのSystem.Numerics.Tensors名前空間が提供するテンソル機能について、その設計仕様と実装の詳細を記述する。テンソルは、機械学習や科学計算で使用される多次元配列を効率的に操作するための基盤を提供する。

### 本機能の処理概要

System.Numerics.Tensorsは、数値計算用のテンソル操作を提供するライブラリである。TensorPrimitives静的クラスによる高速なベクトル化演算と、Tensor<T>クラスによる多次元テンソルの管理を行う。

**業務上の目的・背景**：機械学習、ディープラーニング、科学計算などの分野では、大量の数値データを効率的に処理する必要がある。従来のループベースの処理は遅く、SIMDなどのハードウェア機能を活用するのが困難であった。System.Numerics.Tensorsは、.NETプラットフォーム上でハードウェアアクセラレーションを活用した高性能な数値計算を可能にする。

**機能の利用シーン**：
- 機械学習モデルの推論処理
- 画像処理・信号処理
- 科学シミュレーション
- 金融工学（リスク計算、オプション価格計算）
- ベクトル検索（埋め込みベクトルの類似度計算）
- 統計分析

**主要な処理内容**：
1. テンソル基本演算（加算、減算、乗算、除算）
2. 数学関数（三角関数、指数、対数等）
3. 集約演算（Sum、Product、Average、Min、Max）
4. ベクトル演算（Dot、Distance、CosineSimilarity）
5. 活性化関数（Sigmoid、SoftMax、Tanh）
6. テンソルの形状操作（Reshape、Slice、Transpose）

**関連システム・外部連携**：System.Numerics.Vectors、System.Runtime.Intrinsics（SIMD）、BLASライブラリ相当の機能を提供。

**権限による制御**：特別な権限制御は不要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はUIを持たないライブラリ機能である |

## 機能種別

数値計算 / ベクトル演算 / テンソル操作

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| x | ReadOnlySpan<T> | Yes | 入力テンソル/ベクトル | 空でないこと |
| y | ReadOnlySpan<T> | Depends | 二項演算の第二オペランド | xと同じ長さ |
| destination | Span<T> | Yes（出力時） | 結果格納先 | 必要な長さを確保 |
| lengths | ReadOnlySpan<nint> | Yes（Tensor作成時） | 各次元のサイズ | 正の整数 |
| strides | ReadOnlySpan<nint> | No | 各次元のストライド | 整合性チェック |

### 入力データソース

- プログラムコードからの配列/Span
- Tensor<T>インスタンス
- TensorSpan<T>/ReadOnlyTensorSpan<T>

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | T | スカラー結果（Dot、Sum等） |
| destination | Span<T> | 要素ごとの演算結果 |
| Tensor<T> | Tensor<T> | 新しいテンソルインスタンス |

### 出力先

- 呼び出し元への戻り値
- 引数で指定されたdestination Span

## 処理フロー

### 処理シーケンス

```
1. テンソル作成
   └─ Tensor.Create<T>(values, lengths) または new Tensor<T>(...)

2. 演算実行
   └─ TensorPrimitives.Xxx(source, destination)
   └─ または Tensor演算子（+, -, *, /等）

3. 結果取得
   └─ destinationから直接アクセス
   └─ または戻り値として取得
```

### フローチャート

```mermaid
flowchart TD
    A[入力データ準備] --> B{演算種別}
    B -->|要素演算| C[TensorPrimitives.Xxx]
    B -->|集約演算| D[TensorPrimitives.Sum/Dot等]
    B -->|形状操作| E[Slice/Reshape等]
    C --> F[SIMD最適化パス]
    D --> F
    F --> G{ハードウェア判定}
    G -->|AVX512| H[AVX512実装]
    G -->|AVX2| I[AVX2実装]
    G -->|SSE| J[SSE実装]
    G -->|Fallback| K[スカラー実装]
    H --> L[結果出力]
    I --> L
    J --> L
    K --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 長さ一致 | 二項演算では入力の長さが一致必須 | 二項演算時 |
| BR-02 | NaN伝播 | NaN入力は結果にNaNを伝播 | 浮動小数点演算時 |
| BR-03 | 自動ベクトル化 | 可能な場合SIMDを使用 | 対応プラットフォーム |
| BR-04 | インプレース禁止 | source != destinationを推奨 | 一部演算 |

### 計算ロジック

TensorPrimitives.Dot（ドット積）:
```
result = sum(x[i] * y[i]) for i in 0..length
```

TensorPrimitives.CosineSimilarity（コサイン類似度）:
```
result = Dot(x, y) / (Norm(x) * Norm(y))
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ArgumentException | 例外 | x.Length != y.Length | 同じ長さの入力を使用 |
| ArgumentException | 例外 | destination.Length < source.Length | 十分な長さのdestinationを確保 |
| ArgumentOutOfRangeException | 例外 | 無効なインデックス | 範囲内のインデックスを使用 |

### リトライ仕様

数値計算は即座に結果を返すため、リトライは不要。

## トランザクション仕様

本機能はトランザクションを使用しない。

## パフォーマンス要件

- SIMD（AVX512/AVX2/SSE/ARM NEON）による自動ベクトル化
- スタックアロケーションを活用したゼロアロケーション設計
- メモリレイアウトを考慮したキャッシュ効率的なアクセス
- ジェネリック数値型によるコード共有と最適化

## セキュリティ考慮事項

- 数値オーバーフローは未定義動作を引き起こす可能性
- 信頼できないソースからのテンソルサイズに注意（DoS攻撃対策）

## 備考

- .NET 8以降でTensor<T>クラスが利用可能
- .NET 7以前ではTensorPrimitives静的メソッドのみ
- ONNX Runtime、ML.NETとの相互運用に適している

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

テンソルの基本構造とインターフェースを理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | IReadOnlyTensor.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/IReadOnlyTensor.cs` | 読み取り専用テンソルインターフェース |
| 1-2 | ITensor_1.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/ITensor_1.cs` | 書き込み可能テンソルインターフェース |
| 1-3 | TensorShape.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/TensorShape.cs` | テンソル形状（Lengths、Strides）の管理 |
| 1-4 | Tensor_1.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/Tensor_1.cs` | Tensor<T>クラスの実装 |

**読解のコツ**: TensorShapeがテンソルの次元、長さ、ストライドを管理し、Tensor<T>がデータ配列と形状を組み合わせて多次元アクセスを提供する構造を理解する。

#### Step 2: Span型のテンソルを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | TensorSpan_1.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/TensorSpan_1.cs` | 書き込み可能Spanベーステンソル |
| 2-2 | ReadOnlyTensorSpan_1.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/ReadOnlyTensorSpan_1.cs` | 読み取り専用Spanベーステンソル |
| 2-3 | NIndex.cs | `src/libraries/System.Numerics.Tensors/src/System/Buffers/NIndex.cs` | ネイティブサイズインデックス |
| 2-4 | NRange.cs | `src/libraries/System.Numerics.Tensors/src/System/Buffers/NRange.cs` | ネイティブサイズ範囲 |

**主要処理フロー（Tensor_1.cs）**:
- **23-25行目**: 内部フィールド（TensorShape、配列、開始位置）
- **81-90行目**: インデクサーによる要素アクセス（nint/NIndex）
- **93-97行目**: 範囲スライスによるサブテンソル取得
- **131-134行目**: 暗黙変換（TensorSpan/ReadOnlyTensorSpan）
- **149行目**: AsTensorSpanでSpanベース操作に変換

#### Step 3: TensorPrimitivesを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | TensorPrimitives.Single.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/TensorPrimitives.Single.cs` | float特化の基本演算 |
| 3-2 | TensorPrimitives.Dot.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/TensorPrimitives.Dot.cs` | ドット積演算 |
| 3-3 | TensorPrimitives.Sum.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/TensorPrimitives.Sum.cs` | 合計演算 |
| 3-4 | TensorPrimitives.CosineSimilarity.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/TensorPrimitives.CosineSimilarity.cs` | コサイン類似度 |

**主要処理フロー（TensorPrimitives.Dot.cs）**:
- **31-33行目**: Dot<T>メソッドがジェネリック型制約を使用
- **32-33行目**: Aggregateヘルパーでベクトル化演算を実行

#### Step 4: 数学関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | TensorPrimitives.Sigmoid.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/TensorPrimitives.Sigmoid.cs` | シグモイド活性化関数 |
| 4-2 | TensorPrimitives.SoftMax.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/TensorPrimitives.SoftMax.cs` | ソフトマックス関数 |
| 4-3 | TensorPrimitives.Exp.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/TensorPrimitives.Exp.cs` | 指数関数 |
| 4-4 | TensorPrimitives.Sin.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/TensorPrimitives.Sin.cs` | 三角関数 |

#### Step 5: 内部実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | TensorPrimitives.Helpers.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/TensorPrimitives.Helpers.cs` | SIMD演算ヘルパー |
| 5-2 | TensorOperation.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/TensorOperation.cs` | テンソル演算の抽象化 |

### プログラム呼び出し階層図

```
TensorPrimitives.Dot<T>(x, y)
    │
    └─ Aggregate<T, MultiplyOperator<T>, AddOperator<T>>(x, y)
           │
           ├─ Vector256.LoadUnsafe() [AVX2]
           │      └─ Vector256.Multiply()
           │      └─ Vector256.Sum()
           │
           ├─ Vector128.LoadUnsafe() [SSE/NEON]
           │      └─ Vector128.Multiply()
           │      └─ Vector128.Sum()
           │
           └─ スカラーループ [Fallback]
                  └─ x[i] * y[i]

Tensor<T>.AsTensorSpan()
    │
    └─ new TensorSpan<T>(ref data, in _shape)
           │
           ├─ インデクサーアクセス
           │      └─ TensorShape.GetLinearOffset()
           │
           └─ Slice操作
                  └─ 新しいTensorSpan<T>生成
```

### データフロー図

```
[入力]               [処理]                    [出力]

配列/Span<T>    TensorPrimitives演算      結果Span<T>
   │                   │                      │
   ├─ x ─────────────▶│                      │
   │                   │                      │
   ├─ y ─────────────▶│                      │
   │                   │                      │
   │              ┌────┴────┐                 │
   │              │  SIMD   │                 │
   │              │Vectorize│                 │
   │              └────┬────┘                 │
   │                   │                      │
   │                   ├─────────────────────▶│destination
   │                   │                      │
   │                   └─────────────────────▶│return value


Tensor<T>操作:

Tensor<T>          TensorSpan<T>           結果
   │                    │                    │
   ├─ AsTensorSpan ────▶│                    │
   │                    │                    │
   │              ┌─────┴─────┐              │
   │              │ Indexer   │              │
   │              │ Slice     │              │
   │              │ GetSpan   │              │
   │              └─────┬─────┘              │
   │                    │                    │
   │                    └───────────────────▶│要素/サブテンソル
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Tensor_1.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/Tensor_1.cs` | ソース | Tensor<T>クラス |
| TensorSpan_1.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/TensorSpan_1.cs` | ソース | TensorSpan<T>構造体 |
| ReadOnlyTensorSpan_1.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/ReadOnlyTensorSpan_1.cs` | ソース | ReadOnlyTensorSpan<T>構造体 |
| TensorShape.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/TensorShape.cs` | ソース | テンソル形状管理 |
| ITensor_1.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/ITensor_1.cs` | ソース | テンソルインターフェース |
| IReadOnlyTensor.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/IReadOnlyTensor.cs` | ソース | 読み取り専用テンソルインターフェース |
| NIndex.cs | `src/libraries/System.Numerics.Tensors/src/System/Buffers/NIndex.cs` | ソース | ネイティブインデックス |
| NRange.cs | `src/libraries/System.Numerics.Tensors/src/System/Buffers/NRange.cs` | ソース | ネイティブ範囲 |
| TensorPrimitives.Single.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/TensorPrimitives.Single.cs` | ソース | float特化演算 |
| TensorPrimitives.Dot.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/TensorPrimitives.Dot.cs` | ソース | ドット積 |
| TensorPrimitives.Sum.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/TensorPrimitives.Sum.cs` | ソース | 合計演算 |
| TensorPrimitives.CosineSimilarity.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/TensorPrimitives.CosineSimilarity.cs` | ソース | コサイン類似度 |
| TensorPrimitives.Sigmoid.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/TensorPrimitives.Sigmoid.cs` | ソース | シグモイド関数 |
| TensorPrimitives.SoftMax.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/TensorPrimitives.SoftMax.cs` | ソース | ソフトマックス関数 |
| TensorPrimitives.Helpers.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/TensorPrimitives.Helpers.cs` | ソース | SIMDヘルパー |
| TensorOperation.cs | `src/libraries/System.Numerics.Tensors/src/System/Numerics/Tensors/netcore/TensorOperation.cs` | ソース | 演算抽象化 |
| TensorMarshal.cs | `src/libraries/System.Numerics.Tensors/src/System/Runtime/InteropServices/TensorMarshal.cs` | ソース | マーシャリングサポート |
