# 機能設計書 33-Microsoft.Extensions.Logging

## 概要

本ドキュメントは、Microsoft.Extensions.Logging（ロギング）機能の設計内容を記述する。この機能は.NETアプリケーションにおけるログ記録フレームワークを提供し、複数のログプロバイダーを統合した一貫したロギングAPIを実現する。

### 本機能の処理概要

Microsoft.Extensions.Loggingは、アプリケーションのログ出力を統一的に管理するフレームワークである。ILoggerインターフェースを通じて、コンソール、ファイル、外部サービス等の様々な出力先に対して一貫したAPIでログを記録できる。

**業務上の目的・背景**：アプリケーションの動作状況把握、障害調査、監査ログの記録には、信頼性の高いロギング機構が不可欠である。このライブラリは、ログレベル（Trace/Debug/Information/Warning/Error/Critical）によるフィルタリング、構造化ロギング、スコープ管理を提供し、運用・保守を支援する。複数のログプロバイダーへの同時出力も可能で、分散トレーシングとの統合もサポートする。

**機能の利用シーン**：デバッグ情報の出力、エラーログの記録、パフォーマンス計測、監査証跡の記録、分散トレーシング、構造化ログによる検索性向上など、あらゆるロギングシナリオで使用される。

**主要な処理内容**：
1. ILoggerFactoryを通じたロガーの生成
2. カテゴリ名に基づくロガーのフィルタリング
3. ログレベルに基づくメッセージのフィルタリング
4. 複数のILoggerProviderへの同時出力
5. スコープによるコンテキスト情報の付与
6. 構造化ログメッセージのフォーマット

**関連システム・外部連携**：Microsoft.Extensions.Logging.Console（コンソール出力）、Microsoft.Extensions.Logging.Debug（デバッグ出力）、Serilog、NLog、log4net等のサードパーティロギングライブラリ、Application Insights、OpenTelemetry等の監視サービスと連携する。

**権限による制御**：ログ出力自体に権限制御はないが、ログファイルへのアクセス権限や、クラウドサービスへの送信権限は別途管理が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに該当なし |

## 機能種別

データ連携 / ログ記録 / 監視

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| categoryName | string | Yes | ロガーのカテゴリ名（通常はクラス名） | null不可 |
| logLevel | LogLevel | Yes | ログレベル | Trace/Debug/Information/Warning/Error/Critical/None |
| eventId | EventId | No | イベント識別子 | - |
| message | string | Yes | ログメッセージ（フォーマット文字列） | null許容 |
| args | object[] | No | メッセージの引数 | - |
| exception | Exception | No | 関連する例外 | - |

### 入力データソース

- アプリケーションコードからのログ出力呼び出し
- LoggerFilterOptionsによるフィルタ設定
- IConfigurationからのログ設定

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Logger | ILogger | 生成されたロガー |
| FormattedMessage | string | フォーマット済みログメッセージ |
| ScopeState | object | スコープ状態 |

### 出力先

- コンソール（Console Provider）
- デバッグ出力（Debug Provider）
- ファイル（File Provider）
- イベントログ（EventLog Provider）
- 外部サービス（Application Insights、Seq等）

## 処理フロー

### 処理シーケンス

```
1. ロガーファクトリ初期化
   └─ LoggerFactoryの作成、プロバイダーの登録
2. ロガー作成フェーズ
   └─ CreateLogger(categoryName)で各プロバイダーのロガーを集約
3. フィルタ適用フェーズ
   └─ LoggerFilterOptionsに基づきMessageLoggerを構成
4. ログ出力フェーズ
   └─ Log()呼び出しでフィルタ判定後、各プロバイダーに出力
5. スコープ管理フェーズ（オプション）
   └─ BeginScope()でスコープを開始、Disposeで終了
```

### フローチャート

```mermaid
flowchart TD
    A[LoggerFactory作成] --> B[AddProvider]
    B --> C[CreateLogger呼び出し]
    C --> D[各プロバイダーのロガー作成]
    D --> E[LoggerFilterOptions適用]
    E --> F[Loggerインスタンス返却]
    F --> G{Log呼び出し}
    G --> H{IsEnabled判定}
    H -->|false| I[出力なし]
    H -->|true| J[各MessageLoggerに出力]
    J --> K[フォーマット処理]
    K --> L[プロバイダーに出力]
    G --> M{BeginScope?}
    M -->|Yes| N[スコープ作成]
    N --> O[ScopeLogger設定]
    O --> P[Dispose時にスコープ終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-33-01 | ログレベル優先順位 | Trace < Debug < Information < Warning < Error < Critical | フィルタリング時 |
| BR-33-02 | カテゴリマッチング | 完全一致または前方一致でフィルタルール適用 | フィルタ選択時 |
| BR-33-03 | スコープ継承 | 入れ子のスコープは全て出力に含まれる | CaptureScopes有効時 |
| BR-33-04 | Activity追跡 | SpanId、TraceId等をログに自動付与 | ActivityTrackingOptions設定時 |

### 計算ロジック

フィルタルール選択順序：
1. プロバイダー名とカテゴリ名の両方に一致するルール
2. プロバイダー名のみ一致するルール
3. カテゴリ名のみ一致するルール
4. デフォルトルール

## データベース操作仕様

該当なし（ログプロバイダーによってはDB出力可能）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ObjectDisposedException | 破棄済みファクトリでCreateLogger | 新しいファクトリを使用 |
| - | ArgumentException | 無効なActivityTrackingOptions | 有効なオプションを指定 |

### リトライ仕様

プロバイダーごとの実装に依存。ファクトリレベルではリトライなし。Disposeでの例外は握りつぶされる。

## トランザクション仕様

該当なし

## パフォーマンス要件

- IsEnabled判定: 高速（インライン化推奨）
- ログ出力: プロバイダー依存
- 構造化ログ引数の評価はIsEnabledがtrueの場合のみ実行
- LoggerMessage.Defineによる静的ログメソッド生成で高速化

## セキュリティ考慮事項

- 機密情報（パスワード、トークン等）のログ出力を避ける
- ログファイルのアクセス権限を適切に設定
- 本番環境ではDebug/Traceレベルを無効化

## 備考

- ILogger<T>はジェネリック型のフルネームをカテゴリ名として使用
- LoggerMessage.Defineで高性能なログメソッドを生成可能
- SourceGeneratorによるコンパイル時ログメソッド生成もサポート

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

ロギングシステムの基本的なインターフェースとデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ILogger.cs | `src/libraries/Microsoft.Extensions.Logging.Abstractions/src/ILogger.cs` | ロガーインターフェース。Log、IsEnabled、BeginScopeメソッド |
| 1-2 | LogLevel.cs | `src/libraries/Microsoft.Extensions.Logging.Abstractions/src/LogLevel.cs` | ログレベルの列挙型。Trace〜Noneまでの7段階 |
| 1-3 | EventId.cs | `src/libraries/Microsoft.Extensions.Logging.Abstractions/src/EventId.cs` | イベント識別子の構造体 |
| 1-4 | ILoggerProvider.cs | `src/libraries/Microsoft.Extensions.Logging.Abstractions/src/ILoggerProvider.cs` | プロバイダーインターフェース |

**読解のコツ**: ILoggerFactory→ILoggerProvider→ILoggerの階層を理解する。Factoryはプロバイダーを管理し、プロバイダーはロガーを生成する。

#### Step 2: エントリーポイントを理解する

LoggerFactoryの構築とロガー生成フローを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | LoggerFactory.cs | `src/libraries/Microsoft.Extensions.Logging/src/LoggerFactory.cs` | ファクトリの実装。プロバイダー管理、ロガー生成、フィルタ適用 |
| 2-2 | LoggerFilterOptions.cs | `src/libraries/Microsoft.Extensions.Logging/src/LoggerFilterOptions.cs` | フィルタオプションの設定クラス |

**主要処理フロー**:
1. **82-105行目**: コンストラクタでプロバイダーを登録し、フィルタを初期化
2. **140-163行目**: CreateLogger()でカテゴリ名に対応するLoggerを生成
3. **169-195行目**: AddProvider()で新しいプロバイダーを追加し、既存ロガーを更新
4. **230-262行目**: ApplyFilters()でMessageLoggerとScopeLoggerを構成

#### Step 3: フィルタリングを理解する

ログフィルタリングの仕組みを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | LoggerRuleSelector.cs | `src/libraries/Microsoft.Extensions.Logging/src/LoggerRuleSelector.cs` | フィルタルール選択ロジック |
| 3-2 | LoggerFilterRule.cs | `src/libraries/Microsoft.Extensions.Logging/src/LoggerFilterRule.cs` | フィルタルールの定義 |

**主要処理フロー**:
- **LoggerRuleSelector.Select()**: プロバイダー名とカテゴリ名に基づいてルールを選択
- 優先順位: プロバイダー+カテゴリ > プロバイダーのみ > カテゴリのみ > デフォルト

#### Step 4: ロガー実装を理解する

内部ロガーの実装を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Logger.cs | `src/libraries/Microsoft.Extensions.Logging/src/Logger.cs` | 内部ロガークラス。MessageLoggerとScopeLoggerの配列を保持 |
| 4-2 | LoggerInformation.cs | `src/libraries/Microsoft.Extensions.Logging/src/LoggerInformation.cs` | プロバイダーごとのロガー情報 |

**主要処理フロー**:
- **Log()**: MessageLoggersを反復し、IsEnabled判定後に各プロバイダーのロガーにログ出力
- **BeginScope()**: ScopeLoggersを使用してスコープを開始

### プログラム呼び出し階層図

```
LoggerFactory.CreateLogger(categoryName)
    │
    ├─ _loggers.TryGetValue(categoryName)
    │      │
    │      └─ キャッシュにあれば返却
    │
    └─ new Logger(categoryName, CreateLoggers(categoryName))
           │
           ├─ foreach (registration in _providerRegistrations)
           │      │
           │      └─ new LoggerInformation(provider, categoryName)
           │
           └─ ApplyFilters(loggerInfo)
                  │
                  ├─ LoggerRuleSelector.Select()
                  │
                  └─ MessageLogger/ScopeLogger配列作成

Logger.Log(logLevel, eventId, state, exception, formatter)
    │
    └─ foreach (logger in MessageLoggers)
           │
           ├─ IsEnabled判定
           │
           └─ logger.Log(logLevel, eventId, state, exception, formatter)
```

### データフロー図

```
[入力]                    [処理]                          [出力]

LoggerFactory ───▶ AddProvider() ───▶ _providerRegistrations
                                              │
                                              ▼
CreateLogger() ───▶ LoggerInformation[] ───▶ Logger
                           │                    │
                           ▼                    ▼
                   ApplyFilters() ───▶ MessageLogger[]
                                              │
                                              ▼
Log() ─────────────────────────────▶ foreach MessageLogger
                                              │
                                              ▼
                                      provider.Log()
                                              │
                                              ▼
                                      Console/File/DB等
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| LoggerFactory.cs | `src/libraries/Microsoft.Extensions.Logging/src/LoggerFactory.cs` | ソース | ファクトリ実装 |
| Logger.cs | `src/libraries/Microsoft.Extensions.Logging/src/Logger.cs` | ソース | 内部ロガー実装 |
| LoggerFilterOptions.cs | `src/libraries/Microsoft.Extensions.Logging/src/LoggerFilterOptions.cs` | ソース | フィルタオプション |
| LoggerRuleSelector.cs | `src/libraries/Microsoft.Extensions.Logging/src/LoggerRuleSelector.cs` | ソース | ルール選択 |
| LoggerFilterRule.cs | `src/libraries/Microsoft.Extensions.Logging/src/LoggerFilterRule.cs` | ソース | フィルタルール |
| LoggerInformation.cs | `src/libraries/Microsoft.Extensions.Logging/src/LoggerInformation.cs` | ソース | ロガー情報 |
| LoggerFactoryScopeProvider.cs | `src/libraries/Microsoft.Extensions.Logging/src/LoggerFactoryScopeProvider.cs` | ソース | スコープ管理 |
| LoggingBuilder.cs | `src/libraries/Microsoft.Extensions.Logging/src/LoggingBuilder.cs` | ソース | ビルダー |
| ActivityTrackingOptions.cs | `src/libraries/Microsoft.Extensions.Logging/src/ActivityTrackingOptions.cs` | ソース | Activity追跡オプション |
