# 機能設計書 4-JITコンパイラ

## 概要

本ドキュメントは、.NET runtimeプロジェクトにおけるJIT（Just-In-Time）コンパイラの機能設計について記述する。JITコンパイラは、IL（Intermediate Language）コードを実行時にネイティブコードに変換するコンポーネントであり、RyuJITとして知られる高性能JITエンジンを実装している。

### 本機能の処理概要

**業務上の目的・背景**：JITコンパイラは、.NETアプリケーションの実行性能を決定づける中核コンポーネントである。IL（中間言語）で記述されたマネージドコードを、実行対象のCPUアーキテクチャに最適化されたネイティブコードに変換することで、ポータビリティと高性能を両立させる。Tiered Compilationにより、起動速度と長期実行時のパフォーマンスを最適化する。

**機能の利用シーン**：
- メソッド初回呼び出し時のJITコンパイル
- ホットパスの最適化再コンパイル（Tiered Compilation）
- デバッグ時のJITデバッグ情報生成
- プロファイルガイド最適化（PGO）

**主要な処理内容**：
1. IL命令のパース（インポート）
2. 内部中間表現（GenTree）への変換
3. 最適化パス（インライン展開、ループ最適化、CSE等）
4. レジスタ割り当て
5. ネイティブコード生成（エミット）
6. GC情報、例外ハンドラ情報の生成

**関連システム・外部連携**：
- 実行エンジン（EE）とのJITインターフェース
- GC（ガベージコレクター）との連携（GCセーフポイント情報）
- デバッガーとの連携（シーケンスポイント情報）

**権限による制御**：
- 実行可能メモリ領域の確保と保護

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | JITコンパイラは画面を持たない内部コンポーネント |

## 機能種別

計算処理 / コード変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ILコード | バイト配列 | Yes | コンパイル対象のILメソッドバイト列 | IL検証 |
| メタデータ | 構造体 | Yes | 型情報、メソッドシグネチャ | メタデータ整合性 |
| JITフラグ | enum | No | 最適化レベル、デバッグモード等 | 有効値検証 |

### 入力データソース

- 実行エンジンからのcompileMethod()コールバック
- アセンブリのILメタデータ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ネイティブコード | バイト配列 | 生成されたマシンコード |
| GC情報 | 構造体 | GCセーフポイント、参照情報 |
| 例外ハンドラ情報 | 構造体 | try-catch-finally情報 |
| デバッグ情報 | 構造体 | シーケンスポイント、変数情報 |

### 出力先

- コードヒープ（実行可能メモリ領域）
- GC情報テーブル
- 例外処理テーブル

## 処理フロー

### 処理シーケンス

```
1. compileMethod()呼び出し
   └─ JITコンパイル要求受付
2. Pre-import フェーズ
   └─ メソッド情報の収集
3. Import（インポート）フェーズ
   └─ ILコードのGenTree変換
4. Morph（変形）フェーズ
   └─ GenTreeの正規化
5. Flowgraph Analysis
   └─ 制御フロー解析
6. Optimization（最適化）フェーズ
   └─ インライン展開、CSE、ループ最適化等
7. Rationalize（合理化）フェーズ
   └─ LIR（Linear IR）への変換
8. Lowering（低レベル化）フェーズ
   └─ プラットフォーム固有変換
9. Register Allocation（レジスタ割り当て）
   └─ 線形スキャンレジスタ割り当て
10. Code Generation（コード生成）
    └─ ネイティブ命令エミット
```

### フローチャート

```mermaid
flowchart TD
    A[IL入力] --> B[Import]
    B --> C[GenTree構築]
    C --> D[Morph]
    D --> E[最適化]
    E --> F[Rationalize]
    F --> G[Lowering]
    G --> H[LSRA]
    H --> I[CodeGen]
    I --> J[ネイティブコード出力]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | Tiered Compilation | Tier0（高速）→ Tier1（最適化）の段階的コンパイル | 標準実行時 |
| BR-02 | OSR（On-Stack Replacement） | 実行中メソッドの最適化版への切り替え | ホットループ検出時 |
| BR-03 | インライン展開判定 | メソッドサイズ、呼び出し回数等に基づく判定 | 最適化フェーズ |

### 計算ロジック

- インライン判定：コードサイズ、呼び出し頻度、IL命令種別の重み付け評価
- レジスタ割り当て：線形スキャンアルゴリズムによる生存区間解析

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | JITコンパイラはデータベースを操作しない |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| JIT_REFUSED | コンパイル拒否 | サポートされないILパターン | インタープリターへのフォールバック |
| OOM | メモリ不足 | コードヒープ枯渇 | GCトリガー、メモリ増強 |
| INTERNAL_ERROR | 内部エラー | JITバグ | デバッグビルドでassert |

### リトライ仕様

- コンパイル失敗時：Tier0コンパイルで再試行（オプション）

## トランザクション仕様

JITコンパイラはトランザクション管理を行わない。

## パフォーマンス要件

- Tier0コンパイル：ミリ秒オーダーの高速コンパイル
- Tier1コンパイル：最適化優先（バックグラウンド実行可）
- レジスタ割り当て：線形スキャンによるO(n)計算量

## セキュリティ考慮事項

- W^X（Write XOR Execute）：コード生成後に書き込み不可・実行可に変更
- スタックオーバーフロー検出：スタックプローブの挿入
- 制御フロー整合性：間接呼び出しの検証

## 備考

- RyuJITは.NET Core以降の標準JITコンパイラ
- x64、ARM64、x86、ARM32等の複数アーキテクチャをサポート

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | compiler.h | `src/coreclr/jit/compiler.h` | Compilerクラス、メソッドコンパイル単位のデータ |
| 1-2 | gentree.h | `src/coreclr/jit/gentree.h` | GenTree（中間表現のノード）構造 |
| 1-3 | block.h | `src/coreclr/jit/block.h` | BasicBlock（基本ブロック）構造 |

**読解のコツ**:
- **compiler.h 9-16行目**: Compilerクラスの説明コメント。メソッドごとにインスタンスが作成される。
- **compiler.h 102行目**: `class Compiler;` - メインのJITコンパイラクラス
- **compiler.h 116-117行目**: Compilerコンテキストを使用したメモリアロケータのオーバーロード
- **compiler.h 137行目**: `FLG_CCTOR` - 静的コンストラクタフラグ定義

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | jit.h | `src/coreclr/jit/jit.h` | JITインターフェース定義 |
| 2-2 | ee_il_dll.cpp | `src/coreclr/jit/ee_il_dll.cpp` | compileMethod実装 |

**主要処理フロー**:
- compileMethod()がEEから呼び出されるエントリーポイント
- Compilerインスタンスの作成と各フェーズの実行

#### Step 3: 主要フェーズを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | importer.cpp | `src/coreclr/jit/importer.cpp` | ILインポート処理 |
| 3-2 | morph.cpp | `src/coreclr/jit/morph.cpp` | GenTree変形処理 |
| 3-3 | optimizer.cpp | `src/coreclr/jit/optimizer.cpp` | 最適化処理 |
| 3-4 | lsra.cpp | `src/coreclr/jit/lsra.cpp` | 線形スキャンレジスタ割り当て |
| 3-5 | codegen.cpp | `src/coreclr/jit/codegen.cpp` | コード生成 |

#### Step 4: HFA（Homogeneous Floating-point Aggregate）を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | compiler.h (146-190行目) | `src/coreclr/jit/compiler.h` | HFA型判定関数 |

**主要処理フロー**:
- **146-149行目**: `IsHfa()` - HFA判定
- **150-170行目**: `HfaTypeFromElemKind()` - HFA要素型から変数型への変換
- **171-191行目**: `HfaElemKindFromType()` - 変数型からHFA要素型への変換

### プログラム呼び出し階層図

```
ICorJitCompiler::compileMethod() [ee_il_dll.cpp]
    │
    ├─ Compiler::compInit()
    │      └─ メモリアロケータ初期化
    │
    ├─ Compiler::compCompile() [compiler.cpp]
    │      │
    │      ├─ fgFindBasicBlocks() - 基本ブロック構築
    │      │
    │      ├─ fgImport() [importer.cpp]
    │      │      └─ IL → GenTree 変換
    │      │
    │      ├─ fgMorph() [morph.cpp]
    │      │      └─ GenTree 正規化
    │      │
    │      ├─ optOptimize() [optimizer.cpp]
    │      │      ├─ optInline() - インライン展開
    │      │      ├─ optLoopOpt() - ループ最適化
    │      │      └─ optCSE() - 共通部分式除去
    │      │
    │      ├─ Rationalize() [rationalize.cpp]
    │      │      └─ LIR変換
    │      │
    │      ├─ Lowering() [lower.cpp]
    │      │      └─ プラットフォーム固有変換
    │      │
    │      ├─ LSRA() [lsra.cpp]
    │      │      └─ レジスタ割り当て
    │      │
    │      └─ CodeGen() [codegencommon.cpp]
    │             └─ ネイティブ命令生成
    │
    └─ コード返却
```

### データフロー図

```
[入力]                    [処理]                         [出力]

ILバイト列 ──────▶ Importer ──────▶ GenTree HIR
     │                │
     │                ▼
     │            Optimizer
     │                │
     ▼                ▼
型情報 ──────▶ Rationalize ──────▶ LIR (Linear IR)
     │                │
     │                ▼
     │            Lowering
     │                │
     ▼                ▼
ABI情報 ──────▶ LSRA ──────▶ レジスタ割り当て済みLIR
     │                │
     │                ▼
     │            CodeGen
     │                │
     ▼                ▼
ターゲットCPU ────▶ Emitter ──────▶ ネイティブコード
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| compiler.h | `src/coreclr/jit/compiler.h` | ヘッダ | Compilerクラス定義 |
| compiler.cpp | `src/coreclr/jit/compiler.cpp` | ソース | Compiler実装 |
| jit.h | `src/coreclr/jit/jit.h` | ヘッダ | JITインターフェース |
| gentree.h | `src/coreclr/jit/gentree.h` | ヘッダ | GenTree定義 |
| block.h | `src/coreclr/jit/block.h` | ヘッダ | BasicBlock定義 |
| importer.cpp | `src/coreclr/jit/importer.cpp` | ソース | ILインポート |
| morph.cpp | `src/coreclr/jit/morph.cpp` | ソース | GenTree変形 |
| optimizer.cpp | `src/coreclr/jit/optimizer.cpp` | ソース | 最適化処理 |
| lsra.cpp | `src/coreclr/jit/lsra.cpp` | ソース | レジスタ割り当て |
| codegen.cpp | `src/coreclr/jit/codegen.cpp` | ソース | コード生成 |
| emit.h | `src/coreclr/jit/emit.h` | ヘッダ | 命令エミッター |
| instr.h | `src/coreclr/jit/instr.h` | ヘッダ | 命令定義 |
| regalloc.h | `src/coreclr/jit/regalloc.h` | ヘッダ | レジスタ割り当て |
