# 機能設計書 45-WebcilConverter

## 概要

本ドキュメントは、.NETランタイムにおけるWebcilConverterビルドタスクの機能設計を記述したものである。WebcilConverterは、.NETアセンブリ（PE形式）をWebcil形式に変換する機能を提供し、Blazor WebAssemblyアプリケーションのダウンロード最適化を実現する。

### 本機能の処理概要

WebcilConverterは、標準的なPE COFF形式の.NETアセンブリを、Web環境に最適化されたWebcil形式に変換する。Webcil形式は、PE形式から不要なヘッダーを除去し、WebAssembly環境でのダウンロードサイズを削減する。

**業務上の目的・背景**：Blazor WebAssemblyアプリケーションでは、.NETアセンブリ（DLL）をブラウザにダウンロードして実行する。標準的なPE形式のDLLファイルは、一部のセキュリティソフトウェアやファイアウォールによってブロックされる場合がある。また、PE形式には実行時に不要なヘッダー情報が含まれており、ダウンロードサイズの観点で非効率である。Webcil形式は、これらの問題を解決するために設計された軽量フォーマットである。

**機能の利用シーン**：
- Blazor WebAssemblyアプリケーションの公開時
- WebAssembly向けAOTコンパイル時
- ブラウザ配信用アセンブリの最適化時

**主要な処理内容**：
1. PE形式アセンブリの読み込みと解析
2. PE COFFヘッダーからWebcilヘッダーへの変換
3. セクションヘッダーの変換
4. セクションデータのコピー
5. デバッグディレクトリエントリの調整
6. WebAssemblyラッパーへの格納（オプション）

**関連システム・外部連携**：
- Blazor WebAssemblyビルドパイプライン
- MSBuildシステム

**権限による制御**：特になし。ファイルシステムへの読み書き権限が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | ビルドプロセス（CLI） | 主機能 | Blazor WebAssemblyアプリビルド時に呼び出される |

## 機能種別

ビルドタスク / フォーマット変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| inputPath | string | Yes | 入力PEファイルのパス | ファイルが存在すること |
| outputPath | string | Yes | 出力Webcilファイルのパス | - |

### 入力データソース

- 標準的な.NETアセンブリファイル（PE COFF形式、.dll）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Webcilファイル | binary | 変換後のWebcilファイル |

### 出力先

- Webcilファイル（.webcil/.wasm）: outputPath

## 処理フロー

### 処理シーケンス

```
1. 入力ファイルのオープン
   └─ PE形式ファイルの読み込み

2. PEヘッダー情報の収集
   └─ セクションヘッダーの解析
   └─ デバッグディレクトリの読み取り

3. Webcilヘッダーの生成
   └─ WebcilHeader構造体の初期化
   └─ バージョン情報の設定

4. セクションヘッダーの変換
   └─ 各セクションのオフセット再計算

5. 出力ファイルの生成
   └─ ヘッダーの書き込み
   └─ セクションデータのコピー
   └─ デバッグディレクトリの調整

6. WebAssemblyラッパー（オプション）
   └─ WASMモジュールとして格納
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[入力PEファイル読み込み]
    B --> C[PEヘッダー解析]
    C --> D[セクション情報収集]
    D --> E[Webcilヘッダー生成]
    E --> F[セクションヘッダー変換]
    F --> G{WrapInWebAssembly?}
    G -->|Yes| H[WASMラッパー生成]
    G -->|No| I[直接出力]
    H --> J[出力ファイル書き込み]
    I --> J
    J --> K[デバッグディレクトリ調整]
    K --> L[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-45-01 | マジックナンバー | Webcilファイルは"WbIL"で始まる | 常時 |
| BR-45-02 | バージョン | version_major/minorはConstants定義に従う | 常時 |
| BR-45-03 | セクションオフセット | セクションオフセットはヘッダー後から再計算 | 常時 |
| BR-45-04 | デバッグディレクトリ | ポインタはPEからWebcilへの差分を調整 | デバッグ情報存在時 |

### 計算ロジック

セクションオフセットの計算:
```csharp
curSectionPos = SizeOfHeader() + sizeof(WebcilSectionHeader) * coffHeader.NumberOfSections;
newHeader.pointerToRawData = curSectionPos.Position;
curSectionPos += sectionHeader.SizeOfRawData;
```

デバッグポインタの調整:
```csharp
dataPointerAdjustment = peInfo.SectionStart.Position - wcInfo.SectionStart.Position;
newDataPointer = entry.DataPointer - dataPointerAdjustment;
```

## データベース操作仕様

該当なし（ファイルシステム操作のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ファイル不在 | 入力ファイルが存在しない | 正しいパスを指定 |
| - | 形式エラー | 入力がPE形式でない | 有効なPEファイルを指定 |
| InvalidOperationException | セクションエラー | オフセットがどのセクションにも属さない | ファイル整合性を確認 |

### リトライ仕様

リトライは行わない。

## トランザクション仕様

該当なし

## パフォーマンス要件

- 変換処理は入力ファイルサイズに比例
- メモリ効率のためストリーミング処理を使用

## セキュリティ考慮事項

- 入力ファイルの検証（PE形式チェック）
- 出力ファイルの上書き制御

## 備考

- Webcil形式はBlazor WebAssembly専用の最適化フォーマット
- WASMラッパーにより、ファイアウォール/プロキシでのブロックを回避

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、Webcil形式のデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | WebcilConverter.cs | `src/tasks/Microsoft.NET.WebAssembly.Webcil/WebcilConverter.cs` | 20-39行目のPEFileInfo/WCFileInfoレコード定義 |

**読解のコツ**:
- PEFileInfo: 入力PEファイルから抽出する情報
- WCFileInfo: 出力Webcilファイルに書き込む情報
- FilePositionは位置を表す値型

#### Step 2: エントリーポイントを理解する

変換処理のエントリーポイントを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | WebcilConverter.cs | `src/tasks/Microsoft.NET.WebAssembly.Webcil/WebcilConverter.cs` | 54-56行目のFromPortableExecutable(), 57-84行目のConvertToWebcil() |

**主要処理フロー**:
1. **54-55行目**: ファクトリメソッドでインスタンス作成
2. **59-65行目**: PEReaderでヘッダー情報収集
3. **68-84行目**: WrapInWebAssemblyに応じた出力

#### Step 3: 情報収集処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | WebcilConverter.cs | `src/tasks/Microsoft.NET.WebAssembly.Webcil/WebcilConverter.cs` | 110-176行目のGatherInfo() |

**主要処理フロー**:
- **117-128行目**: WebcilHeaderの初期化（マジック"WbIL"）
- **134-164行目**: 各セクションのオフセット再計算
- **168-175行目**: peInfoとwcInfoの生成

#### Step 4: 出力処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | WebcilConverter.cs | `src/tasks/Microsoft.NET.WebAssembly.Webcil/WebcilConverter.cs` | 86-96行目のWriteConversionTo(), 178-191行目のWriteHeader() |

### プログラム呼び出し階層図

```
FromPortableExecutable(inputPath, outputPath)
    │
    └─ ConvertToWebcil()
           │
           ├─ PEReader生成
           │
           ├─ GatherInfo()
           │      ├─ WebcilHeader初期化
           │      ├─ セクションヘッダー変換
           │      └─ デバッグディレクトリ読み取り
           │
           └─ WrapInWebAssembly?
                  │
                  ├─ Yes: WebcilWasmWrapper.WriteWasmWrappedWebcil()
                  │
                  └─ No: WriteConversionTo()
                           ├─ WriteHeader()
                           ├─ WriteSectionHeaders()
                           ├─ CopySections()
                           └─ OverwriteDebugDirectoryEntries()
```

### データフロー図

```
[入力]                    [処理]                           [出力]

PE COFF (.dll)
    │
    ├─ DOSヘッダー ─────▶ (削除)
    │
    ├─ PEヘッダー ──────▶ WebcilHeader変換 ───────────▶ Webcilヘッダー
    │
    ├─ セクションヘッダー ▶ オフセット再計算 ──────────▶ Webcilセクションヘッダー
    │
    └─ セクションデータ ──▶ そのままコピー ──────────▶ セクションデータ
                                    │
                                    ▼
                             WrapInWebAssembly?
                                    │
                             ┌──────┴──────┐
                             ▼             ▼
                        .webcil        .wasm
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| WebcilConverter.cs | `src/tasks/Microsoft.NET.WebAssembly.Webcil/WebcilConverter.cs` | ソース | PE→Webcil変換ロジック |
| WebcilHeader.cs | `src/tasks/Microsoft.NET.WebAssembly.Webcil/WebcilHeader.cs` | ソース | Webcilヘッダー構造定義 |
| WebcilSectionHeader.cs | `src/tasks/Microsoft.NET.WebAssembly.Webcil/WebcilSectionHeader.cs` | ソース | セクションヘッダー定義 |
| WebcilWasmWrapper.cs | `src/tasks/Microsoft.NET.WebAssembly.Webcil/WebcilWasmWrapper.cs` | ソース | WASMラッパー生成 |
| Constants.cs | `src/tasks/Microsoft.NET.WebAssembly.Webcil/Internal/Constants.cs` | ソース | バージョン定数 |
