# 機能設計書 51-minipal

## 概要

本ドキュメントは、.NETランタイムにおけるminipal（ミニPAL：Mini Platform Abstraction Layer）の機能設計を記述する。minipalは、.NETランタイムの各コンポーネント（CoreCLR、Mono、ネイティブホスト等）で共通して使用される軽量なプラットフォーム抽象化レイヤーである。

### 本機能の処理概要

minipalは、Windows、Linux、macOS、iOS、Android、WebAssemblyなど多様なプラットフォームで動作する.NETランタイムのために、OSやハードウェア固有の機能を抽象化し、統一されたC言語インターフェースを提供する機能である。

**業務上の目的・背景**：.NETランタイムは複数のプラットフォームで動作する必要があり、各プラットフォーム固有のAPIを直接呼び出すとコードの重複やメンテナンス負担が増大する。minipalはこの問題を解決し、プラットフォーム間の差異を吸収することで、ランタイム開発者が単一のAPIセットで開発できるようにする。

**機能の利用シーン**：minipalは以下の場面で利用される：
- ランタイム初期化時のCPU機能検出（AVX、AVX2、AVX512、ARM NEON、SVE等）
- 暗号論的に安全な乱数生成（セキュリティトークン、GUIDの生成等）
- UTF-8/UTF-16文字列変換（国際化対応、ファイルパス処理）
- スレッド同期プリミティブ（Mutex操作）
- 時刻取得とタイムスタンプ処理

**主要な処理内容**：
1. CPU機能検出（`minipal_getcpufeatures`）：実行時にCPUがサポートするSIMD命令セットを検出
2. 暗号論的乱数生成（`minipal_get_cryptographically_secure_random_bytes`）：OS固有のAPIを使用した安全な乱数生成
3. UTF-8/UTF-16変換（`minipal_convert_utf8_to_utf16`等）：文字エンコーディング変換
4. GUID生成（`minipal_guid_*`関数群）：一意識別子の生成
5. デバッガ検出（`minipal_debugger_*`関数群）：デバッグ環境の検出

**関連システム・外部連携**：
- Windows: BCrypt API、Windows API
- Linux: getrandom syscall、/dev/urandom
- macOS/iOS: CommonCrypto (CCRandomGenerateBytes)
- WebAssembly: ブラウザのCrypto API

**権限による制御**：minipalは低レベルライブラリであり、特別な権限制御は行わない。呼び出し元のコンポーネントが適切な権限管理を行う責務を持つ。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | minipalは内部ライブラリのため、直接的なUI関連はない |

## 機能種別

システム基盤機能 / プラットフォーム抽象化

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| buffer | uint8_t* | Yes | 乱数を格納するバッファ | NULL不可 |
| bufferLength | int32_t | Yes | バッファサイズ（バイト数） | 正の値 |
| source | const char* | Yes | 変換元UTF-8文字列 | NULL不可 |
| sourceLength | size_t | Yes | 変換元文字列長 | 0以上 |
| destination | CHAR16_T* | Yes | 変換先UTF-16バッファ | NULL不可 |
| destinationLength | size_t | Yes | 変換先バッファサイズ | 十分なサイズ |

### 入力データソース

- OS提供のエントロピーソース（/dev/urandom、BCrypt等）
- CPUIDレジスタ（CPU機能検出用）
- システム時刻（タイムスタンプ取得用）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| cpufeatures | int | CPU機能フラグのビットマスク |
| random_result | int32_t | 0:成功、-1:失敗 |
| converted_length | size_t | 変換された文字数 |

### 出力先

- 呼び出し元が提供するバッファ
- 戻り値

## 処理フロー

### 処理シーケンス

```
1. CPU機能検出 (minipal_getcpufeatures)
   └─ CPUIDを実行し、対応命令セットを判定
   └─ OSのXState機能有効化状態を確認
   └─ 結果をビットマスクとして返却

2. 暗号論的乱数生成 (minipal_get_cryptographically_secure_random_bytes)
   └─ プラットフォーム固有APIを呼び出し
   └─ Windows: BCryptGenRandom
   └─ macOS: CCRandomGenerateBytes
   └─ Linux: getrandom() → fallback to /dev/urandom
   └─ 成功/失敗を返却

3. UTF-8/UTF-16変換
   └─ 入力文字列のバリデーション
   └─ マルチバイトシーケンスの解析
   └─ サロゲートペアの処理
   └─ 変換結果をバッファに格納
```

### フローチャート

```mermaid
flowchart TD
    A[開始: CPU機能検出] --> B{プラットフォーム判定}
    B -->|x86/x64| C[CPUID命令実行]
    B -->|ARM64| D[HWCAP/sysctl取得]
    B -->|RISC-V| E[hwprobe syscall]
    C --> F[XState有効化確認]
    D --> G[機能フラグ設定]
    E --> G
    F --> G
    G --> H[ビットマスク返却]
    H --> I[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-51-1 | ベースラインISA検証 | SSE2（x64）またはAdvSIMD（ARM64）がサポートされていない場合、Invalid フラグを設定 | CPU機能検出時 |
| BR-51-2 | 乱数フォールバック | getrandomが利用不可の場合、/dev/urandomにフォールバック | Linux環境 |
| BR-51-3 | 無効UTF-8処理 | 無効なUTF-8シーケンスは置換文字(U+FFFD)で置換 | UTF-8変換時、MINIPAL_MB_NO_REPLACE_INVALID_CHARS未指定時 |

### 計算ロジック

CPU機能フラグは以下のビットマスクで表現される：
- x86/x64: AVX(1<<0), AVX2(1<<1), AVX512(1<<2), ...
- ARM64: AES(1<<0), CRC32(1<<1), DP(1<<2), ...

## データベース操作仕様

### 操作別データベース影響一覧

minipalはデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| -1 | 乱数生成失敗 | OS APIが失敗 | 呼び出し元でリトライまたは代替処理 |
| MINIPAL_ERROR_INSUFFICIENT_BUFFER | バッファ不足 | 変換先バッファが小さい | より大きなバッファで再実行 |
| MINIPAL_ERROR_NO_UNICODE_TRANSLATION | 変換不可 | 無効なUTF-8シーケンス | 置換文字使用またはエラー報告 |

### リトライ仕様

- getrandomでEINTRが発生した場合は自動的にリトライ
- ファイルディスクリプタオープン失敗時は一度だけリトライ

## トランザクション仕様

minipalはトランザクション処理を行わない。すべての操作はステートレスである。

## パフォーマンス要件

- CPU機能検出: 起動時に1回のみ実行、結果はキャッシュ可能
- 乱数生成: OSカーネルコール1回で完了することが望ましい
- UTF-8変換: ASCII文字に対して8バイト/ループの高速パス実装

## セキュリティ考慮事項

- 暗号論的乱数生成は、OS提供の暗号強度保証されたAPIのみを使用
- 非暗号論的乱数は、暗号論的乱数にXORすることで追加のエントロピーを混入
- UTF-8変換は、サロゲートペアの不正使用を検出し拒否

## 備考

minipalはC言語で実装されており、.NETランタイムの最も低レベルな基盤として機能する。新しいプラットフォームのサポートを追加する際は、minipalの各機能に対応する実装を提供する必要がある。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、minipalで定義されているデータ型とマクロを理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | cpufeatures.h | `src/native/minipal/cpufeatures.h` | CPU機能フラグの定義（XArchIntrinsicConstants_*, ARM64IntrinsicConstants_*） |
| 1-2 | types.h | `src/native/minipal/types.h` | 基本型の定義 |
| 1-3 | utf8.h | `src/native/minipal/utf8.h` | UTF-8変換関数のシグネチャとエラーコード |

**読解のコツ**: CPU機能フラグはビットマスクとして定義されており、JITコンパイラのHardwareIntrinsicHelpers.csと対応関係にある。

#### Step 2: エントリーポイントを理解する

各機能のエントリーポイントとなる公開関数を特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | cpufeatures.c | `src/native/minipal/cpufeatures.c` | `minipal_getcpufeatures`関数の実装 |
| 2-2 | random.c | `src/native/minipal/random.c` | 乱数生成関数の実装 |
| 2-3 | utf8.c | `src/native/minipal/utf8.c` | UTF-8変換関数の実装 |

**主要処理フロー**:
1. **219-449行目(cpufeatures.c)**: `minipal_getcpufeatures`がCPUIDを使用してx86/x64の機能を検出
2. **452-610行目(cpufeatures.c)**: ARM64のHWCAP/sysctlによる機能検出
3. **79-203行目(random.c)**: プラットフォーム別の暗号論的乱数生成

#### Step 3: プラットフォーム固有実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | random.c | `src/native/minipal/random.c` | Windows/Linux/macOS別の乱数生成実装 |
| 3-2 | cpufeatures.c | `src/native/minipal/cpufeatures.c` | XState有効化確認（AVX/AVX512サポート） |

**主要処理フロー**:
- **94-100行目(random.c)**: macOSでのCCRandomGenerateBytes使用
- **101-103行目(random.c)**: WindowsでのBCryptGenRandom使用
- **106-141行目(random.c)**: Linuxでのgetrandom syscall使用とフォールバック

#### Step 4: UTF-8変換の詳細を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | utf8.c | `src/native/minipal/utf8.c` | マルチバイトシーケンスの解析ロジック |

**主要処理フロー**:
- **361-744行目**: GetCharCount - UTF-8からUTF-16への文字数計算
- **754-1295行目**: GetChars - 実際の変換処理
- **1297-1715行目**: GetBytes - UTF-16からUTF-8への変換

### プログラム呼び出し階層図

```
.NET Runtime (CoreCLR/Mono)
    │
    ├─ minipal_getcpufeatures()
    │      ├─ __cpuid() [x86/x64]
    │      ├─ xmmYmmStateSupport()
    │      ├─ avx512StateSupport()
    │      ├─ getauxval(AT_HWCAP) [ARM64 Linux]
    │      └─ sysctlbyname() [ARM64 macOS]
    │
    ├─ minipal_get_cryptographically_secure_random_bytes()
    │      ├─ BCryptGenRandom() [Windows]
    │      ├─ CCRandomGenerateBytes() [macOS]
    │      ├─ getrandom() [Linux]
    │      └─ open("/dev/urandom") + read() [fallback]
    │
    └─ minipal_convert_utf8_to_utf16()
           ├─ GetCharCount()
           └─ GetChars()
```

### データフロー図

```
[入力]                    [処理]                         [出力]

CPU情報           ───▶ minipal_getcpufeatures() ───▶ CPU機能フラグ(int)
(CPUID/HWCAP)

OSエントロピー    ───▶ minipal_get_cryptographically  ───▶ 乱数バッファ
                      _secure_random_bytes()

UTF-8文字列       ───▶ minipal_convert_utf8_to_utf16() ───▶ UTF-16文字列
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| cpufeatures.h | `src/native/minipal/cpufeatures.h` | ヘッダ | CPU機能フラグ定義 |
| cpufeatures.c | `src/native/minipal/cpufeatures.c` | ソース | CPU機能検出実装 |
| cpuid.h | `src/native/minipal/cpuid.h` | ヘッダ | CPUID命令ラッパー |
| random.h | `src/native/minipal/random.h` | ヘッダ | 乱数生成関数宣言 |
| random.c | `src/native/minipal/random.c` | ソース | 乱数生成実装 |
| utf8.h | `src/native/minipal/utf8.h` | ヘッダ | UTF-8変換関数宣言 |
| utf8.c | `src/native/minipal/utf8.c` | ソース | UTF-8変換実装 |
| guid.h | `src/native/minipal/guid.h` | ヘッダ | GUID操作関数宣言 |
| guid.c | `src/native/minipal/guid.c` | ソース | GUID操作実装 |
| mutex.h | `src/native/minipal/mutex.h` | ヘッダ | Mutex操作関数宣言 |
| mutex.c | `src/native/minipal/mutex.c` | ソース | Mutex操作実装 |
| time.h | `src/native/minipal/time.h` | ヘッダ | 時刻関数宣言 |
| time.c | `src/native/minipal/time.c` | ソース | 時刻関数実装 |
| debugger.h | `src/native/minipal/debugger.h` | ヘッダ | デバッガ検出関数宣言 |
| debugger.c | `src/native/minipal/debugger.c` | ソース | デバッガ検出実装 |
| minipal.h | `src/coreclr/minipal/minipal.h` | ヘッダ | CoreCLR固有のminipal拡張 |
