# 機能設計書 6-インタープリター

## 概要

本ドキュメントは、.NET runtimeプロジェクトにおけるILインタープリターの機能設計について記述する。インタープリターはJITコンパイルが利用できない環境向けにILコードを逐次実行する機能を提供する。

### 本機能の処理概要

**業務上の目的・背景**：インタープリターは、JITコンパイルが制限または禁止されている環境でマネージドコードを実行するための代替手段を提供する。特にWebAssembly環境やiOS等のJIT禁止プラットフォームでの実行、およびAOTコンパイルされていないコードの実行を可能にする。起動時間の短縮やメモリ使用量の削減にも寄与する。

**機能の利用シーン**：
- WebAssembly環境でのJIT不可時のフォールバック
- iOSでのAOT未対応コードの実行
- 起動時間短縮のための初期実行
- デバッグ時のステップ実行

**主要な処理内容**：
1. IL命令のデコード
2. スタックベースの演算実行
3. メソッド呼び出しの処理
4. 例外処理の実行
5. GC情報の生成と連携

**関連システム・外部連携**：
- JITコンパイラとの連携（Tiered Compilation）
- GCとの連携（GCセーフポイント）
- デバッガーとの連携

**権限による制御**：
- 特になし（ランタイム内部コンポーネント）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 2 | Wasm Browser Advanced Sample | 補助機能 | JITが利用できない環境でのILインタープリター実行 |

## 機能種別

計算処理 / コード実行

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ILバイト列 | uint8_t* | Yes | 実行対象のILメソッド | IL検証 |
| 引数 | void*[] | No | メソッド引数 | 型検証 |
| thisポインタ | void* | No | インスタンスメソッドのthis | null検証 |

### 入力データソース

- アセンブリのILメタデータ
- JITからのフォールバック要求

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 戻り値 | void* | メソッドの戻り値 |
| 例外 | Exception | スローされた例外 |

### 出力先

- 呼び出し元への戻り値返却
- 例外伝播

## 処理フロー

### 処理シーケンス

```
1. メソッド情報の取得
   └─ メタデータからの型・シグネチャ解決
2. スタックフレーム確保
   └─ ローカル変数、評価スタックの初期化
3. IL命令ループ
   └─ 命令フェッチ→デコード→実行
4. 呼び出し処理
   └─ メソッド呼び出し、フィールドアクセス
5. 例外処理
   └─ try-catch-finally の処理
6. 戻り値返却
   └─ スタックからの結果取得
```

### フローチャート

```mermaid
flowchart TD
    A[メソッド呼び出し] --> B[スタックフレーム確保]
    B --> C[IL命令フェッチ]
    C --> D[命令デコード]
    D --> E[命令実行]
    E --> F{次の命令?}
    F -->|Yes| C
    F -->|No| G{例外発生?}
    G -->|Yes| H[例外ハンドラ検索]
    G -->|No| I[戻り値返却]
    H --> J{ハンドラあり?}
    J -->|Yes| C
    J -->|No| K[例外伝播]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | JITフォールバック | JIT不可時にインタープリターで実行 | JIT禁止環境 |
| BR-02 | Tier0相当 | 最適化なしの即時実行 | 初回呼び出し時 |

### 計算ロジック

- スタック型変換：g_stackTypeFromInterpTypeによる型マッピング
- IL命令実行：オペコードに基づくディスパッチ

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | インタープリターはデータベースを操作しない |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| BADCODE | 不正IL | サポートされないオペコード | JITへのエスカレーション |
| InvalidCastException | 型キャスト失敗 | 不正な型変換 | コード修正 |

### リトライ仕様

- 不正IL：リトライなし（コンパイルエラー相当）

## トランザクション仕様

インタープリターはトランザクション管理を行わない。

## パフォーマンス要件

- 実行速度：JITの1/10〜1/100（最適化なし）
- 起動時間：JITより高速（コンパイル不要）
- メモリ使用量：コードキャッシュ不要による削減

## セキュリティ考慮事項

- IL検証：不正なILパターンの検出と拒否
- スタック境界検証：オーバーフロー防止

## 備考

- CoreCLR（x64/ARM64）とMono（WebAssembly等）で異なるインタープリター実装
- Tiered Compilationとの組み合わせで起動高速化

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | interpreter.h | `src/coreclr/interpreter/interpreter.h` | インタープリター定義 |
| 1-2 | compiler.cpp | `src/coreclr/interpreter/compiler.cpp` | スタック型マッピング |

**読解のコツ**:
- **compiler.cpp 12-25行目**: `g_stackTypeFromInterpType` - InterpTypeからStackTypeへの変換テーブル
- **compiler.cpp 27-38行目**: `g_interpTypeFromStackType` - StackTypeからInterpTypeへの変換テーブル
- **compiler.cpp 43行目**: `g_stackTypeString` - デバッグ用スタック型文字列

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | eeinterp.cpp | `src/coreclr/interpreter/eeinterp.cpp` | EE連携インターフェース |
| 2-2 | intops.cpp | `src/coreclr/interpreter/intops.cpp` | 命令実装 |

**主要処理フロー**:
- **compiler.cpp 41行目**: `t_InterpJitInfoTls` - スレッドローカルJIT情報
- **compiler.cpp 52-59行目**: `IsInterpDumpActive()` - デバッグダンプ判定
- **compiler.cpp 61-69行目**: `AssertOpCodeNotImplemented()` - 未実装オペコード処理

#### Step 3: GC連携を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | compiler.cpp (71-80行目) | `src/coreclr/interpreter/compiler.cpp` | InterpIAllocator |

**主要処理フロー**:
- **compiler.cpp 71-80行目**: `InterpIAllocator` - GCInfoEncoderへのメモリアロケータ実装

### プログラム呼び出し階層図

```
EE / JITフォールバック
    │
    ├─ InterpCompiler::Compile()
    │      └─ ILバイト列の解析
    │
    ├─ interp_exec_method() [仮想]
    │      │
    │      ├─ IL命令フェッチ
    │      │
    │      ├─ オペコードディスパッチ
    │      │      ├─ StackTypeI4操作
    │      │      ├─ StackTypeI8操作
    │      │      ├─ StackTypeR4/R8操作
    │      │      ├─ StackTypeO操作（参照型）
    │      │      └─ StackTypeVT操作（値型）
    │      │
    │      └─ 例外処理
    │
    └─ GCInfoEncoder
           └─ GC情報生成
```

### データフロー図

```
[入力]                    [処理]                         [出力]

ILバイト列 ──────▶ オペコードデコード ──────▶ 命令種別
     │                    │
     │                    ▼
     │              スタック操作
     │                    │
     ▼                    ▼
引数/ローカル ──────▶ 演算実行 ──────▶ 結果スタック
     │                    │
     │                    ▼
     │              メソッド呼び出し
     │                    │
     ▼                    ▼
戻り値 ◀──────── スタックポップ ──────▶ 呼び出し元
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| interpreter.h | `src/coreclr/interpreter/interpreter.h` | ヘッダ | インタープリター定義 |
| compiler.h | `src/coreclr/interpreter/compiler.h` | ヘッダ | InterpCompiler定義 |
| compiler.cpp | `src/coreclr/interpreter/compiler.cpp` | ソース | コンパイラ実装 |
| intops.h | `src/coreclr/interpreter/intops.h` | ヘッダ | 命令定義 |
| intops.cpp | `src/coreclr/interpreter/intops.cpp` | ソース | 命令実装 |
| eeinterp.h | `src/coreclr/interpreter/eeinterp.h` | ヘッダ | EE連携 |
| eeinterp.cpp | `src/coreclr/interpreter/eeinterp.cpp` | ソース | EE連携実装 |
| stackmap.h | `src/coreclr/interpreter/stackmap.h` | ヘッダ | スタックマップ |
| stackmap.cpp | `src/coreclr/interpreter/stackmap.cpp` | ソース | スタックマップ実装 |
| interpconfig.h | `src/coreclr/interpreter/interpconfig.h` | ヘッダ | 構成設定 |
