# 機能設計書 8-System.Collections.Concurrent

## 概要

本ドキュメントは、.NET runtimeプロジェクトにおけるSystem.Collections.Concurrentライブラリの機能設計について記述する。System.Collections.Concurrentはスレッドセーフなコレクション実装を提供する。

### 本機能の処理概要

**業務上の目的・背景**：System.Collections.Concurrentは、マルチスレッド環境で安全に使用できるコレクション型を提供する。明示的なロックを必要とせず、複数スレッドからの同時アクセスを効率的に処理する。プロデューサー・コンシューマーパターン、並列処理、非同期プログラミングなどのシナリオで活用される。

**機能の利用シーン**：
- マルチスレッド環境でのデータ共有
- プロデューサー・コンシューマーパターン
- 並列処理でのワーク分散
- 非同期タスク間のデータ受け渡し

**主要な処理内容**：
1. ConcurrentDictionary：スレッドセーフな辞書
2. ConcurrentQueue：スレッドセーフなFIFOキュー
3. ConcurrentStack：スレッドセーフなLIFOスタック
4. ConcurrentBag：スレッドセーフな順序なしコレクション
5. BlockingCollection：ブロッキング機能付きコレクション

**関連システム・外部連携**：
- System.Threading（スレッド同期プリミティブ）
- System.Threading.Tasks（並列タスク）
- System.Threading.Channels（代替チャネル実装）

**権限による制御**：
- 特になし（汎用ライブラリ）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 直接関連する画面なし |

## 機能種別

データ構造 / スレッドセーフコレクション

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| キー | TKey | Yes | 辞書キー | notnull制約 |
| 値 | TValue | Yes | 格納値 | 型制約による |
| 並行レベル | int | No | 同時アクセススレッド数予測 | 正の整数 |

### 入力データソース

- 複数スレッドからの同時アクセス
- プロデューサースレッドからのデータ投入

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 値 | TValue | 取得された値 |
| 成否 | bool | 操作成功/失敗 |

### 出力先

- コンシューマースレッドへの値返却

## 処理フロー

### 処理シーケンス

```
1. 操作要求（Add/Remove/TryGet）
   └─ スレッドからのアクセス
2. ロック取得
   └─ ストライプロックまたはロックフリー操作
3. データ操作
   └─ 内部構造の更新
4. ロック解放
   └─ 他スレッドへのアクセス許可
5. 結果返却
   └─ 成功/失敗の通知
```

### フローチャート

```mermaid
flowchart TD
    A[TryAdd要求] --> B[ハッシュ計算]
    B --> C[バケットロック取得]
    C --> D{キー存在?}
    D -->|Yes| E[false返却]
    D -->|No| F[ノード追加]
    F --> G[true返却]
    E --> H[ロック解放]
    G --> H
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | ストライプロック | バケット単位のロックでロック競合削減 | ConcurrentDictionary |
| BR-02 | ロックフリー操作 | CAS操作によるロックフリー実装 | ConcurrentQueue/Stack |
| BR-03 | スレッドローカル | スレッドごとのローカルリストで競合削減 | ConcurrentBag |

### 計算ロジック

- ConcurrentDictionary：ストライプロック + ハッシュテーブル
- ConcurrentQueue：ロックフリーリンクリスト
- ConcurrentBag：スレッドローカルリスト + ワークスティーリング

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | コレクションライブラリはデータベースを操作しない |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ArgumentNullException | 引数null | nullキー | null検証 |
| OperationCanceledException | キャンセル | BlockingCollection操作キャンセル | キャンセル処理 |
| InvalidOperationException | 不正操作 | 完了済みコレクションへの追加 | 状態確認 |

### リトライ仕様

- ロックフリー操作：内部でCASリトライ

## トランザクション仕様

コレクションライブラリはトランザクション管理を行わない。

## パフォーマンス要件

- ConcurrentDictionary読み取り：ロックなしO(1)
- ConcurrentQueue/Stack：O(1)エンキュー/デキュー
- スケーラビリティ：コア数に応じた線形スケール

## セキュリティ考慮事項

- スレッドセーフ保証：データ競合の防止
- メモリ可視性：volatile読み書き

## 備考

- .NET Framework 4.0で導入
- System.Threading.Channelsが高スループットシナリオでの代替

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ConcurrentDictionary.cs | `src/libraries/System.Private.CoreLib/src/System/Collections/Concurrent/ConcurrentDictionary.cs` | ストライプロック実装 |

**読解のコツ**:
- **ConcurrentDictionary.cs 13-19行目**: クラス説明コメント - スレッドセーフな辞書
- **ConcurrentDictionary.cs 22行目**: `ConcurrentDictionary<TKey, TValue>`クラス定義
- **ConcurrentDictionary.cs 31行目**: `volatile Tables _tables` - テーブル参照のvolatile
- **ConcurrentDictionary.cs 32-34行目**: `_budget`, `_growLockArray`, `_comparerIsDefaultForClasses`
- **ConcurrentDictionary.cs 56行目**: `DefaultCapacity = 31` - デフォルト初期容量
- **ConcurrentDictionary.cs 66行目**: `MaxLockNumber = 1024` - 最大ストライプ数

#### Step 2: ロック戦略を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ConcurrentDictionary.cs (24-49行目) | `src/libraries/System.Private.CoreLib/src/System/Collections/Concurrent/ConcurrentDictionary.cs` | ロック設計 |

**主要処理フロー**:
- **24-30行目**: `_tables`フィールドの説明 - volatileアクセスの重要性
- **36-42行目**: `_comparerIsDefaultForClasses`の説明 - 最適化フラグ
- **44-49行目**: `_initialCapacity`の説明 - 初期容量保持

#### Step 3: その他の並行コレクションを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ConcurrentStack.cs | `src/libraries/System.Collections.Concurrent/src/System/Collections/Concurrent/ConcurrentStack.cs` | ロックフリー実装 |
| 3-2 | ConcurrentBag.cs | `src/libraries/System.Collections.Concurrent/src/System/Collections/Concurrent/ConcurrentBag.cs` | スレッドローカル |
| 3-3 | BlockingCollection.cs | `src/libraries/System.Collections.Concurrent/src/System/Collections/Concurrent/BlockingCollection.cs` | ブロッキング |

### プログラム呼び出し階層図

```
ConcurrentDictionary<TKey, TValue>
    │
    ├─ TryAdd(key, value)
    │      ├─ GetBucket() - ハッシュからバケット決定
    │      ├─ AcquireLock() - ストライプロック取得
    │      └─ Node追加 → ReleaseLock()
    │
    ├─ TryGetValue(key, out value)
    │      └─ ロックなし読み取り（volatile）
    │
    └─ GetOrAdd(key, valueFactory)
           └─ TryGetValue() || TryAdd()

ConcurrentQueue<T>
    │
    ├─ Enqueue(item)
    │      └─ CAS操作でtail更新
    │
    └─ TryDequeue(out item)
           └─ CAS操作でhead更新
```

### データフロー図

```
[入力]                    [処理]                         [出力]

スレッド1 ──────▶ TryAdd ──────▶ バケット1ロック
スレッド2 ──────▶ TryAdd ──────▶ バケット2ロック
     │                │
     │                ▼
     │            並行実行
     │                │
     ▼                ▼
成功/失敗 ◀──────── 結果返却 ──────▶ データ更新
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ConcurrentDictionary.cs | `src/libraries/System.Private.CoreLib/src/System/Collections/Concurrent/ConcurrentDictionary.cs` | ソース | スレッドセーフ辞書 |
| ConcurrentQueue.cs | `src/libraries/System.Private.CoreLib/src/System/Collections/Concurrent/ConcurrentQueue.cs` | ソース | スレッドセーフキュー |
| ConcurrentStack.cs | `src/libraries/System.Collections.Concurrent/src/System/Collections/Concurrent/ConcurrentStack.cs` | ソース | スレッドセーフスタック |
| ConcurrentBag.cs | `src/libraries/System.Collections.Concurrent/src/System/Collections/Concurrent/ConcurrentBag.cs` | ソース | スレッドセーフバッグ |
| BlockingCollection.cs | `src/libraries/System.Collections.Concurrent/src/System/Collections/Concurrent/BlockingCollection.cs` | ソース | ブロッキングコレクション |
| Partitioner.cs | `src/libraries/System.Collections.Concurrent/src/System/Collections/Concurrent/Partitioner.cs` | ソース | パーティショナー |
| OrderablePartitioner.cs | `src/libraries/System.Collections.Concurrent/src/System/Collections/Concurrent/OrderablePartitioner.cs` | ソース | 順序付きパーティショナー |
