# 画面設計書 21-Counter（BlazorWebWasm）

## 概要

本ドキュメントは、BlazorWebWasm テストアセットにおけるカウンターページ（Counter.razor）の画面設計書です。InteractiveWebAssembly レンダリングモードで動作する対話型コンポーネントとして実装されています。

### 本画面の処理概要

この画面は、Blazor WebAssembly のインタラクティブなクライアントサイドレンダリング機能をデモンストレーションするためのカウンターページです。ボタンをクリックすることでカウント値が増加し、その値がリアルタイムで画面に反映されます。

**業務上の目的・背景**：本画面は .NET Runtime のテストアセットとして、Blazor WebAssembly の InteractiveWebAssembly レンダリングモードが正しく機能することを検証するために存在します。サーバーサイドとクライアントサイドを分離したアーキテクチャ（BlazorWebWasm と BlazorWebWasm.Client）において、クライアント側のコンポーネントがブラウザ上で正しく動作することを確認する目的で作成されています。

**画面へのアクセス方法**：アプリケーション起動後、ナビゲーションメニューから「Counter」リンクをクリックするか、ブラウザで直接 `/counter` パスにアクセスすることで本画面に遷移します。

**主要な操作・処理内容**：
1. 画面表示時に現在のカウント値（初期値: 0）を表示
2. 「Click me」ボタンをクリックするとカウント値が1増加
3. 増加したカウント値がリアルタイムで画面に反映される
4. 初回レンダリング時にコンソールにログを出力（デバッグ用）

**画面遷移**：Home ページ（`/`）との相互遷移が可能です。ナビゲーションメニューを通じて、Home ページおよび Weather ページへ遷移できます。エラー発生時は Error ページ（`/Error`）へ遷移します。

**権限による表示制御**：本画面は認証・認可による表示制御を実装していません。すべてのユーザーがアクセス可能です。

## 関連機能

| 機能No | 機能名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 2 | Mono | 主機能 | InteractiveWebAssemblyモードでカウンター機能を実行 |
| 59 | WebAssembly Workload | 主機能 | Blazor WebAssemblyアプリケーションのビルドと実行をサポート |

## 画面種別

機能デモ / インタラクティブコンポーネント

## URL/ルーティング

- **URL**: `/counter`
- **レンダリングモード**: `@rendermode InteractiveWebAssembly`

## 入出力項目

| 項目名 | 入出力 | データ型 | 必須 | 説明 |
|--------|--------|----------|------|------|
| currentCount | 出力 | int | - | 現在のカウント値 |

## 表示項目

| 項目名 | 表示形式 | データソース | 説明 |
|--------|----------|--------------|------|
| ページタイトル | テキスト | 固定値 | "Counter" |
| 見出し | h1 | 固定値 | "Counter" |
| カウント表示 | テキスト（role="status"） | currentCount変数 | "Current count: {currentCount}" |
| 増加ボタン | button | - | "Click me" |

## イベント仕様

### 1-ボタンクリック（Click me）

**トリガー**: 「Click me」ボタンをクリック

**処理フロー**:
1. `@onclick` イベントにより `IncrementCount()` メソッドが呼び出される
2. `currentCount` フィールドの値が1増加（`currentCount++`）
3. Blazor のステート変更検知により自動的に再レンダリング
4. 更新されたカウント値が画面に表示される

**データ変更**: `currentCount` が +1 される

### 2-初回レンダリング後処理

**トリガー**: コンポーネントの初回レンダリング完了時

**処理フロー**:
1. `OnAfterRender(bool firstRender)` がフレームワークから呼び出される
2. `firstRender` が true の場合、コンソールに "Counter.OnAfterRender" を出力
3. 以降のレンダリングでは処理をスキップ

## データベース更新仕様

### 操作別データベース影響一覧

本画面はデータベースとの連携を行いません。すべての状態はクライアントサイドのメモリ上で管理されます。

| 操作（イベント） | 対象テーブル | 操作種別 | 概要 |
|----------------|-------------|---------|------|
| - | - | - | データベース操作なし |

## メッセージ仕様

| メッセージID | メッセージ種別 | メッセージ内容 | 表示条件 |
|-------------|---------------|---------------|---------|
| - | ログ出力 | "Counter.OnAfterRender" | 初回レンダリング時（コンソール出力） |

## 例外処理

| 例外種別 | 発生条件 | 対処 |
|---------|---------|------|
| 未ハンドル例外 | WebAssembly ランタイムエラー発生時 | Error ページ（/Error）へ遷移 |

## 備考

- 本画面は `.NET Runtime` リポジトリのテストアセットであり、Blazor WebAssembly の機能検証を目的としています。
- `InteractiveWebAssembly` レンダリングモードを使用しているため、コンポーネントはクライアント側の WebAssembly ランタイム上で実行されます。
- サーバーサイド（BlazorWebWasm プロジェクト）とクライアントサイド（BlazorWebWasm.Client プロジェクト）が分離されたアーキテクチャを採用しています。

---

## コードリーディングガイド

本画面を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

本画面はシンプルな構造であり、主要なデータはコンポーネント内のフィールドとして定義されています。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Counter.razor | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm.Client/Pages/Counter.razor` | @code ブロック内の currentCount フィールド（13行目）を確認 |

**読解のコツ**: Blazor コンポーネントでは `@code` ブロック内に C# コードを記述します。フィールド変数の変更は自動的に UI の再レンダリングをトリガーします。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Program.cs | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm.Client/Program.cs` | WebAssemblyHostBuilder の初期化（3行目）を確認 |
| 2-2 | Routes.razor | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm/Components/Routes.razor` | Router コンポーネントの設定を確認。AdditionalAssemblies で Client アセンブリを参照（1行目） |

**主要処理フロー**:
1. **Program.cs 3行目**: `WebAssemblyHostBuilder.CreateDefault(args)` で Blazor WebAssembly ホストを初期化
2. **Program.cs 5行目**: `builder.Build().RunAsync()` でアプリケーションを起動
3. **Routes.razor 1行目**: Router が `/counter` パスを Counter.razor にマッピング

#### Step 3: コンポーネント実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Counter.razor | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm.Client/Pages/Counter.razor` | ページディレクティブ、レンダリングモード、イベントハンドラの実装を確認 |

**主要処理フロー**:
- **1行目**: `@page "/counter"` でルーティングパスを定義
- **2行目**: `@rendermode InteractiveWebAssembly` で WebAssembly クライアントサイドレンダリングを指定
- **10行目**: `@onclick="IncrementCount"` でボタンクリックイベントをバインド
- **15-17行目**: `IncrementCount()` メソッドでカウントをインクリメント
- **20-25行目**: `OnAfterRender()` で初回レンダリング後のログ出力

#### Step 4: レイアウトとナビゲーションを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | MainLayout.razor | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm/Components/Layout/MainLayout.razor` | 全体レイアウト構造（サイドバー、メインコンテンツ領域）を確認 |
| 4-2 | NavMenu.razor | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm/Components/Layout/NavMenu.razor` | ナビゲーションメニューの実装を確認。Counter へのリンク（18行目） |

### プログラム呼び出し階層図

```
WebAssemblyHostBuilder.CreateDefault()
    │
    └─ builder.Build().RunAsync()
           │
           └─ Router (Routes.razor)
                  │
                  ├─ MainLayout.razor
                  │      ├─ NavMenu.razor
                  │      └─ @Body
                  │
                  └─ Counter.razor (URL: /counter)
                         │
                         ├─ IncrementCount() [ボタンクリック時]
                         │      └─ currentCount++
                         │
                         └─ OnAfterRender() [レンダリング後]
                                └─ Console.WriteLine() [初回のみ]
```

### データフロー図

```
[入力]               [処理]                    [出力]

ボタンクリック ───▶ IncrementCount()
                         │
                         ▼
                   currentCount++
                         │
                         ▼
                   Blazor再レンダリング ───▶ 画面表示更新
                                              "Current count: N"
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Counter.razor | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm.Client/Pages/Counter.razor` | ソース | カウンター画面のメインコンポーネント |
| Program.cs | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm.Client/Program.cs` | ソース | クライアントサイドエントリーポイント |
| Program.cs | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm/Program.cs` | ソース | サーバーサイドエントリーポイント |
| Routes.razor | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm/Components/Routes.razor` | ソース | ルーティング設定 |
| MainLayout.razor | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm/Components/Layout/MainLayout.razor` | ソース | アプリケーションレイアウト |
| NavMenu.razor | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm/Components/Layout/NavMenu.razor` | ソース | ナビゲーションメニュー |
| App.razor | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm/Components/App.razor` | ソース | アプリケーションルートコンポーネント |
| _Imports.razor | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm.Client/_Imports.razor` | ソース | 共通名前空間インポート |
