# 画面設計書 22-Error（BlazorWebWasm）

## 概要

本ドキュメントは、BlazorWebWasm テストアセットにおけるエラーページ（Error.razor）の画面設計書です。アプリケーション実行中に未ハンドル例外が発生した場合に表示されるエラー情報画面として実装されています。

### 本画面の処理概要

この画面は、Blazor WebAssembly アプリケーションでエラーが発生した際に表示されるエラーページです。エラーの発生を通知し、Request ID を表示して問題追跡を可能にします。また、開発環境での詳細なエラー情報の表示方法についてのガイダンスを提供します。

**業務上の目的・背景**：本画面は .NET Runtime のテストアセットとして、Blazor WebAssembly アプリケーションにおけるエラーハンドリング機能を検証するために存在します。本番環境ではユーザーフレンドリーなエラーメッセージを表示し、開発環境では詳細なデバッグ情報へのアクセス方法を案内することで、適切なエラー対応フローを実現します。

**画面へのアクセス方法**：直接 `/Error` パスにアクセスするか、アプリケーション内で未ハンドル例外が発生した際に自動的に遷移します。サーバーサイドの `UseExceptionHandler("/Error")` 設定により、例外発生時にこのページへリダイレクトされます。

**主要な操作・処理内容**：
1. エラー発生の通知メッセージを表示
2. Request ID が存在する場合、その値を表示（問題追跡用）
3. Development 環境への切り替え方法を案内
4. セキュリティに関する注意事項を表示

**画面遷移**：本画面はエラー発生時の終着点であり、通常の画面遷移では到達しません。Home ページや Counter ページでエラーが発生した場合にリダイレクトされます。ユーザーはブラウザの戻るボタンまたはナビゲーションメニューを使用して他の画面に遷移できます。

**権限による表示制御**：本画面は認証・認可による表示制御を実装していません。すべてのユーザーがアクセス可能です。

## 関連機能

| 機能No | 機能名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 2 | Mono | 主機能 | Blazor WebAssemblyランタイム上でエラーページを表示 |
| 28 | System.Diagnostics.DiagnosticSource | 主機能 | エラー診断情報とRequest IDの取得 |
| 59 | WebAssembly Workload | 補助機能 | Blazor WebAssemblyアプリケーションのビルドと実行をサポート |

## 画面種別

エラーページ / システム情報表示

## URL/ルーティング

- **URL**: `/Error`
- **レンダリングモード**: サーバーサイドレンダリング（SSR）

## 入出力項目

| 項目名 | 入出力 | データ型 | 必須 | 説明 |
|--------|--------|----------|------|------|
| HttpContext | 入力 | HttpContext? | いいえ | カスケードパラメータとして受け取る HTTP コンテキスト |
| RequestId | 出力 | string? | いいえ | Activity.Current?.Id または HttpContext?.TraceIdentifier |

## 表示項目

| 項目名 | 表示形式 | データソース | 説明 |
|--------|----------|--------------|------|
| ページタイトル | テキスト | 固定値 | "Error" |
| エラー見出し | h1（text-danger） | 固定値 | "Error." |
| エラーサブ見出し | h2（text-danger） | 固定値 | "An error occurred while processing your request." |
| Request ID | code 要素 | RequestId プロパティ | Request ID 値（存在する場合のみ表示） |
| Development Mode 見出し | h3 | 固定値 | "Development Mode" |
| 開発環境案内 | p | 固定値 | 開発環境への切り替え方法の説明 |
| セキュリティ警告 | p（strong） | 固定値 | 本番環境での注意事項 |

## イベント仕様

### 1-コンポーネント初期化

**トリガー**: コンポーネントの初期化時（OnInitialized）

**処理フロー**:
1. `OnInitialized()` メソッドがフレームワークから呼び出される
2. `Activity.Current?.Id` から現在のアクティビティ ID を取得
3. アクティビティ ID が null の場合、`HttpContext?.TraceIdentifier` から取得
4. 取得した値を `RequestId` プロパティに設定

**データ変更**: `RequestId` プロパティが設定される

## データベース更新仕様

### 操作別データベース影響一覧

本画面はデータベースとの連携を行いません。

| 操作（イベント） | 対象テーブル | 操作種別 | 概要 |
|----------------|-------------|---------|------|
| - | - | - | データベース操作なし |

## メッセージ仕様

| メッセージID | メッセージ種別 | メッセージ内容 | 表示条件 |
|-------------|---------------|---------------|---------|
| MSG-001 | エラー | "Error." | 常時表示 |
| MSG-002 | エラー | "An error occurred while processing your request." | 常時表示 |
| MSG-003 | 情報 | "Request ID: {RequestId}" | RequestId が存在する場合 |
| MSG-004 | 警告 | "The Development environment shouldn't be enabled for deployed applications." | 常時表示 |

## 例外処理

| 例外種別 | 発生条件 | 対処 |
|---------|---------|------|
| 該当なし | エラーページ自体では例外処理は行わない | - |

## 備考

- 本画面は `.NET Runtime` リポジトリのテストアセットであり、Blazor WebAssembly のエラーハンドリング機能を検証する目的で作成されています。
- `System.Diagnostics.Activity` を使用してリクエストのトレース情報を取得しています。
- 開発環境と本番環境で表示内容を切り替える設計パターンを示しています。
- `ASPNETCORE_ENVIRONMENT` 環境変数による環境切り替えの説明が含まれています。

---

## コードリーディングガイド

本画面を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

本画面では診断情報を扱うため、System.Diagnostics 名前空間の理解が重要です。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Error.razor | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm/Components/Pages/Error.razor` | @code ブロック内の HttpContext と RequestId の定義（27-35行目） |

**読解のコツ**: `[CascadingParameter]` 属性は親コンポーネントから値を受け取るための Blazor の仕組みです。`HttpContext` はサーバーサイドレンダリング時にのみ値が設定されます。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Program.cs | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm/Program.cs` | エラーハンドラの設定（38行目）を確認 |

**主要処理フロー**:
1. **Program.cs 38行目**: `app.UseExceptionHandler("/Error", createScopeForErrors: true)` で例外発生時のリダイレクト先を設定
2. **Error.razor 34-35行目**: `OnInitialized()` で Request ID を取得

#### Step 3: コンポーネント実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Error.razor | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm/Components/Pages/Error.razor` | ページディレクティブ、条件付きレンダリング、診断情報取得ロジック |

**主要処理フロー**:
- **1行目**: `@page "/Error"` でルーティングパスを定義
- **2行目**: `@using System.Diagnostics` で Activity クラスを使用可能に
- **9-14行目**: `@if (ShowRequestId)` で条件付き表示
- **27-35行目**: `@code` ブロックで RequestId の取得ロジックを実装

### プログラム呼び出し階層図

```
例外発生
    │
    └─ UseExceptionHandler ミドルウェア
           │
           └─ /Error へリダイレクト
                  │
                  └─ Error.razor
                         │
                         ├─ OnInitialized()
                         │      │
                         │      ├─ Activity.Current?.Id
                         │      │      └─ [null の場合]
                         │      │
                         │      └─ HttpContext?.TraceIdentifier
                         │
                         └─ ShowRequestId
                                └─ !string.IsNullOrEmpty(RequestId)
```

### データフロー図

```
[入力]                    [処理]                     [出力]

例外発生 ─────────────▶ ExceptionHandler
                              │
                              ▼
HttpContext ───────────▶ CascadingParameter
                              │
                              ▼
Activity.Current ──────▶ OnInitialized()
                              │
                              ▼
                        RequestId 設定 ─────────▶ 画面表示
                                                  "Request ID: xxx"
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Error.razor | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm/Components/Pages/Error.razor` | ソース | エラーページのメインコンポーネント |
| Program.cs | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm/Program.cs` | ソース | サーバーサイドエントリーポイント（例外ハンドラ設定） |
| MainLayout.razor | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm/Components/Layout/MainLayout.razor` | ソース | アプリケーションレイアウト |
| App.razor | `src/mono/wasm/testassets/BlazorWebWasm/BlazorWebWasm/Components/App.razor` | ソース | アプリケーションルートコンポーネント |
