# 通知設計書 13-FileSystemWatcher.Renamed

## 概要

本ドキュメントは、.NET の `System.IO.FileSystemWatcher` クラスが提供する `Renamed` イベントについて、その設計仕様と実装詳細を記述した通知設計書である。

### 本通知の処理概要

FileSystemWatcher.Renamed イベントは、監視対象のディレクトリ内のファイルまたはディレクトリの名前が変更された際に発生するイベント通知機能を提供する。

**業務上の目的・背景**：ファイル名の変更検知は、ファイル同期システムでのリネーム伝播、バージョン管理、ファイル追跡、監査ログの記録など、様々な業務シナリオで必要とされる。Renamed イベントは、ファイルやディレクトリの名前変更をリアルタイムに検知し、変更前後の名前情報を提供することで、正確な追跡と同期処理を可能にする。

**通知の送信タイミング**：監視対象ディレクトリ（Path プロパティで指定）内で、ファイルまたはディレクトリの名前が変更された瞬間にイベントが発生する。Linux では IN_MOVED_FROM と IN_MOVED_TO のペアとして検知される。IncludeSubdirectories が true の場合、サブディレクトリ内でのリネームも検知対象となる。

**通知の受信者**：FileSystemWatcher インスタンスの Renamed イベントにハンドラを登録したアプリケーションコード。SynchronizingObject プロパティが設定されている場合、指定されたスレッドコンテキストでハンドラが呼び出される。

**通知内容の概要**：RenamedEventArgs オブジェクトとして、変更種別（WatcherChangeTypes.Renamed）、新しい名前（Name）、新しいフルパス（FullPath）、古い名前（OldName）、古いフルパス（OldFullPath）が通知される。

**期待されるアクション**：受信者は通知を受け取り後、同期先でのファイル名変更、データベースやインデックスの更新、キャッシュキーの変更、監査ログへの記録などの業務処理を実行することが期待される。

## 通知種別

アプリ内通知（イベント通知）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（ThreadPool経由） |
| 優先度 | 中（通常のイベント配信） |
| リトライ | なし（イベントは一度のみ配信） |

### 送信先決定ロジック

Renamed イベントに登録された RenamedEventHandler デリゲートが送信先となる。複数のハンドラが登録されている場合、マルチキャストデリゲートとして順次呼び出される。SynchronizingObject が設定されている場合は BeginInvoke 経由で非同期呼び出しが行われる。

## 通知テンプレート

### イベント引数

| 項目 | 内容 |
|-----|------|
| イベント型 | RenamedEventArgs |
| sender | FileSystemWatcher インスタンス |
| ChangeType | WatcherChangeTypes.Renamed |

### イベントデータ構造

```csharp
public class RenamedEventArgs : FileSystemEventArgs
{
    public WatcherChangeTypes ChangeType { get; }  // Renamed
    public string FullPath { get; }                 // 新しいフルパス
    public string? Name { get; }                    // 新しい名前
    public string OldFullPath { get; }              // 古いフルパス
    public string? OldName { get; }                 // 古い名前
}
```

### 添付ファイル

該当なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| ChangeType | 変更種別 | WatcherChangeTypes.Renamed | Yes |
| FullPath | 新しいフルパス | 監視ディレクトリパス + 新しいファイル名 | Yes |
| Name | 新しいファイル/ディレクトリ名 | IN_MOVED_TO の name フィールド | No |
| OldFullPath | 古いフルパス | 監視ディレクトリパス + 古いファイル名 | Yes |
| OldName | 古いファイル/ディレクトリ名 | IN_MOVED_FROM の name フィールド | No |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| OS通知 | ファイル/ディレクトリ名変更 | EnableRaisingEvents == true かつ新旧名のいずれかが Filter にマッチ | 監視対象ディレクトリ内でのリネームを検知 |
| OS通知 | IN_MOVED_FROM + IN_MOVED_TO (Linux) | 同じ cookie 値のペア | Linux inotify による移動（リネーム）イベント |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| EnableRaisingEvents == false | イベント発生が無効化されている |
| Filter/Filters 不一致 | 新旧両方のファイル名がフィルタパターンにマッチしない |
| NotifyFilters 不一致 | NotifyFilters.FileName/DirectoryName が設定されていない |
| Disposed | FileSystemWatcher が破棄済み |
| 片方のイベントのみ | IN_MOVED_FROM のみ（タイムアウト）→ Deleted、IN_MOVED_TO のみ → Created として処理 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[OS: IN_MOVED_FROM 検知] --> B[cookie と名前を保存]
    B --> C{IN_MOVED_TO 到着?}
    C -->|2ms以内| D{同じ cookie?}
    C -->|タイムアウト| E[Deleted イベント生成]
    D -->|一致| F{フィルタチェック}
    D -->|不一致| G[個別イベント処理]
    F -->|新旧どちらかマッチ| H[RenamedEventArgs 生成]
    F -->|両方不一致| I[通知スキップ]
    H --> J{SynchronizingObject?}
    J -->|null| K[直接ハンドラ呼出]
    J -->|設定あり| L[BeginInvoke経由]
    K --> M[OnRenamed 実行]
    L --> M
    M --> N[終了]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（ファイルシステム通知のためデータベースを使用しない）

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| InternalBufferOverflowException | 内部バッファがオーバーフローした場合 | Error イベントで通知、バッファサイズ増加を検討 |
| ObjectDisposedException | 破棄後のアクセス | アプリケーション側でライフサイクル管理 |
| イベントペア不一致 | IN_MOVED_FROM/TO のペアが成立しない | Deleted/Created として個別に処理 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし（イベントはリトライされない） |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし（OSイベント依存） |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし（24時間監視可能）

## セキュリティ考慮事項

- ファイルパスにはファイルシステムのアクセス権限が適用される
- 監視対象ディレクトリへの読み取り権限が必要
- 新旧両方のパス情報が通知されるため、ファイル追跡が可能
- Linux では inotify インスタンス数とウォッチ数にユーザーごとの制限がある

## 備考

- Windows と Linux で内部実装が異なる（Windows: ReadDirectoryChangesW、Linux: inotify）
- Linux では IN_MOVED_FROM と IN_MOVED_TO がペアで発生し、cookie 値で関連付けられる
- ペアが成立しない場合（2ms タイムアウト）、個別のイベント（Deleted/Created）として処理される
- 監視対象外への移動は Deleted、監視対象外からの移動は Created として検知される

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、イベント通知で使用されるデータ構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | RenamedEventArgs.cs | `src/libraries/System.IO.FileSystem.Watcher/src/System/IO/RenamedEventArgs.cs` | リネームイベント引数。OldFullPath、OldName プロパティの追加 |
| 1-2 | FileSystemEventArgs.cs | `src/libraries/System.IO.FileSystem.Watcher/src/System/IO/FileSystemEventArgs.cs` | 基底クラス。ChangeType、FullPath、Name の実装 |

**読解のコツ**: RenamedEventArgs は FileSystemEventArgs を継承し、古い名前情報を追加している。Combine メソッドの共有に注目。

#### Step 2: エントリーポイントを理解する

処理の起点となる FileSystemWatcher クラスのイベント定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | FileSystemWatcher.cs | `src/libraries/System.IO.FileSystem.Watcher/src/System/IO/FileSystemWatcher.cs` | Renamed イベントの定義と OnRenamed メソッドの実装 |

**主要処理フロー**:
1. **行 318-328**: Renamed イベントの add/remove アクセサ定義
2. **行 517-528**: OnRenamed メソッドによるイベント発火
3. **行 410-418**: NotifyRenameEventArgs メソッドで新旧名のフィルタマッチング

**読解のコツ**: _onRenamedHandler フィールドがデリゲートチェーンを保持し、新旧どちらかの名前がフィルタにマッチすればイベントが発火する点に注目。

#### Step 3: Linux 実装を理解する

Linux 環境での inotify を使用したペアイベント処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | FileSystemWatcher.Linux.cs | `src/libraries/System.IO.FileSystem.Watcher/src/System/IO/FileSystemWatcher.Linux.cs` | IN_MOVED_FROM/TO のペア処理ロジック |
| 3-2 | Interop.INotify.cs | `src/libraries/Common/src/Interop/Linux/System.Native/Interop.INotify.cs` | IN_MOVED_FROM/TO の定数定義 |

**主要処理フロー**:
- **行 549-593 (Linux.cs)**: IN_MOVED_FROM イベントの保存と IN_MOVED_TO 待機
- **行 567 (Linux.cs)**: cookie 値による IN_MOVED_TO とのマッチング
- **行 604-631 (Linux.cs)**: IN_MOVED_FROM 受信時のタイムアウト待機（2ms）
- **行 693-701 (Linux.cs)**: IN_MOVED_TO 受信時の Renamed/Created 判定
- **行 968-969 (Linux.cs)**: WatcherEvent.Renamed ファクトリメソッド

### プログラム呼び出し階層図

```
FileSystemWatcher.Renamed イベント
    │
    ├─ OnRenamed(RenamedEventArgs e)
    │      └─ [SynchronizingObject判定]
    │             ├─ handler(this, e)
    │             └─ syncObj.BeginInvoke(handler, ...)
    │
    └─ NotifyRenameEventArgs(action, name, oldName)
           ├─ MatchPattern(name) || MatchPattern(oldName)
           └─ OnRenamed(new RenamedEventArgs(...))

[Linux ProcessEvent]
    │
    ├─ IN_MOVED_FROM 受信
    │      └─ cookie/name 保存、Poll で IN_MOVED_TO 待機
    │
    ├─ IN_MOVED_TO 受信
    │      └─ cookie マッチング
    │             ├─ [マッチ] WatcherEvent.Renamed 生成
    │             └─ [不一致] WatcherEvent.Created 生成
    │
    └─ タイムアウト
           └─ WatcherEvent.Deleted 生成
```

### データフロー図

```
[入力]                    [処理]                           [出力]

IN_MOVED_FROM       movedFromName/cookie 保存
    │                     │
    ▼                     ▼
IN_MOVED_TO  ────▶  cookie マッチング  ────▶  RenamedEventHandler
                          │                     デリゲート呼び出し
                    [マッチ成功]
                          │
                          ▼
                    WatcherEvent.Renamed
                          │
                          ▼
                    EmitEvent()
                          │
                          ▼
                    NotifyRenameEventArgs()
                          │
                          ▼
                    OnRenamed()
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| FileSystemWatcher.cs | `src/libraries/System.IO.FileSystem.Watcher/src/System/IO/FileSystemWatcher.cs` | ソース | メインクラス、Renamed イベント定義 |
| FileSystemWatcher.Linux.cs | `src/libraries/System.IO.FileSystem.Watcher/src/System/IO/FileSystemWatcher.Linux.cs` | ソース | Linux 固有実装（inotify ペア処理） |
| RenamedEventArgs.cs | `src/libraries/System.IO.FileSystem.Watcher/src/System/IO/RenamedEventArgs.cs` | ソース | リネームイベント引数クラス |
| FileSystemEventArgs.cs | `src/libraries/System.IO.FileSystem.Watcher/src/System/IO/FileSystemEventArgs.cs` | ソース | 基底イベント引数クラス |
| Interop.INotify.cs | `src/libraries/Common/src/Interop/Linux/System.Native/Interop.INotify.cs` | ソース | IN_MOVED_FROM/TO 定数定義 |
