# 通知設計書 16-Interop.Sys.INotify (IN_MODIFY)

## 概要

本ドキュメントは、Linux の inotify サブシステムが提供する `IN_MODIFY` 通知について、.NET ランタイムでの実装詳細と設計仕様を記述した通知設計書である。

### 本通知の処理概要

IN_MODIFY は、ファイルの内容が変更（書き込み）された際に Linux カーネルの inotify サブシステムが生成する通知イベントである。.NET の FileSystemWatcher では、この通知を Changed イベントに変換してアプリケーションに通知する。

**業務上の目的・背景**：ファイル内容の変更検知は、設定ファイルの自動リロード、ログファイルの監視、データファイルの同期、コンパイル/ビルドトリガーなど、多くの業務シナリオで中核的な役割を果たす。IN_MODIFY は NotifyFilters で LastAccess、LastWrite、Security、Size のいずれかが設定されている場合に有効となり、ファイル内容の変更を検知するために使用される。

**通知の送信タイミング**：監視対象ディレクトリ内のファイルが write() システムコールで変更された際に、カーネルが inotify イベントを生成する。このイベントは .NET の FileSystemWatcher.Linux 実装で処理され、Changed イベントとしてアプリケーションに通知される。

**通知の受信者**：FileSystemWatcher インスタンスの Changed イベントにハンドラを登録したアプリケーションコード。NotifyFilters に LastAccess、LastWrite、Security、Size のいずれかが含まれている必要がある。

**通知内容の概要**：inotify イベントとして、ウォッチ記述子（wd）、イベントマスク（IN_MODIFY）、cookie（通常0）、ファイル名（name）が通知される。FileSystemWatcher は WatcherChangeTypes.Changed としてアプリケーションに伝達する。

**期待されるアクション**：受信者は通知を受け取り後、設定のリロード、ファイル同期処理の開始、ビルドのトリガー、変更ログの記録などの処理を実行することが期待される。

## 通知種別

OSカーネル通知（inotify イベント）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（カーネルからの割り込み） |
| 優先度 | 通常（他の inotify イベントと同等） |
| リトライ | なし（カーネルは一度のみ通知） |

### 送信先決定ロジック

inotify_add_watch() で登録されたウォッチ記述子に対応する inotify ファイルディスクリプタに対してイベントが送信される。.NET では INotify クラスが全ての FileSystemWatcher で共有する単一の inotify インスタンスを管理する。

## 通知テンプレート

### inotify イベント構造

| 項目 | 内容 |
|-----|------|
| イベント型 | struct inotify_event |
| マスク値 | IN_MODIFY (0x00000002) |
| cookie | 0（ペアイベントではない） |

### イベントデータ構造

```c
struct inotify_event {
    int      wd;       // ウォッチ記述子
    uint32_t mask;     // IN_MODIFY = 0x00000002
    uint32_t cookie;   // 0
    uint32_t len;      // name の長さ
    char     name[];   // ファイル名（NULL終端）
};
```

### .NET での変換後

```csharp
// IN_MODIFY は Changed イベントに変換される
FileSystemEventArgs {
    ChangeType = WatcherChangeTypes.Changed,
    FullPath = "監視ディレクトリ/ファイル名",
    Name = "ファイル名"
}
```

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| wd | ウォッチ記述子 | inotify_add_watch() の戻り値 | Yes |
| mask | イベントマスク | IN_MODIFY (0x00000002) | Yes |
| cookie | イベント関連付け | 常に 0 | Yes |
| len | 名前長 | ファイル名のバイト長 + パディング | Yes |
| name | ファイル名 | 対象ファイルの名前 | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| システムコール | write() | ウォッチにIN_MODIFYが設定されている | ファイル書き込み時 |
| システムコール | truncate() | ウォッチにIN_MODIFYが設定されている | ファイルサイズ変更時 |
| システムコール | mmap() 書き込み | ウォッチにIN_MODIFYが設定されている | メモリマップ経由の書き込み |

### .NET での有効化条件

| 条件 | 説明 |
|-----|------|
| NotifyFilters.LastAccess | このフィルタで IN_MODIFY が有効化 |
| NotifyFilters.LastWrite | このフィルタで IN_MODIFY が有効化 |
| NotifyFilters.Security | このフィルタで IN_MODIFY が有効化 |
| NotifyFilters.Size | このフィルタで IN_MODIFY が有効化 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| NotifyFilters に該当フィルタなし | 上記4つのいずれも設定されていない場合 |
| ディレクトリへの変更 | ディレクトリは IN_MODIFY を生成しない |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[ファイル write() 実行] --> B[カーネル: inotify イベント生成]
    B --> C[inotify キューに追加]
    C --> D[.NET: ProcessEvents スレッド]
    D --> E[TryReadEvent でイベント読み取り]
    E --> F{mask & IN_MODIFY?}
    F -->|Yes| G[WatcherEvent.Changed 生成]
    F -->|No| H[他のイベント処理]
    G --> I[_eventQueue に追加]
    I --> J[EmitEvent]
    J --> K[fsw.NotifyFileSystemEventArgs]
    K --> L[OnChanged 呼び出し]
    L --> M[終了]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（OSカーネル通知のためデータベースを使用しない）

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| IN_Q_OVERFLOW | inotify イベントキューが溢れた | Error イベント発生、監視再起動 |
| EBADF | 無効なファイルディスクリプタ | 監視を停止 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし（カーネルイベントはリトライされない） |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | カーネル設定依存（/proc/sys/fs/inotify/max_queued_events） |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし（24時間監視可能）

## セキュリティ考慮事項

- IN_MODIFY はファイル変更パターンを明らかにするため、機密性の高い環境では監視ログの取り扱いに注意
- 監視対象ディレクトリへの読み取り権限が必要
- ファイル内容自体は通知に含まれない（変更があったことのみ通知）

## 備考

- IN_MODIFY は最も頻繁に使用される inotify イベントの一つ
- 大量のファイル書き込みでイベントキューがオーバーフローする可能性がある
- 単一の write() 呼び出しで複数の IN_MODIFY イベントが発生する場合がある
- ファイルの fsync() では IN_MODIFY は発生しない（データは既に書き込まれている）

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、IN_MODIFY の定義と関連する定数を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Interop.INotify.cs | `src/libraries/Common/src/Interop/Linux/System.Native/Interop.INotify.cs` | IN_MODIFY 定数の定義（0x00000002） |

**読解のコツ**: NotifyEvents enum で IN_MODIFY = 0x00000002 として定義されている点に注目。

#### Step 2: フィルタマッピングを理解する

NotifyFilters から inotify イベントへの変換ロジックを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | FileSystemWatcher.Linux.cs | `src/libraries/System.IO.FileSystem.Watcher/src/System/IO/FileSystemWatcher.Linux.cs` | TranslateFilters メソッドでの IN_MODIFY マッピング |

**主要処理フロー**:
- **行 1263-1267 (Linux.cs)**: filtersForModify に LastAccess、LastWrite、Security、Size が含まれる
- **行 1279-1282 (Linux.cs)**: IN_MODIFY をサブスクライブする条件判定

#### Step 3: イベント処理を理解する

IN_MODIFY イベントの処理フローを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | FileSystemWatcher.Linux.cs | `src/libraries/System.IO.FileSystem.Watcher/src/System/IO/FileSystemWatcher.Linux.cs` | ProcessEvent での IN_MODIFY 処理 |

**主要処理フロー**:
- **行 527-529 (Linux.cs)**: EventMask に IN_MODIFY が含まれる定義
- **行 685-688 (Linux.cs)**: IN_MODIFY イベントで WatcherEvent.Changed を生成

### プログラム呼び出し階層図

```
[Linux カーネル]
inotify イベント (IN_MODIFY)
    │
    ▼
[.NET FileSystemWatcher.Linux]
INotify.ProcessEvents()
    │
    ├─ TryReadEvent() → NotifyEvent 構造体
    │
    └─ ProcessEvent()
           │
           ├─ mask & IN_MODIFY 判定
           │
           └─ WatcherEvent.Changed(dir, name)
                  │
                  ▼
           Watcher.QueueEvent()
                  │
                  ▼
           EmitEvent()
                  │
                  ▼
           fsw.NotifyFileSystemEventArgs(Changed, name)
                  │
                  ▼
           OnChanged()
```

### データフロー図

```
[入力]                    [処理]                           [出力]

ファイル write()        カーネル inotify
システムコール  ───▶    イベント生成  ───▶   inotify キュー
                              │
                              ▼
                        .NET read() ───▶   NotifyEvent 構造体
                              │
                              ▼
                        ProcessEvent()
                              │
                              ▼
                        WatcherEvent.Changed
                              │
                              ▼
                        FileSystemEventArgs ───▶  Changed イベント
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Interop.INotify.cs | `src/libraries/Common/src/Interop/Linux/System.Native/Interop.INotify.cs` | ソース | IN_MODIFY 定数定義 |
| FileSystemWatcher.Linux.cs | `src/libraries/System.IO.FileSystem.Watcher/src/System/IO/FileSystemWatcher.Linux.cs` | ソース | IN_MODIFY イベント処理、TranslateFilters |
| FileSystemWatcher.cs | `src/libraries/System.IO.FileSystem.Watcher/src/System/IO/FileSystemWatcher.cs` | ソース | NotifyFilters、Changed イベント定義 |
