# 通知設計書 6-INotifyCompletion.OnCompleted

## 概要

本ドキュメントは、.NET Runtimeにおける`INotifyCompletion.OnCompleted`メソッドの設計を記述する。このメソッドは、非同期操作が完了した際に継続処理（continuation）をスケジュールするためのインターフェースメソッドである。

### 本通知の処理概要

`INotifyCompletion.OnCompleted`メソッドは、async/awaitパターンの基盤となるインターフェースメソッドである。awaiter（待機オブジェクト）が完了した際に、指定された継続処理（Action）を呼び出すようスケジュールする。C#コンパイラが生成するステートマシンコードからこのメソッドが呼び出される。

**業務上の目的・背景**：C# 5.0で導入されたasync/awaitパターンは、非同期プログラミングを大幅に簡素化した。このパターンを実現するためには、非同期操作の完了を待機し、完了時に継続処理を実行するメカニズムが必要である。`INotifyCompletion`インターフェースは、このメカニズムの標準的なコントラクトを定義する。カスタムのawaitableパターンを実装する際にも使用される。

**通知の送信タイミング**：awaitされた操作が完了した時点で、登録された継続処理が実行される。操作が既に完了している場合、継続処理は同期的に（または非同期的に）即座に実行される。

**通知の受信者**：登録された`Action`デリゲート（継続処理）が受信者となる。通常はコンパイラ生成のステートマシンのMoveNextメソッド。

**通知内容の概要**：継続処理自体には引数が渡されない。awaiterの`GetResult()`メソッドを呼び出すことで、非同期操作の結果を取得する。

**期待されるアクション**：継続処理（MoveNext）が実行され、ステートマシンが次の状態に遷移する。

## 通知種別

コールバック/継続通知（アプリ内通知）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（継続処理はキャプチャされたコンテキストで実行） |
| 優先度 | 高（非同期処理の継続に必須） |
| リトライ | なし |

### 送信先決定ロジック

`OnCompleted`メソッドに渡された`Action`デリゲートが唯一の継続処理として登録される。複数の継続処理は通常サポートされない（1回のawaitに対して1つの継続）。

## 通知テンプレート

### インターフェース定義

```csharp
namespace System.Runtime.CompilerServices
{
    public interface INotifyCompletion
    {
        void OnCompleted(Action continuation);
    }
}
```

### メソッドシグネチャ

| 項目 | 内容 |
|-----|------|
| メソッド名 | OnCompleted |
| 引数 | `Action continuation` - 完了時に実行する継続処理 |
| 戻り値 | void |
| 例外 | `ArgumentNullException` - continuationがnullの場合 |

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| continuation | 完了時に実行されるActionデリゲート | コンパイラ生成コード | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| プログラム実行 | 非同期操作の完了 | 操作が完了した場合 | Task完了、I/O完了など |
| プログラム実行 | 既に完了済み | IsCompletedがtrue | 同期的に継続を実行 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| なし | 継続は常に1回だけ実行される |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[await式の評価] --> B[awaiter.IsCompleted確認]
    B -->|true| C[awaiter.GetResult呼び出し]
    B -->|false| D[awaiter.OnCompleted呼び出し]
    D --> E[継続処理を登録]
    E --> F[非同期操作の実行]
    F --> G{操作完了?}
    G -->|No| F
    G -->|Yes| H[ExecutionContextをキャプチャ]
    H --> I[継続処理を実行]
    I --> J[awaiter.GetResult呼び出し]
    C --> K[await式の評価完了]
    J --> K
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（メモリ内コールバック）

### 更新テーブル一覧

該当なし（メモリ内コールバック）

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| ArgumentNullException | continuationがnullの場合 | nullチェックを行う |
| 継続処理内例外 | 継続処理の実行中に例外が発生した場合 | 例外は呼び出し元に伝播 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし（制限なし） |
| 1日あたり上限 | なし（制限なし） |

### 配信時間帯

制限なし

## セキュリティ考慮事項

- `OnCompleted`はExecutionContextを伝播するため、セキュリティコンテキストが維持される
- セキュリティクリティカルなシナリオでは`ICriticalNotifyCompletion.UnsafeOnCompleted`を使用
- 継続処理には適切なスレッドコンテキストでの実行が保証される（SynchronizationContext経由）

## 備考

- `System.Runtime.CompilerServices`名前空間に定義
- C#コンパイラのasync/awaitステートマシン生成で使用される
- `TaskAwaiter`、`ValueTaskAwaiter`等の標準awaiterがこのインターフェースを実装
- `ICriticalNotifyCompletion`は`INotifyCompletion`を継承し、`UnsafeOnCompleted`を追加
- カスタムawaitableパターンを実装する際に必要なインターフェース

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: インターフェース定義を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | INotifyCompletion.cs | `src/libraries/System.Private.CoreLib/src/System/Runtime/CompilerServices/INotifyCompletion.cs` | インターフェース定義（行13-22） |

**読解のコツ**: `OnCompleted`メソッドのセマンティクスをXMLドキュメントコメントで確認。継続処理を「スケジュールする」という点が重要。

**主要処理フロー**:
1. **行13-16**: XMLドキュメント - 操作完了時に継続をスケジュールする
2. **行17-22**: `INotifyCompletion`インターフェースと`OnCompleted`メソッドの定義

#### Step 2: TaskAwaiterでの実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | TaskAwaiter.cs | `src/libraries/System.Private.CoreLib/src/System/Runtime/CompilerServices/TaskAwaiter.cs` | OnCompletedの実装（行46-49） |

**主要処理フロー**:
- **行20**: `TaskAwaiter`は`ICriticalNotifyCompletion`を実装（`INotifyCompletion`を継承）
- **行46-49**: `OnCompleted`メソッド実装 - `OnCompletedInternal`を呼び出し
- **行174-187**: `OnCompletedInternal`静的メソッド - 継続処理をタスクに設定

#### Step 3: 内部実装の詳細を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | TaskAwaiter.cs | `src/libraries/System.Private.CoreLib/src/System/Runtime/CompilerServices/TaskAwaiter.cs` | OnCompletedInternal（行174-187） |

**主要処理フロー**:
- **行176**: nullチェック
- **行180-183**: ETWイベントが有効な場合の継続ラッピング
- **行186**: `task.SetContinuationForAwait`で継続を設定

### プログラム呼び出し階層図

```
await someTask;
    │
    └─ コンパイラ生成コード
           │
           ├─ awaiter = someTask.GetAwaiter()
           │
           ├─ if (!awaiter.IsCompleted)
           │      │
           │      └─ awaiter.OnCompleted(MoveNextAction)
           │             │
           │             └─ TaskAwaiter.OnCompletedInternal(task, continuation, true, true)
           │                    │
           │                    ├─ ArgumentNullException.ThrowIfNull(continuation)
           │                    │
           │                    ├─ [ETWイベント処理]
           │                    │
           │                    └─ task.SetContinuationForAwait(continuation, ...)
           │
           └─ result = awaiter.GetResult()
```

### データフロー図

```
[入力]                     [処理]                          [出力]

await式 ───▶ GetAwaiter() ───▶ IsCompleted確認
    │              │                 │
    │              ▼                 ▼
    │         TaskAwaiter      false: OnCompleted()
    │              │                 │
    │              │                 ▼
    │              │          継続処理の登録
    │              │                 │
    │              │                 ▼
    │              │         非同期操作の完了待ち
    │              │                 │
    ▼              ▼                 ▼
Action        タスク完了 ───▶ 継続処理の実行
(MoveNext)         │                 │
                   ▼                 ▼
              GetResult() ───▶ 結果の取得
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| INotifyCompletion.cs | `src/libraries/System.Private.CoreLib/src/System/Runtime/CompilerServices/INotifyCompletion.cs` | ソース | インターフェース定義 |
| TaskAwaiter.cs | `src/libraries/System.Private.CoreLib/src/System/Runtime/CompilerServices/TaskAwaiter.cs` | ソース | Task用awaiter実装 |
| ValueTaskAwaiter.cs | `src/libraries/System.Private.CoreLib/src/System/Runtime/CompilerServices/ValueTaskAwaiter.cs` | ソース | ValueTask用awaiter実装 |
| AsyncTaskMethodBuilder.cs | `src/libraries/System.Private.CoreLib/src/System/Runtime/CompilerServices/AsyncTaskMethodBuilder.cs` | ソース | asyncメソッドビルダー |
| YieldAwaitable.cs | `src/libraries/System.Private.CoreLib/src/System/Runtime/CompilerServices/YieldAwaitable.cs` | ソース | Task.Yield用awaiter |
