# API設計書

## 概要

本ドキュメントは、eShopアプリケーションのREST API及びgRPC APIの設計仕様を定義します。このシステムはマイクロサービスアーキテクチャを採用しており、以下の4つの主要なAPIサービスで構成されています：

- **Catalog API**: 商品カタログの管理（REST API）
- **Ordering API**: 注文管理（REST API、認証必須）
- **Webhooks API**: Webhook購読管理（REST API、認証必須）
- **Basket API**: 買い物かご管理（gRPC API）

## 共通仕様

### ベースURL

```
# 各サービスのベースURL
Catalog API:   https://{host}/api/catalog
Ordering API:  https://{host}/api/orders
Webhooks API:  https://{host}/api/webhooks
Basket API:    grpc://{host}  (gRPC)
```

### 認証方式

- **Catalog API**: 認証不要（公開API）
- **Ordering API**: Bearer Token認証（OIDC/OAuth2.0）
- **Webhooks API**: Bearer Token認証（OIDC/OAuth2.0）
- **Basket API**: gRPC認証（ユーザーコンテキストから取得）

### 共通ヘッダー

| ヘッダー名 | 必須 | 説明 |
| --- | --- | --- |
| Authorization | 条件付き | Bearer Token（認証が必要なAPIのみ） |
| Content-Type | ○ | application/json |
| Accept | - | application/json |
| x-requestid | 条件付き | リクエストID（Ordering APIの一部で必須） |

### APIバージョニング

本システムはAPIバージョニングをサポートしています。バージョンはURLパスまたはクエリパラメータで指定できます。

- V1: `api-version=1.0`
- V2: `api-version=2.0`

### 共通エラーレスポンス

| ステータスコード | 説明 |
| --- | --- |
| 400 | Bad Request - リクエストパラメータ不正 |
| 401 | Unauthorized - 認証エラー |
| 403 | Forbidden - 権限エラー |
| 404 | Not Found - リソース未検出 |
| 422 | Unprocessable Entity - バリデーションエラー |
| 500 | Internal Server Error - サーバーエラー |

**エラーレスポンス形式（ProblemDetails）**

```json
{
  "type": "https://tools.ietf.org/html/rfc7231#section-6.5.1",
  "title": "Bad Request",
  "status": 400,
  "detail": "エラーの詳細説明"
}
```

## API一覧

### Catalog API

| カテゴリ | エンドポイント | メソッド | 説明 | バージョン |
| --- | --- | --- | --- | --- |
| Items | /api/catalog/items | GET | カタログアイテム一覧取得 | V1, V2 |
| Items | /api/catalog/items/by | GET | 複数ID指定でアイテム取得 | V1, V2 |
| Items | /api/catalog/items/{id} | GET | 単一アイテム取得 | V1, V2 |
| Items | /api/catalog/items/by/{name} | GET | 名前でアイテム検索 | V1 |
| Items | /api/catalog/items/{id}/pic | GET | アイテム画像取得 | V1, V2 |
| Search | /api/catalog/items/withsemanticrelevance/{text} | GET | セマンティック検索 | V1 |
| Search | /api/catalog/items/withsemanticrelevance | GET | セマンティック検索 | V2 |
| Types | /api/catalog/items/type/{typeId}/brand/{brandId?} | GET | タイプ・ブランドでフィルタ | V1 |
| Brands | /api/catalog/items/type/all/brand/{brandId?} | GET | ブランドでフィルタ | V1 |
| Types | /api/catalog/catalogtypes | GET | カタログタイプ一覧 | V1, V2 |
| Brands | /api/catalog/catalogbrands | GET | カタログブランド一覧 | V1, V2 |
| Items | /api/catalog/items | PUT | アイテム更新（V1） | V1 |
| Items | /api/catalog/items/{id} | PUT | アイテム更新（V2） | V2 |
| Items | /api/catalog/items | POST | アイテム作成 | V1, V2 |
| Items | /api/catalog/items/{id} | DELETE | アイテム削除 | V1, V2 |

### Ordering API

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| Orders | /api/orders | GET | ユーザーの注文一覧取得 |
| Orders | /api/orders/{orderId} | GET | 注文詳細取得 |
| Orders | /api/orders | POST | 注文作成 |
| Orders | /api/orders/draft | POST | 注文ドラフト作成 |
| Orders | /api/orders/cancel | PUT | 注文キャンセル |
| Orders | /api/orders/ship | PUT | 注文出荷 |
| CardTypes | /api/orders/cardtypes | GET | カードタイプ一覧取得 |

### Webhooks API

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| Webhooks | /api/webhooks | GET | Webhook購読一覧取得 |
| Webhooks | /api/webhooks/{id} | GET | Webhook購読詳細取得 |
| Webhooks | /api/webhooks | POST | Webhook購読作成 |
| Webhooks | /api/webhooks/{id} | DELETE | Webhook購読削除 |

### Basket API (gRPC)

| サービス | メソッド | 説明 |
| --- | --- | --- |
| Basket | GetBasket | 買い物かご取得 |
| Basket | UpdateBasket | 買い物かご更新 |
| Basket | DeleteBasket | 買い物かご削除 |

---

## 各APIエンドポイント定義

### Catalog API

#### 1. カタログアイテム一覧取得（V1）

カタログアイテムの一覧をページネーション付きで取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/catalog/items` |
| 認証 | 不要 |
| バージョン | V1 |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| pageSize | int | - | 10 | 1ページあたりのアイテム数 |
| pageIndex | int | - | 0 | ページインデックス |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "pageIndex": 0,
  "pageSize": 10,
  "count": 100,
  "data": [
    {
      "id": 1,
      "name": ".NET Bot Black Hoodie",
      "description": "A nice hoodie with .NET Bot",
      "price": 19.50,
      "pictureFileName": "1.png",
      "catalogTypeId": 2,
      "catalogType": null,
      "catalogBrandId": 2,
      "catalogBrand": null,
      "availableStock": 100,
      "restockThreshold": 10,
      "maxStockThreshold": 200,
      "onReorder": false
    }
  ]
}
```

---

#### 2. カタログアイテム一覧取得（V2）

カタログアイテムの一覧をフィルタ付きで取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/catalog/items` |
| 認証 | 不要 |
| バージョン | V2 |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| pageSize | int | - | 10 | 1ページあたりのアイテム数 |
| pageIndex | int | - | 0 | ページインデックス |
| name | string | - | - | アイテム名でフィルタ |
| type | int | - | - | カタログタイプIDでフィルタ |
| brand | int | - | - | カタログブランドIDでフィルタ |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "pageIndex": 0,
  "pageSize": 10,
  "count": 50,
  "data": [
    {
      "id": 1,
      "name": ".NET Bot Black Hoodie",
      "description": "A nice hoodie with .NET Bot",
      "price": 19.50,
      "pictureFileName": "1.png",
      "catalogTypeId": 2,
      "catalogType": null,
      "catalogBrandId": 2,
      "catalogBrand": null,
      "availableStock": 100,
      "restockThreshold": 10,
      "maxStockThreshold": 200,
      "onReorder": false
    }
  ]
}
```

---

#### 3. 複数ID指定でアイテム取得

指定したIDリストに該当するカタログアイテムを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/catalog/items/by` |
| 認証 | 不要 |
| バージョン | V1, V2 |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| ids | int[] | ○ | - | 取得するアイテムIDのリスト |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "id": 1,
    "name": ".NET Bot Black Hoodie",
    "description": "A nice hoodie with .NET Bot",
    "price": 19.50,
    "pictureFileName": "1.png",
    "catalogTypeId": 2,
    "catalogType": null,
    "catalogBrandId": 2,
    "catalogBrand": null,
    "availableStock": 100,
    "restockThreshold": 10,
    "maxStockThreshold": 200,
    "onReorder": false
  },
  {
    "id": 2,
    "name": ".NET Black & White Mug",
    "description": "A mug with .NET logo",
    "price": 8.50,
    "pictureFileName": "2.png",
    "catalogTypeId": 1,
    "catalogType": null,
    "catalogBrandId": 2,
    "catalogBrand": null,
    "availableStock": 89,
    "restockThreshold": 5,
    "maxStockThreshold": 100,
    "onReorder": true
  }
]
```

---

#### 4. 単一アイテム取得

指定したIDのカタログアイテムを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/catalog/items/{id}` |
| 認証 | 不要 |
| バージョン | V1, V2 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | int | ○ | カタログアイテムID（1以上の整数） |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "id": 1,
  "name": ".NET Bot Black Hoodie",
  "description": "A nice hoodie with .NET Bot",
  "price": 19.50,
  "pictureFileName": "1.png",
  "catalogTypeId": 2,
  "catalogType": {
    "id": 2,
    "type": "T-Shirt"
  },
  "catalogBrandId": 2,
  "catalogBrand": {
    "id": 2,
    "brand": ".NET"
  },
  "availableStock": 100,
  "restockThreshold": 10,
  "maxStockThreshold": 200,
  "onReorder": false
}
```

**レスポンス（エラー時）**

ステータスコード: `400 Bad Request`

```json
{
  "detail": "Id is not valid"
}
```

ステータスコード: `404 Not Found`

---

#### 5. アイテム画像取得

指定したIDのカタログアイテムの画像を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/catalog/items/{id}/pic` |
| 認証 | 不要 |
| バージョン | V1, V2 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | int | ○ | カタログアイテムID |

**レスポンス（成功時）**

ステータスコード: `200 OK`

Content-Type: `image/png`, `image/jpeg`, `image/gif`, `image/bmp`, `image/webp`, `image/svg+xml` など

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`

---

#### 6. 名前でアイテム検索

指定した名前で始まるカタログアイテムを検索します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/catalog/items/by/{name}` |
| 認証 | 不要 |
| バージョン | V1 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| name | string | ○ | 検索する名前（最低1文字以上） |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| pageSize | int | - | 10 | 1ページあたりのアイテム数 |
| pageIndex | int | - | 0 | ページインデックス |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "pageIndex": 0,
  "pageSize": 10,
  "count": 5,
  "data": [
    {
      "id": 1,
      "name": ".NET Bot Black Hoodie",
      "description": "A nice hoodie with .NET Bot",
      "price": 19.50,
      "pictureFileName": "1.png",
      "catalogTypeId": 2,
      "catalogBrandId": 2,
      "availableStock": 100,
      "restockThreshold": 10,
      "maxStockThreshold": 200,
      "onReorder": false
    }
  ]
}
```

---

#### 7. セマンティック検索（V1）

AIを使用してセマンティック関連性の高いカタログアイテムを検索します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/catalog/items/withsemanticrelevance/{text}` |
| 認証 | 不要 |
| バージョン | V1 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| text | string | ○ | 検索テキスト（最低1文字以上） |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| pageSize | int | - | 10 | 1ページあたりのアイテム数 |
| pageIndex | int | - | 0 | ページインデックス |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "pageIndex": 0,
  "pageSize": 10,
  "count": 100,
  "data": [
    {
      "id": 1,
      "name": ".NET Bot Black Hoodie",
      "description": "A nice hoodie with .NET Bot",
      "price": 19.50,
      "pictureFileName": "1.png",
      "catalogTypeId": 2,
      "catalogBrandId": 2,
      "availableStock": 100,
      "restockThreshold": 10,
      "maxStockThreshold": 200,
      "onReorder": false
    }
  ]
}
```

---

#### 8. セマンティック検索（V2）

AIを使用してセマンティック関連性の高いカタログアイテムを検索します（クエリパラメータ版）。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/catalog/items/withsemanticrelevance` |
| 認証 | 不要 |
| バージョン | V2 |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| text | string | ○ | - | 検索テキスト（最低1文字以上） |
| pageSize | int | - | 10 | 1ページあたりのアイテム数 |
| pageIndex | int | - | 0 | ページインデックス |

**レスポンス（成功時）**

ステータスコード: `200 OK`

（レスポンス形式はV1と同様）

---

#### 9. タイプ・ブランドでフィルタ

指定したタイプとブランドのカタログアイテムを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/catalog/items/type/{typeId}/brand/{brandId?}` |
| 認証 | 不要 |
| バージョン | V1 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| typeId | int | ○ | カタログタイプID |
| brandId | int | - | カタログブランドID（オプション） |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| pageSize | int | - | 10 | 1ページあたりのアイテム数 |
| pageIndex | int | - | 0 | ページインデックス |

**レスポンス（成功時）**

ステータスコード: `200 OK`

（レスポンス形式はアイテム一覧と同様）

---

#### 10. ブランドでフィルタ

指定したブランドのカタログアイテムを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/catalog/items/type/all/brand/{brandId?}` |
| 認証 | 不要 |
| バージョン | V1 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| brandId | int | - | カタログブランドID（オプション） |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| pageSize | int | - | 10 | 1ページあたりのアイテム数 |
| pageIndex | int | - | 0 | ページインデックス |

**レスポンス（成功時）**

ステータスコード: `200 OK`

（レスポンス形式はアイテム一覧と同様）

---

#### 11. カタログタイプ一覧

カタログタイプの一覧を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/catalog/catalogtypes` |
| 認証 | 不要 |
| バージョン | V1, V2 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "id": 1,
    "type": "Mug"
  },
  {
    "id": 2,
    "type": "T-Shirt"
  },
  {
    "id": 3,
    "type": "Sheet"
  },
  {
    "id": 4,
    "type": "USB Memory Stick"
  }
]
```

---

#### 12. カタログブランド一覧

カタログブランドの一覧を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/catalog/catalogbrands` |
| 認証 | 不要 |
| バージョン | V1, V2 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "id": 1,
    "brand": "Azure"
  },
  {
    "id": 2,
    "brand": ".NET"
  },
  {
    "id": 3,
    "brand": "Visual Studio"
  },
  {
    "id": 4,
    "brand": "SQL Server"
  },
  {
    "id": 5,
    "brand": "Other"
  }
]
```

---

#### 13. アイテム更新（V1）

カタログアイテムを更新します。価格が変更された場合、IntegrationEventが発行されます。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /api/catalog/items` |
| 認証 | 不要 |
| バージョン | V1 |

**リクエストボディ**

```json
{
  "id": 1,
  "name": ".NET Bot Black Hoodie Updated",
  "description": "An updated description",
  "price": 24.99,
  "pictureFileName": "1.png",
  "catalogTypeId": 2,
  "catalogBrandId": 2,
  "availableStock": 150,
  "restockThreshold": 10,
  "maxStockThreshold": 200
}
```

| フィールド名 | 型 | 必須 | バリデーション | 説明 |
| --- | --- | --- | --- | --- |
| id | int | ○ | - | カタログアイテムID |
| name | string | ○ | Required | アイテム名 |
| description | string | - | - | 説明 |
| price | decimal | ○ | - | 価格 |
| pictureFileName | string | - | - | 画像ファイル名 |
| catalogTypeId | int | ○ | - | カタログタイプID |
| catalogBrandId | int | ○ | - | カタログブランドID |
| availableStock | int | ○ | - | 在庫数 |
| restockThreshold | int | ○ | - | 再発注閾値 |
| maxStockThreshold | int | ○ | - | 最大在庫閾値 |

**レスポンス（成功時）**

ステータスコード: `201 Created`

Location: `/api/catalog/items/{id}`

**レスポンス（エラー時）**

ステータスコード: `400 Bad Request`

```json
{
  "detail": "Item id must be provided in the request body."
}
```

ステータスコード: `404 Not Found`

```json
{
  "detail": "Item with id 999 not found."
}
```

---

#### 14. アイテム更新（V2）

カタログアイテムを更新します（パスパラメータでID指定）。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /api/catalog/items/{id}` |
| 認証 | 不要 |
| バージョン | V2 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | int | ○ | カタログアイテムID |

**リクエストボディ**

（V1と同様）

**レスポンス（成功時）**

ステータスコード: `201 Created`

Location: `/api/catalog/items/{id}`

---

#### 15. アイテム作成

新しいカタログアイテムを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /api/catalog/items` |
| 認証 | 不要 |
| バージョン | V1, V2 |

**リクエストボディ**

```json
{
  "name": "New Product",
  "description": "A brand new product",
  "price": 29.99,
  "pictureFileName": "new.png",
  "catalogTypeId": 1,
  "catalogBrandId": 2,
  "availableStock": 50,
  "restockThreshold": 5,
  "maxStockThreshold": 100
}
```

| フィールド名 | 型 | 必須 | バリデーション | 説明 |
| --- | --- | --- | --- | --- |
| name | string | ○ | Required | アイテム名 |
| description | string | - | - | 説明 |
| price | decimal | ○ | - | 価格 |
| pictureFileName | string | - | - | 画像ファイル名 |
| catalogTypeId | int | ○ | - | カタログタイプID |
| catalogBrandId | int | ○ | - | カタログブランドID |
| availableStock | int | ○ | - | 在庫数 |
| restockThreshold | int | ○ | - | 再発注閾値 |
| maxStockThreshold | int | ○ | - | 最大在庫閾値 |

**レスポンス（成功時）**

ステータスコード: `201 Created`

Location: `/api/catalog/items/{id}`

---

#### 16. アイテム削除

指定したIDのカタログアイテムを削除します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `DELETE /api/catalog/items/{id}` |
| 認証 | 不要 |
| バージョン | V1, V2 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | int | ○ | カタログアイテムID |

**レスポンス（成功時）**

ステータスコード: `204 No Content`

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`

---

### Ordering API

#### 1. ユーザーの注文一覧取得

認証されたユーザーの注文一覧を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/orders` |
| 認証 | 必要 |
| バージョン | V1 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "orderNumber": 1,
    "date": "2024-01-15T10:30:00Z",
    "status": "Submitted",
    "total": 49.99
  },
  {
    "orderNumber": 2,
    "date": "2024-01-16T14:20:00Z",
    "status": "Shipped",
    "total": 129.99
  }
]
```

---

#### 2. 注文詳細取得

指定した注文番号の詳細を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/orders/{orderId}` |
| 認証 | 必要 |
| バージョン | V1 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| orderId | int | ○ | 注文番号 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "orderNumber": 1,
  "date": "2024-01-15T10:30:00Z",
  "status": "Submitted",
  "description": "Order submitted",
  "street": "123 Main St",
  "city": "Seattle",
  "state": "WA",
  "zipcode": "98101",
  "country": "USA",
  "orderItems": [
    {
      "productName": ".NET Bot Black Hoodie",
      "units": 2,
      "unitPrice": 19.50,
      "pictureUrl": "/api/catalog/items/1/pic"
    }
  ],
  "total": 39.00
}
```

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`

---

#### 3. 注文作成

新しい注文を作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /api/orders` |
| 認証 | 必要 |
| バージョン | V1 |

**リクエストヘッダー**

| ヘッダー名 | 必須 | 説明 |
| --- | --- | --- |
| x-requestid | ○ | リクエストID（GUID形式、べき等性保証用） |

**リクエストボディ**

```json
{
  "userId": "user-123",
  "userName": "John Doe",
  "city": "Seattle",
  "street": "123 Main St",
  "state": "WA",
  "country": "USA",
  "zipCode": "98101",
  "cardNumber": "4111111111111111",
  "cardHolderName": "John Doe",
  "cardExpiration": "2025-12-31T00:00:00Z",
  "cardSecurityNumber": "123",
  "cardTypeId": 1,
  "buyer": "John Doe",
  "items": [
    {
      "id": "item-1",
      "productId": 1,
      "productName": ".NET Bot Black Hoodie",
      "unitPrice": 19.50,
      "oldUnitPrice": 19.50,
      "quantity": 2,
      "pictureUrl": "/api/catalog/items/1/pic"
    }
  ]
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| userId | string | ○ | ユーザーID |
| userName | string | ○ | ユーザー名 |
| city | string | ○ | 配送先市区町村 |
| street | string | ○ | 配送先住所 |
| state | string | ○ | 配送先州/県 |
| country | string | ○ | 配送先国 |
| zipCode | string | ○ | 配送先郵便番号 |
| cardNumber | string | ○ | カード番号 |
| cardHolderName | string | ○ | カード名義人 |
| cardExpiration | DateTime | ○ | カード有効期限 |
| cardSecurityNumber | string | ○ | セキュリティコード |
| cardTypeId | int | ○ | カードタイプID |
| buyer | string | ○ | 購入者名 |
| items | BasketItem[] | ○ | 注文アイテムリスト |

**レスポンス（成功時）**

ステータスコード: `200 OK`

**レスポンス（エラー時）**

ステータスコード: `400 Bad Request`

```
RequestId is missing.
```

---

#### 4. 注文ドラフト作成

注文のドラフトを作成し、合計金額を計算します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /api/orders/draft` |
| 認証 | 必要 |
| バージョン | V1 |

**リクエストボディ**

```json
{
  "buyerId": "user-123",
  "items": [
    {
      "id": "item-1",
      "productId": 1,
      "productName": ".NET Bot Black Hoodie",
      "unitPrice": 19.50,
      "oldUnitPrice": 19.50,
      "quantity": 2,
      "pictureUrl": "/api/catalog/items/1/pic"
    }
  ]
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| buyerId | string | ○ | 購入者ID |
| items | BasketItem[] | ○ | 注文アイテムリスト |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "orderItems": [
    {
      "productId": 1,
      "productName": ".NET Bot Black Hoodie",
      "unitPrice": 19.50,
      "discount": 0,
      "units": 2,
      "pictureUrl": "/api/catalog/items/1/pic"
    }
  ],
  "total": 39.00
}
```

---

#### 5. 注文キャンセル

指定した注文をキャンセルします。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /api/orders/cancel` |
| 認証 | 必要 |
| バージョン | V1 |

**リクエストヘッダー**

| ヘッダー名 | 必須 | 説明 |
| --- | --- | --- |
| x-requestid | ○ | リクエストID（GUID形式） |

**リクエストボディ**

```json
{
  "orderNumber": 1
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| orderNumber | int | ○ | キャンセルする注文番号 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

**レスポンス（エラー時）**

ステータスコード: `400 Bad Request`

```
Empty GUID is not valid for request ID
```

ステータスコード: `500 Internal Server Error`

```json
{
  "detail": "Cancel order failed to process."
}
```

---

#### 6. 注文出荷

指定した注文を出荷状態に変更します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /api/orders/ship` |
| 認証 | 必要 |
| バージョン | V1 |

**リクエストヘッダー**

| ヘッダー名 | 必須 | 説明 |
| --- | --- | --- |
| x-requestid | ○ | リクエストID（GUID形式） |

**リクエストボディ**

```json
{
  "orderNumber": 1
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| orderNumber | int | ○ | 出荷する注文番号 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

**レスポンス（エラー時）**

ステータスコード: `400 Bad Request`

```
Empty GUID is not valid for request ID
```

ステータスコード: `500 Internal Server Error`

```json
{
  "detail": "Ship order failed to process."
}
```

---

#### 7. カードタイプ一覧取得

利用可能なカードタイプの一覧を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/orders/cardtypes` |
| 認証 | 必要 |
| バージョン | V1 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "id": 1,
    "name": "Amex"
  },
  {
    "id": 2,
    "name": "Visa"
  },
  {
    "id": 3,
    "name": "MasterCard"
  }
]
```

---

### Webhooks API

#### 1. Webhook購読一覧取得

認証されたユーザーのWebhook購読一覧を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/webhooks` |
| 認証 | 必要 |
| バージョン | V1 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "id": 1,
    "type": 1,
    "date": "2024-01-15T10:30:00Z",
    "destUrl": "https://example.com/webhook",
    "token": "secret-token",
    "userId": "user-123"
  }
]
```

---

#### 2. Webhook購読詳細取得

指定したIDのWebhook購読詳細を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/webhooks/{id}` |
| 認証 | 必要 |
| バージョン | V1 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | int | ○ | Webhook購読ID |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "id": 1,
  "type": 1,
  "date": "2024-01-15T10:30:00Z",
  "destUrl": "https://example.com/webhook",
  "token": "secret-token",
  "userId": "user-123"
}
```

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`

```
Subscriptions 999 not found
```

---

#### 3. Webhook購読作成

新しいWebhook購読を作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /api/webhooks` |
| 認証 | 必要 |
| バージョン | V1 |

**リクエストボディ**

```json
{
  "url": "https://example.com/webhook",
  "token": "secret-token",
  "event": "OrderShipped",
  "grantUrl": "https://example.com/grant"
}
```

| フィールド名 | 型 | 必須 | バリデーション | 説明 |
| --- | --- | --- | --- | --- |
| url | string | ○ | 有効なURL形式 | Webhook送信先URL |
| token | string | - | - | 認証トークン |
| event | string | ○ | WebhookType列挙値 | イベントタイプ（CatalogItemPriceChange, OrderShipped, OrderPaid） |
| grantUrl | string | ○ | 有効なURL形式 | 権限確認用URL |

**WebhookType列挙値**

| 値 | 説明 |
| --- | --- |
| CatalogItemPriceChange | カタログアイテムの価格変更 |
| OrderShipped | 注文出荷 |
| OrderPaid | 注文支払完了 |

**レスポンス（成功時）**

ステータスコード: `201 Created`

Location: `/api/webhooks/{id}`

**レスポンス（エラー時）**

ステータスコード: `400 Bad Request`

```
Invalid grant URL: https://invalid-url.com/grant
```

---

#### 4. Webhook購読削除

指定したIDのWebhook購読を削除します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `DELETE /api/webhooks/{id}` |
| 認証 | 必要 |
| バージョン | V1 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | int | ○ | Webhook購読ID |

**レスポンス（成功時）**

ステータスコード: `202 Accepted`

Location: `/api/webhooks/{id}`

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`

```
Subscriptions 999 not found
```

---

### Basket API (gRPC)

Basket APIはgRPCプロトコルで提供されます。

#### サービス定義

```protobuf
service Basket {
    rpc GetBasket(GetBasketRequest) returns (CustomerBasketResponse) {}
    rpc UpdateBasket(UpdateBasketRequest) returns (CustomerBasketResponse) {}
    rpc DeleteBasket(DeleteBasketRequest) returns (DeleteBasketResponse) {}
}
```

#### 1. GetBasket

ユーザーの買い物かごを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| メソッド | GetBasket |
| 認証 | 不要（AllowAnonymous） |

**リクエストメッセージ**

```protobuf
message GetBasketRequest {
}
```

**レスポンスメッセージ**

```protobuf
message CustomerBasketResponse {
    repeated BasketItem items = 1;
}

message BasketItem {
    int32 product_id = 2;
    int32 quantity = 6;
}
```

**エラーレスポンス**

ユーザーが認証されていない場合、空のレスポンスを返します。

---

#### 2. UpdateBasket

ユーザーの買い物かごを更新します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| メソッド | UpdateBasket |
| 認証 | 必要 |

**リクエストメッセージ**

```protobuf
message UpdateBasketRequest {
    repeated BasketItem items = 2;
}
```

**レスポンスメッセージ**

```protobuf
message CustomerBasketResponse {
    repeated BasketItem items = 1;
}
```

**エラーレスポンス**

| gRPCステータス | 説明 |
| --- | --- |
| UNAUTHENTICATED | 認証されていない |
| NOT_FOUND | 買い物かごが存在しない |

---

#### 3. DeleteBasket

ユーザーの買い物かごを削除します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| メソッド | DeleteBasket |
| 認証 | 必要 |

**リクエストメッセージ**

```protobuf
message DeleteBasketRequest {
}
```

**レスポンスメッセージ**

```protobuf
message DeleteBasketResponse {
}
```

**エラーレスポンス**

| gRPCステータス | 説明 |
| --- | --- |
| UNAUTHENTICATED | 認証されていない |

---

## データモデル

### CatalogItem

| フィールド | 型 | 説明 |
| --- | --- | --- |
| id | int | 一意識別子 |
| name | string | アイテム名（必須） |
| description | string | 説明 |
| price | decimal | 価格 |
| pictureFileName | string | 画像ファイル名 |
| catalogTypeId | int | カタログタイプID |
| catalogType | CatalogType | カタログタイプオブジェクト |
| catalogBrandId | int | カタログブランドID |
| catalogBrand | CatalogBrand | カタログブランドオブジェクト |
| availableStock | int | 在庫数 |
| restockThreshold | int | 再発注閾値 |
| maxStockThreshold | int | 最大在庫閾値 |
| onReorder | bool | 再発注中フラグ |

### CatalogType

| フィールド | 型 | 説明 |
| --- | --- | --- |
| id | int | 一意識別子 |
| type | string | タイプ名（必須） |

### CatalogBrand

| フィールド | 型 | 説明 |
| --- | --- | --- |
| id | int | 一意識別子 |
| brand | string | ブランド名（必須） |

### Order

| フィールド | 型 | 説明 |
| --- | --- | --- |
| orderNumber | int | 注文番号 |
| date | DateTime | 注文日時 |
| status | string | 注文ステータス |
| description | string | 説明 |
| street | string | 配送先住所 |
| city | string | 配送先市区町村 |
| state | string | 配送先州/県 |
| zipcode | string | 配送先郵便番号 |
| country | string | 配送先国 |
| orderItems | Orderitem[] | 注文アイテムリスト |
| total | decimal | 合計金額 |

### WebhookSubscription

| フィールド | 型 | 説明 |
| --- | --- | --- |
| id | int | 一意識別子 |
| type | WebhookType | イベントタイプ |
| date | DateTime | 登録日時 |
| destUrl | string | 送信先URL（必須） |
| token | string | 認証トークン |
| userId | string | ユーザーID（必須） |

---

## 備考

### AI機能（セマンティック検索）

Catalog APIのセマンティック検索機能は、以下の条件で利用可能です：

- OpenAI APIまたはOllama APIが設定されている場合
- 商品にEmbeddingベクトルが設定されている場合

AIが無効な場合、通常の名前検索にフォールバックします。

### イベント駆動アーキテクチャ

本システムはRabbitMQを使用したイベント駆動アーキテクチャを採用しています。以下のIntegration Eventが発行されます：

- **ProductPriceChangedIntegrationEvent**: 商品価格変更時
- **OrderStartedIntegrationEvent**: 注文開始時
- **OrderStatusChangedToAwaitingValidationIntegrationEvent**: 注文検証待ち状態変更時
- **OrderStatusChangedToPaidIntegrationEvent**: 支払完了時
- **OrderStatusChangedToShippedIntegrationEvent**: 出荷時

### ヘルスチェックエンドポイント

開発環境では以下のヘルスチェックエンドポイントが利用可能です：

- `/health`: 全体的なヘルスチェック
- `/alive`: 生存確認（liveタグ付きのチェックのみ）
