# eShop Reference Application コードリーディングガイドライン

## はじめに

このガイドラインは、eShop Reference Applicationのコードベースを効率的に理解するための手引きです。
C#/.NET Aspireに精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者（Java、TypeScript、Python等からの移行者）
- コードレビューを行う担当者

---

## 1. 言語基礎

> このセクションでは、C#の基本構文と概念を解説します。

### 1.1 プログラム構造

C#はクラスベースのオブジェクト指向言語です。.NET 6以降では「Top-level statements」により、簡潔なエントリーポイントが可能です。

**実例:** `src/Catalog.API/Program.cs:1-23`
```csharp
var builder = WebApplication.CreateBuilder(args);

builder.AddServiceDefaults();
builder.AddApplicationServices();
builder.Services.AddProblemDetails();

var withApiVersioning = builder.Services.AddApiVersioning();

builder.AddDefaultOpenApi(withApiVersioning);

var app = builder.Build();

app.MapDefaultEndpoints();
app.UseStatusCodePages();
app.MapCatalogApi();
app.UseDefaultOpenApi();
app.Run();
```

**解説:**
- `var` キーワード: 型推論により変数宣言を簡潔に記述
- `WebApplication.CreateBuilder(args)`: ASP.NET Coreアプリケーションのビルダーパターン
- メソッドチェーン: `builder.Services.AddXxx()` 形式でサービスを追加

### 1.2 データ型と変数

C#は静的型付け言語で、プロパティやフィールドに型を明示的に指定します。

**実例:** `src/Catalog.API/Model/CatalogItem.cs:1-40`
```csharp
public class CatalogItem
{
    public int Id { get; set; }

    [Required]
    public string Name { get; set; }

    public string? Description { get; set; }

    public decimal Price { get; set; }

    public string? PictureFileName { get; set; }

    public int CatalogTypeId { get; set; }

    public CatalogType? CatalogType { get; set; }

    public int CatalogBrandId { get; set; }

    public CatalogBrand? CatalogBrand { get; set; }

    public int AvailableStock { get; set; }

    public int RestockThreshold { get; set; }

    public int MaxStockThreshold { get; set; }

    [JsonIgnore]
    public Vector? Embedding { get; set; }

    public bool OnReorder { get; set; }

    public CatalogItem(string name) { Name = name; }
}
```

**解説:**
- `string?`: Nullable参照型（nullを許容）
- `[Required]`: データ注釈による検証
- `{ get; set; }`: 自動実装プロパティ
- `[JsonIgnore]`: JSONシリアライズ時に除外する属性

### 1.3 制御構造

条件分岐やループの基本構文です。

**実例:** `src/Catalog.API/Model/CatalogItem.cs:62-79`
```csharp
public int RemoveStock(int quantityDesired)
{
    if (AvailableStock == 0)
    {
        throw new CatalogDomainException($"Empty stock, product item {Name} is sold out");
    }

    if (quantityDesired <= 0)
    {
        throw new CatalogDomainException($"Item units desired should be greater than zero");
    }

    int removed = Math.Min(quantityDesired, this.AvailableStock);
    this.AvailableStock -= removed;
    return removed;
}
```

**解説:**
- `if` 文: 条件分岐の基本
- `throw`: 例外のスロー
- `$"..."`: 文字列補間（変数を埋め込み可能）

### 1.4 関数/メソッド定義

非同期処理を含むメソッドの定義方法です。

**実例:** `src/Catalog.API/Apis/CatalogApi.cs:124-159`
```csharp
public static async Task<Ok<PaginatedItems<CatalogItem>>> GetAllItems(
    [AsParameters] PaginationRequest paginationRequest,
    [AsParameters] CatalogServices services,
    [Description("The name of the item to return")] string? name,
    [Description("The type of items to return")] int? type,
    [Description("The brand of items to return")] int? brand)
{
    var pageSize = paginationRequest.PageSize;
    var pageIndex = paginationRequest.PageIndex;

    var root = (IQueryable<CatalogItem>)services.Context.CatalogItems;

    if (name is not null)
    {
        root = root.Where(c => c.Name.StartsWith(name));
    }

    var totalItems = await root.LongCountAsync();
    var itemsOnPage = await root
        .OrderBy(c => c.Name)
        .Skip(pageSize * pageIndex)
        .Take(pageSize)
        .ToListAsync();

    return TypedResults.Ok(new PaginatedItems<CatalogItem>(pageIndex, pageSize, totalItems, itemsOnPage));
}
```

**解説:**
- `async Task<T>`: 非同期メソッドの戻り値
- `await`: 非同期処理の待機
- `[AsParameters]`: パラメータバインディング属性
- LINQ: `Where()`, `OrderBy()`, `Skip()`, `Take()` によるクエリ構築

### 1.5 モジュール/インポート

名前空間とusingディレクティブの使用方法です。

**実例:** `src/Catalog.API/GlobalUsings.cs`（グローバルusing）
```csharp
global using Microsoft.EntityFrameworkCore;
global using eShop.Catalog.API.Model;
global using eShop.Catalog.API.Infrastructure;
```

**解説:**
- `global using`: プロジェクト全体で使用するusingディレクティブ
- `namespace`: 名前空間の定義（ファイルスコープ形式も可能）

---

## 2. プロジェクト固有の概念

> このセクションでは、当プロジェクト特有の概念を解説します。

### 2.1 フレームワーク固有の概念

#### .NET Aspire

eShopは.NET Aspireを使用したクラウドネイティブアプリケーションです。

**実例:** `src/eShop.AppHost/Program.cs:1-101`
```csharp
var builder = DistributedApplication.CreateBuilder(args);

builder.AddForwardedHeaders();

var redis = builder.AddRedis("redis");
var rabbitMq = builder.AddRabbitMQ("eventbus")
    .WithLifetime(ContainerLifetime.Persistent);
var postgres = builder.AddPostgres("postgres")
    .WithImage("ankane/pgvector")
    .WithImageTag("latest")
    .WithLifetime(ContainerLifetime.Persistent);

var catalogDb = postgres.AddDatabase("catalogdb");
var identityDb = postgres.AddDatabase("identitydb");

// Services
var identityApi = builder.AddProject<Projects.Identity_API>("identity-api", launchProfileName)
    .WithExternalHttpEndpoints()
    .WithReference(identityDb);
```

**主要概念:**
- `DistributedApplication.CreateBuilder()`: 分散アプリケーションのビルダー
- `AddRedis()`, `AddRabbitMQ()`, `AddPostgres()`: インフラリソースの追加
- `AddProject<T>()`: マイクロサービスプロジェクトの追加
- `WithReference()`: サービス間の依存関係定義

#### Entity Framework Core

データアクセスにはEF Coreを使用しています。

**実例:** `src/Catalog.API/Infrastructure/CatalogContext.cs:1-28`
```csharp
public class CatalogContext : DbContext
{
    public CatalogContext(DbContextOptions<CatalogContext> options, IConfiguration configuration) : base(options)
    {
    }

    public required DbSet<CatalogItem> CatalogItems { get; set; }
    public required DbSet<CatalogBrand> CatalogBrands { get; set; }
    public required DbSet<CatalogType> CatalogTypes { get; set; }

    protected override void OnModelCreating(ModelBuilder builder)
    {
        builder.HasPostgresExtension("vector");
        builder.ApplyConfiguration(new CatalogBrandEntityTypeConfiguration());
        builder.ApplyConfiguration(new CatalogTypeEntityTypeConfiguration());
        builder.ApplyConfiguration(new CatalogItemEntityTypeConfiguration());
        builder.UseIntegrationEventLogs();
    }
}
```

#### Minimal APIs

ASP.NET Core Minimal APIsによるエンドポイント定義です。

**実例:** `src/Catalog.API/Apis/CatalogApi.cs:12-113`
```csharp
public static IEndpointRouteBuilder MapCatalogApi(this IEndpointRouteBuilder app)
{
    var vApi = app.NewVersionedApi("Catalog");
    var api = vApi.MapGroup("api/catalog").HasApiVersion(1, 0).HasApiVersion(2, 0);

    api.MapGet("/items", GetAllItems)
        .WithName("ListItems")
        .WithSummary("List catalog items")
        .WithDescription("Get a paginated list of items in the catalog.")
        .WithTags("Items");

    api.MapGet("/items/{id:int}", GetItemById)
        .WithName("GetItem")
        .WithSummary("Get catalog item");

    api.MapPost("/items", CreateItem)
        .WithName("CreateItem")
        .WithSummary("Create a catalog item");

    return app;
}
```

### 2.2 プロジェクト独自のパターン

#### DDDパターン（Domain-Driven Design）

Ordering.Domainプロジェクトでは厳格なDDDパターンを採用しています。

**Entityパターン:** `src/Ordering.Domain/SeedWork/Entity.cs:1-87`
```csharp
public abstract class Entity
{
    int _Id;
    public virtual int Id
    {
        get => _Id;
        protected set => _Id = value;
    }

    private List<INotification> _domainEvents;
    public IReadOnlyCollection<INotification> DomainEvents => _domainEvents?.AsReadOnly();

    public void AddDomainEvent(INotification eventItem)
    {
        _domainEvents = _domainEvents ?? new List<INotification>();
        _domainEvents.Add(eventItem);
    }
}
```

**Aggregateパターン:** `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs:1-70`
```csharp
public class Order : Entity, IAggregateRoot
{
    public DateTime OrderDate { get; private set; }
    [Required]
    public Address Address { get; private set; }
    public OrderStatus OrderStatus { get; private set; }

    private readonly List<OrderItem> _orderItems;
    public IReadOnlyCollection<OrderItem> OrderItems => _orderItems.AsReadOnly();

    public void AddOrderItem(int productId, string productName, decimal unitPrice,
        decimal discount, string pictureUrl, int units = 1)
    {
        var existingOrderForProduct = _orderItems.SingleOrDefault(o => o.ProductId == productId);
        if (existingOrderForProduct != null)
        {
            existingOrderForProduct.AddUnits(units);
        }
        else
        {
            var orderItem = new OrderItem(productId, productName, unitPrice, discount, pictureUrl, units);
            _orderItems.Add(orderItem);
        }
    }
}
```

#### CQRSパターン

MediatRを使用したCommand/Query分離パターンです。

**Commandパターン:** `src/Ordering.API/Application/Commands/CreateOrderCommand.cs:1-84`
```csharp
[DataContract]
public class CreateOrderCommand : IRequest<bool>
{
    [DataMember]
    private readonly List<OrderItemDTO> _orderItems;

    [DataMember]
    public string UserId { get; private set; }

    [DataMember]
    public string City { get; private set; }

    public IEnumerable<OrderItemDTO> OrderItems => _orderItems;
}
```

**CommandHandler:** `src/Ordering.API/Application/Commands/CreateOrderCommandHandler.cs:1-53`
```csharp
public class CreateOrderCommandHandler : IRequestHandler<CreateOrderCommand, bool>
{
    private readonly IOrderRepository _orderRepository;
    private readonly IMediator _mediator;

    public async Task<bool> Handle(CreateOrderCommand message, CancellationToken cancellationToken)
    {
        var address = new Address(message.Street, message.City, message.State, message.Country, message.ZipCode);
        var order = new Order(message.UserId, message.UserName, address, message.CardTypeId,
            message.CardNumber, message.CardSecurityNumber, message.CardHolderName, message.CardExpiration);

        foreach (var item in message.OrderItems)
        {
            order.AddOrderItem(item.ProductId, item.ProductName, item.UnitPrice,
                item.Discount, item.PictureUrl, item.Units);
        }

        _orderRepository.Add(order);
        return await _orderRepository.UnitOfWork.SaveEntitiesAsync(cancellationToken);
    }
}
```

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `{Domain}.API` | APIマイクロサービス | `Catalog.API`, `Ordering.API`, `Basket.API` |
| `{Domain}.Domain` | ドメインロジック層 | `Ordering.Domain` |
| `{Domain}.Infrastructure` | インフラ層 | `Ordering.Infrastructure` |
| `{Noun}Processor` | バックグラウンド処理 | `OrderProcessor`, `PaymentProcessor` |
| `EventBus{Transport}` | イベントバス実装 | `EventBusRabbitMQ` |
| `eShop.{Role}` | 共通コンポーネント | `eShop.AppHost`, `eShop.ServiceDefaults` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `I{Name}` | インターフェース | `IOrderRepository`, `IEventBus` |
| `{Name}Handler` | コマンド/イベントハンドラ | `CreateOrderCommandHandler` |
| `{Name}Command` | CQRSコマンド | `CreateOrderCommand`, `CancelOrderCommand` |
| `{Name}Query` | CQRSクエリ | `GetOrderByIdQuery` |
| `{Name}IntegrationEvent` | マイクロサービス間イベント | `OrderStartedIntegrationEvent` |
| `{Name}DomainEvent` | ドメインイベント | `OrderStartedDomainEvent` |
| `{Name}Repository` | リポジトリ | `OrderRepository`, `BuyerRepository` |
| `{Name}Context` | EF Coreコンテキスト | `CatalogContext`, `OrderingContext` |
| `{Name}Service` | サービスクラス | `BasketService`, `OrderingService` |
| `{Name}Api` | APIエンドポイント定義 | `CatalogApi` |
| `_{name}` (アンダースコア) | プライベートフィールド | `_orderItems`, `_domainEvents` |

### 3.3 プログラム分類一覧

| プログラム種別 | 命名パターン | 役割 |
|---------------|-------------|------|
| エントリーポイント | `Program.cs` | アプリケーション起動 |
| APIエンドポイント | `{Domain}Api.cs` | HTTPエンドポイント定義 |
| gRPCサービス | `{Name}Service.cs` | gRPCサービス実装 |
| コマンド | `{Action}{Noun}Command.cs` | 状態変更要求 |
| コマンドハンドラ | `{Action}{Noun}CommandHandler.cs` | コマンド処理 |
| 統合イベント | `{Event}IntegrationEvent.cs` | サービス間メッセージ |
| イベントハンドラ | `{Event}IntegrationEventHandler.cs` | イベント処理 |
| Razorコンポーネント | `{Name}.razor` | UIコンポーネント |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
eShop/
├── src/
│   ├── eShop.AppHost/           # .NET Aspireオーケストレータ
│   ├── eShop.ServiceDefaults/   # 共通サービス設定
│   ├── Catalog.API/             # 商品カタログサービス
│   │   ├── Apis/                # Minimal APIエンドポイント
│   │   ├── Infrastructure/      # EF Core・DB関連
│   │   ├── IntegrationEvents/   # サービス間イベント
│   │   ├── Model/               # ドメインモデル
│   │   └── Services/            # ビジネスサービス
│   ├── Basket.API/              # ショッピングカートサービス
│   │   ├── Grpc/                # gRPCサービス実装
│   │   ├── Proto/               # Protocol Buffers定義
│   │   └── Repositories/        # データアクセス
│   ├── Ordering.API/            # 注文管理サービス
│   │   └── Application/         # アプリケーション層
│   │       ├── Commands/        # CQRSコマンド
│   │       ├── DomainEventHandlers/  # ドメインイベント
│   │       └── IntegrationEvents/    # 統合イベント
│   ├── Ordering.Domain/         # 注文ドメイン層
│   │   ├── AggregatesModel/     # 集約ルート
│   │   ├── Events/              # ドメインイベント
│   │   └── SeedWork/            # DDDビルディングブロック
│   ├── Ordering.Infrastructure/ # 注文インフラ層
│   ├── OrderProcessor/          # 注文処理バックグラウンドサービス
│   ├── PaymentProcessor/        # 決済処理バックグラウンドサービス
│   ├── Identity.API/            # 認証・認可サービス
│   ├── WebApp/                  # Blazor Webアプリケーション
│   │   ├── Components/          # Razorコンポーネント
│   │   │   ├── Pages/           # ページコンポーネント
│   │   │   └── Layout/          # レイアウトコンポーネント
│   │   └── Services/            # クライアントサービス
│   ├── WebAppComponents/        # 共有UIコンポーネント
│   ├── EventBus/                # イベントバス抽象
│   ├── EventBusRabbitMQ/        # RabbitMQ実装
│   ├── IntegrationEventLogEF/   # イベントログ永続化
│   ├── Webhooks.API/            # Webhookサービス
│   ├── WebhookClient/           # Webhookクライアント
│   ├── ClientApp/               # .NET MAUIモバイルアプリ
│   └── HybridApp/               # Blazor Hybridアプリ
├── tests/
│   ├── Basket.UnitTests/        # Basketユニットテスト
│   ├── Ordering.UnitTests/      # Orderingユニットテスト
│   ├── Catalog.FunctionalTests/ # Catalog機能テスト
│   └── Ordering.FunctionalTests/# Ordering機能テスト
└── e2e/                         # E2Eテスト（Playwright）
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `src/eShop.AppHost` | 分散アプリオーケストレータ | `Program.cs` |
| `src/eShop.ServiceDefaults` | 共通設定（Health Check、OpenTelemetry） | `Extensions.cs` |
| `src/Catalog.API` | 商品カタログCRUD | `Program.cs`, `CatalogApi.cs` |
| `src/Basket.API` | カートgRPCサービス | `BasketService.cs` |
| `src/Ordering.API` | 注文CQRS API | `OrdersApi.cs`, `CreateOrderCommand.cs` |
| `src/Ordering.Domain` | 注文DDDドメイン | `Order.cs`, `Entity.cs` |
| `src/WebApp` | Blazor Serverフロントエンド | `App.razor`, `Catalog.razor` |
| `src/EventBusRabbitMQ` | RabbitMQイベントバス | `RabbitMQEventBus.cs` |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

eShopはマイクロサービスアーキテクチャを採用し、.NET Aspireによるオーケストレーションを行っています。

```mermaid
graph TB
    subgraph "フロントエンド"
        WebApp[WebApp<br/>Blazor Server]
        ClientApp[ClientApp<br/>.NET MAUI]
    end

    subgraph "BFF層"
        MobileBFF[Mobile BFF<br/>YARP]
    end

    subgraph "APIゲートウェイ層"
        IdentityAPI[Identity.API<br/>Duende IdentityServer]
    end

    subgraph "ビジネスサービス層"
        CatalogAPI[Catalog.API]
        BasketAPI[Basket.API<br/>gRPC]
        OrderingAPI[Ordering.API<br/>CQRS]
        WebhooksAPI[Webhooks.API]
    end

    subgraph "バックグラウンドサービス"
        OrderProcessor[OrderProcessor]
        PaymentProcessor[PaymentProcessor]
    end

    subgraph "インフラストラクチャ"
        Redis[(Redis)]
        PostgreSQL[(PostgreSQL)]
        RabbitMQ[RabbitMQ]
    end

    WebApp --> CatalogAPI
    WebApp --> BasketAPI
    WebApp --> OrderingAPI
    ClientApp --> MobileBFF
    MobileBFF --> CatalogAPI
    MobileBFF --> OrderingAPI

    CatalogAPI --> PostgreSQL
    BasketAPI --> Redis
    OrderingAPI --> PostgreSQL

    CatalogAPI --> RabbitMQ
    OrderingAPI --> RabbitMQ
    OrderProcessor --> RabbitMQ
    PaymentProcessor --> RabbitMQ
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| プレゼンテーション層 | UIコンポーネント、ページ表示 | `WebApp/Components/Pages/*.razor` |
| API層 | HTTPエンドポイント、リクエスト処理 | `Catalog.API/Apis/CatalogApi.cs` |
| アプリケーション層 | ユースケース、CQRS | `Ordering.API/Application/Commands/*` |
| ドメイン層 | ビジネスロジック、エンティティ | `Ordering.Domain/AggregatesModel/*` |
| インフラストラクチャ層 | DB、外部サービス連携 | `Ordering.Infrastructure/Repositories/*` |

### 5.3 データフロー

#### 注文作成フロー

```mermaid
sequenceDiagram
    participant UI as WebApp
    participant Basket as Basket.API
    participant Order as Ordering.API
    participant Domain as Order Aggregate
    participant EventBus as RabbitMQ
    participant Processor as OrderProcessor
    participant Payment as PaymentProcessor

    UI->>Basket: GetBasket (gRPC)
    Basket-->>UI: BasketItems
    UI->>Order: CreateOrder (HTTP)
    Order->>Domain: new Order()
    Domain->>Domain: AddOrderItem()
    Order->>EventBus: OrderStartedIntegrationEvent
    Order-->>UI: Order Created
    EventBus->>Processor: Handle Event
    Processor->>EventBus: GracePeriodConfirmedIntegrationEvent
    EventBus->>Order: Set AwaitingValidation
    EventBus->>Payment: Process Payment
    Payment->>EventBus: PaymentSucceeded
```

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

各マイクロサービスは`Program.cs`から起動します。

**実例:** `src/WebApp/Program.cs:1-34`
```csharp
var builder = WebApplication.CreateBuilder(args);

builder.AddServiceDefaults();
builder.Services.AddRazorComponents().AddInteractiveServerComponents();
builder.AddApplicationServices();

var app = builder.Build();

app.MapDefaultEndpoints();

if (!app.Environment.IsDevelopment())
{
    app.UseExceptionHandler("/Error");
    app.UseHsts();
}

app.UseAntiforgery();
app.UseHttpsRedirection();
app.UseStaticFiles();

app.MapRazorComponents<App>().AddInteractiveServerRenderMode();
app.MapForwarder("/product-images/{id}", "https+http://catalog-api", "/api/catalog/items/{id}/pic");

app.Run();
```

### 6.2 ビジネスロジック

**BasketState（カート状態管理）:** `src/WebApp/Services/BasketState.cs:1-156`
```csharp
public class BasketState(
    BasketService basketService,
    CatalogService catalogService,
    OrderingService orderingService,
    AuthenticationStateProvider authenticationStateProvider) : IBasketState
{
    private Task<IReadOnlyCollection<BasketItem>>? _cachedBasket;

    public async Task AddAsync(CatalogItem item)
    {
        var items = (await FetchBasketItemsAsync())
            .Select(i => new BasketQuantity(i.ProductId, i.Quantity)).ToList();

        // 既存アイテムの更新または新規追加
        bool found = false;
        for (var i = 0; i < items.Count; i++)
        {
            if (items[i].ProductId == item.Id)
            {
                items[i] = items[i] with { Quantity = items[i].Quantity + 1 };
                found = true;
                break;
            }
        }

        if (!found)
        {
            items.Add(new BasketQuantity(item.Id, 1));
        }

        _cachedBasket = null;
        await basketService.UpdateBasketAsync(items);
        await NotifyChangeSubscribersAsync();
    }

    public async Task CheckoutAsync(BasketCheckoutInfo checkoutInfo)
    {
        var buyerId = await authenticationStateProvider.GetBuyerIdAsync();
        var orderItems = await FetchBasketItemsAsync();

        var request = new CreateOrderRequest(
            UserId: buyerId,
            // ... 他のパラメータ
            Items: [.. orderItems]);

        await orderingService.CreateOrder(request, checkoutInfo.RequestId);
        await DeleteBasketAsync();
    }
}
```

### 6.3 データアクセス

**OrderRepository:** `src/Ordering.Infrastructure/Repositories/OrderRepository.cs:1-38`
```csharp
public class OrderRepository : IOrderRepository
{
    private readonly OrderingContext _context;

    public IUnitOfWork UnitOfWork => _context;

    public OrderRepository(OrderingContext context)
    {
        _context = context ?? throw new ArgumentNullException(nameof(context));
    }

    public Order Add(Order order)
    {
        return _context.Orders.Add(order).Entity;
    }

    public async Task<Order> GetAsync(int orderId)
    {
        var order = await _context.Orders.FindAsync(orderId);

        if (order != null)
        {
            await _context.Entry(order)
                .Collection(i => i.OrderItems).LoadAsync();
        }

        return order;
    }

    public void Update(Order order)
    {
        _context.Entry(order).State = EntityState.Modified;
    }
}
```

### 6.4 ユーティリティ/共通機能

**ServiceDefaults:** `src/eShop.ServiceDefaults/Extensions.cs:1-129`
```csharp
public static partial class Extensions
{
    public static IHostApplicationBuilder AddServiceDefaults(this IHostApplicationBuilder builder)
    {
        builder.AddBasicServiceDefaults();
        builder.Services.AddServiceDiscovery();

        builder.Services.ConfigureHttpClientDefaults(http =>
        {
            http.AddStandardResilienceHandler();
            http.AddServiceDiscovery();
        });

        return builder;
    }

    public static IHostApplicationBuilder ConfigureOpenTelemetry(this IHostApplicationBuilder builder)
    {
        builder.Logging.AddOpenTelemetry(logging =>
        {
            logging.IncludeFormattedMessage = true;
            logging.IncludeScopes = true;
        });

        builder.Services.AddOpenTelemetry()
            .WithMetrics(metrics =>
            {
                metrics.AddAspNetCoreInstrumentation()
                    .AddHttpClientInstrumentation()
                    .AddRuntimeInstrumentation();
            })
            .WithTracing(tracing =>
            {
                tracing.AddAspNetCoreInstrumentation()
                    .AddGrpcClientInstrumentation()
                    .AddHttpClientInstrumentation();
            });

        return builder;
    }

    public static WebApplication MapDefaultEndpoints(this WebApplication app)
    {
        if (app.Environment.IsDevelopment())
        {
            app.MapHealthChecks("/health");
            app.MapHealthChecks("/alive", new HealthCheckOptions
            {
                Predicate = r => r.Tags.Contains("live")
            });
        }

        return app;
    }
}
```

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| Dependency Injection | コンストラクタ注入 | 高 | 全サービスクラス |
| Repository Pattern | データアクセス抽象化 | 高 | `OrderRepository.cs` |
| CQRS | コマンドクエリ分離 | 高 | `Ordering.API/Application/Commands/*` |
| Event Sourcing | イベント駆動 | 高 | `IntegrationEvents/*` |
| Aggregate Root | DDD集約ルート | 中 | `Ordering.Domain/AggregatesModel/*` |
| Primary Constructor | C#12コンストラクタ簡略化 | 高 | `BasketService.cs` |

### 各パターンの詳細

#### パターン1: Primary Constructor（C#12）

**目的:** コンストラクタ注入を簡潔に記述

**実装例:**
```csharp
// ファイル: src/Basket.API/Grpc/BasketService.cs:8-10
public class BasketService(
    IBasketRepository repository,
    ILogger<BasketService> logger) : Basket.BasketBase
{
    public override async Task<CustomerBasketResponse> GetBasket(
        GetBasketRequest request,
        ServerCallContext context)
    {
        var userId = context.GetUserIdentity();
        var data = await repository.GetBasketAsync(userId);
        // ...
    }
}
```

**解説:** Primary Constructorにより、コンストラクタパラメータがクラスのフィールドとして自動的に利用可能になります。

#### パターン2: 拡張メソッドによるサービス登録

**目的:** サービス登録ロジックの整理

**実装例:**
```csharp
// ファイル: src/Catalog.API/Extensions/Extensions.cs:1-51
public static class Extensions
{
    public static void AddApplicationServices(this IHostApplicationBuilder builder)
    {
        builder.AddNpgsqlDbContext<CatalogContext>("catalogdb", configureDbContextOptions: dbContextOptionsBuilder =>
        {
            dbContextOptionsBuilder.UseNpgsql(b => b.UseVector());
        });

        builder.Services.AddMigration<CatalogContext, CatalogContextSeed>();
        builder.Services.AddTransient<IIntegrationEventLogService, IntegrationEventLogService<CatalogContext>>();
        builder.Services.AddTransient<ICatalogIntegrationEventService, CatalogIntegrationEventService>();

        builder.AddRabbitMqEventBus("eventbus")
               .AddSubscription<OrderStatusChangedToAwaitingValidationIntegrationEvent,
                                OrderStatusChangedToAwaitingValidationIntegrationEventHandler>()
               .AddSubscription<OrderStatusChangedToPaidIntegrationEvent,
                                OrderStatusChangedToPaidIntegrationEventHandler>();
    }
}
```

**解説:** `this IHostApplicationBuilder`により、ビルダーの拡張メソッドとして使用可能。

#### パターン3: TypedResults（Minimal API）

**目的:** 型安全なHTTPレスポンス

**実装例:**
```csharp
// ファイル: src/Catalog.API/Apis/CatalogApi.cs:170-191
public static async Task<Results<Ok<CatalogItem>, NotFound, BadRequest<ProblemDetails>>> GetItemById(
    HttpContext httpContext,
    [AsParameters] CatalogServices services,
    [Description("The catalog item id")] int id)
{
    if (id <= 0)
    {
        return TypedResults.BadRequest<ProblemDetails>(new (){
            Detail = "Id is not valid"
        });
    }

    var item = await services.Context.CatalogItems
        .Include(ci => ci.CatalogBrand)
        .SingleOrDefaultAsync(ci => ci.Id == id);

    if (item == null)
    {
        return TypedResults.NotFound();
    }

    return TypedResults.Ok(item);
}
```

**解説:** `Results<T1, T2, T3>`により、複数の戻り値型を型安全に表現。

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定
2. 処理の流れを追跡（呼び出し関係を追う）
3. データの変換を確認
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: 商品一覧表示フロー

**概要:** ユーザーがトップページにアクセスし、商品一覧を表示するまでの流れ

**処理フロー:**
```
WebApp/Catalog.razor → CatalogService → Catalog.API → CatalogContext → PostgreSQL
```

**詳細な追跡:**

1. **Blazorページのレンダリング** (`src/WebApp/Components/Pages/Catalog/Catalog.razor:1-62`)
   ```razor
   @page "/"
   @inject CatalogService CatalogService

   @code {
       CatalogResult? catalogResult;

       protected override async Task OnInitializedAsync()
       {
           catalogResult = await CatalogService.GetCatalogItems(
               Page.GetValueOrDefault(1) - 1,
               PageSize,
               BrandId,
               ItemTypeId);
       }
   }
   ```

2. **APIエンドポイント呼び出し** (`src/Catalog.API/Apis/CatalogApi.cs:124-159`)
   ```csharp
   public static async Task<Ok<PaginatedItems<CatalogItem>>> GetAllItems(
       [AsParameters] PaginationRequest paginationRequest,
       [AsParameters] CatalogServices services,
       string? name, int? type, int? brand)
   {
       var root = (IQueryable<CatalogItem>)services.Context.CatalogItems;

       if (name is not null) root = root.Where(c => c.Name.StartsWith(name));
       if (type is not null) root = root.Where(c => c.CatalogTypeId == type);
       if (brand is not null) root = root.Where(c => c.CatalogBrandId == brand);

       var totalItems = await root.LongCountAsync();
       var itemsOnPage = await root
           .OrderBy(c => c.Name)
           .Skip(pageSize * pageIndex)
           .Take(pageSize)
           .ToListAsync();

       return TypedResults.Ok(new PaginatedItems<CatalogItem>(pageIndex, pageSize, totalItems, itemsOnPage));
   }
   ```

#### 例2: チェックアウトフロー

**概要:** カートの商品を注文として確定するまでの流れ

**処理フロー:**
```
Checkout.razor → BasketState → OrderingService → Ordering.API → CreateOrderCommand → Order Aggregate → RabbitMQ
```

**詳細な追跡:**

1. **チェックアウトページ** (`src/WebApp/Components/Pages/Checkout/Checkout.razor:101-116`)
   ```csharp
   private async Task HandleValidSubmitAsync()
   {
       Info.CardTypeId = 1;
       await Basket.CheckoutAsync(Info);
       Nav.NavigateTo("user/orders");
   }
   ```

2. **BasketStateのチェックアウト処理** (`src/WebApp/Services/BasketState.cs:78-109`)
   ```csharp
   public async Task CheckoutAsync(BasketCheckoutInfo checkoutInfo)
   {
       var buyerId = await authenticationStateProvider.GetBuyerIdAsync();
       var orderItems = await FetchBasketItemsAsync();

       var request = new CreateOrderRequest(
           UserId: buyerId,
           City: checkoutInfo.City!,
           Items: [.. orderItems]);

       await orderingService.CreateOrder(request, checkoutInfo.RequestId);
       await DeleteBasketAsync();
   }
   ```

3. **注文作成コマンドハンドラ** (`src/Ordering.API/Application/Commands/CreateOrderCommandHandler.cs:29-52`)
   ```csharp
   public async Task<bool> Handle(CreateOrderCommand message, CancellationToken cancellationToken)
   {
       var orderStartedIntegrationEvent = new OrderStartedIntegrationEvent(message.UserId);
       await _orderingIntegrationEventService.AddAndSaveEventAsync(orderStartedIntegrationEvent);

       var address = new Address(message.Street, message.City, message.State, message.Country, message.ZipCode);
       var order = new Order(message.UserId, message.UserName, address,
           message.CardTypeId, message.CardNumber, message.CardSecurityNumber,
           message.CardHolderName, message.CardExpiration);

       foreach (var item in message.OrderItems)
       {
           order.AddOrderItem(item.ProductId, item.ProductName, item.UnitPrice,
               item.Discount, item.PictureUrl, item.Units);
       }

       _orderRepository.Add(order);
       return await _orderRepository.UnitOfWork.SaveEntitiesAsync(cancellationToken);
   }
   ```

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイントを特定したか
- [ ] 呼び出し関係を把握したか
- [ ] データの変換ポイントを確認したか
- [ ] エラーハンドリングを確認したか
- [ ] 最終的な出力を確認したか

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. `README.md` - プロジェクト概要
2. `docs/code-to-docs/インフラ設計書/` - アーキテクチャ概要
3. `docs/code-to-docs/機能一覧/` - 機能俯瞰

#### 特定機能を理解したい場合
1. `docs/code-to-docs/機能一覧/` - 対象機能の確認
2. `docs/code-to-docs/機能設計書/` - 詳細設計
3. `src/{対象サービス}/` - ソースコード

#### 改修作業を行う場合
1. 対象機能の設計書
2. `docs/code-to-docs/API設計書/` - API仕様
3. `tests/` - テストケース

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| README.md | プロジェクト概要、セットアップ手順 | 最初に参照 |
| 機能一覧 | システム機能の一覧 | 機能調査時 |
| 画面設計書 | UIコンポーネント仕様 | フロントエンド開発時 |
| API設計書 | REST/gRPC API仕様 | API開発・連携時 |
| データベース設計書 | ER図、テーブル定義 | データ設計時 |
| バッチ設計書 | バックグラウンドジョブ仕様 | バッチ開発時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: DIコンテナにどこでサービスが登録されているか分からない
A: 各プロジェクトの`Extensions.cs`または`Program.cs`を確認してください。`AddApplicationServices()`拡張メソッド内でサービス登録が行われています。

#### Q: イベントがどこで処理されているか分からない
A: `{サービス}/IntegrationEvents/EventHandling/`ディレクトリを確認してください。ハンドラクラス名は`{EventName}Handler`形式です。

#### Q: エンドポイントのルーティングが分からない
A: 各APIプロジェクトの`Apis/`ディレクトリにある`{Domain}Api.cs`を確認してください。`MapGet()`, `MapPost()`等でルートが定義されています。

#### Q: DDDのエンティティ関係が分からない
A: `Ordering.Domain/AggregatesModel/`を参照してください。`Order`が集約ルートで、`OrderItem`が子エンティティです。

### デバッグのヒント

1. **ログ確認**: .NET Aspireダッシュボードで各サービスのログを確認
2. **トレース確認**: OpenTelemetryによる分散トレースをAspireダッシュボードで確認
3. **ブレークポイント**: コマンドハンドラやイベントハンドラにブレークポイントを設置
4. **RabbitMQ管理画面**: イベントの発行・消費状況を確認

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| Aggregate Root | DDDにおける集約の入り口となるエンティティ |
| CQRS | Command Query Responsibility Segregation（コマンドクエリ責任分離） |
| Integration Event | マイクロサービス間で共有されるイベント |
| Domain Event | ドメイン層内で発生するイベント |
| Minimal API | ASP.NET Coreの軽量なエンドポイント定義方式 |
| .NET Aspire | クラウドネイティブアプリケーション開発フレームワーク |
| Primary Constructor | C#12で導入されたコンストラクタ簡略記法 |

### B. ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `src/eShop.AppHost/Program.cs` | アプリケーションオーケストレータ | サービス構成、依存関係定義 |
| `src/Catalog.API/Apis/CatalogApi.cs` | カタログAPIエンドポイント | 商品CRUD操作 |
| `src/Basket.API/Grpc/BasketService.cs` | カートgRPCサービス | カート操作 |
| `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | 注文集約ルート | 注文ビジネスロジック |
| `src/EventBusRabbitMQ/RabbitMQEventBus.cs` | RabbitMQイベントバス | イベント発行・購読 |

### C. 参考資料

- [.NET Aspire 公式ドキュメント](https://learn.microsoft.com/dotnet/aspire/)
- [ASP.NET Core Minimal APIs](https://learn.microsoft.com/aspnet/core/fundamentals/minimal-apis)
- [Entity Framework Core](https://learn.microsoft.com/ef/core/)
- [Domain-Driven Design](https://docs.microsoft.com/dotnet/architecture/microservices/microservice-ddd-cqrs-patterns/)
- [MediatR](https://github.com/jbogard/MediatR)
