# セキュリティ設計書

## 概要

本ドキュメントは、eShopマイクロサービスアプリケーションのセキュリティ設計について記述する。本システムは、Identity Server (Duende IdentityServer) を中心とした OAuth 2.0 / OpenID Connect ベースの認証・認可基盤を採用し、各マイクロサービス間でJWTトークンを用いたセキュアな通信を実現している。

## 認証設計

### 認証方式

本システムでは、Duende IdentityServer を使用した OAuth 2.0 / OpenID Connect 認証を採用している。

| 項目 | 設定内容 | ソースファイル |
| --- | --- | --- |
| 認証プロバイダー | Duende IdentityServer | `src/Identity.API/Program.cs` |
| ユーザーストア | ASP.NET Core Identity (PostgreSQL) | `src/Identity.API/Program.cs` |
| トークン形式 | JWT (JSON Web Token) | `src/eShop.ServiceDefaults/AuthenticationExtensions.cs` |
| サポートする Grant Type | Authorization Code, Implicit | `src/Identity.API/Configuration/Config.cs` |
| PKCE サポート | 有効 (MAUIクライアント) | `src/Identity.API/Configuration/Config.cs` |

### クライアント構成

| クライアントID | クライアント名 | Grant Type | 用途 |
| --- | --- | --- | --- |
| maui | eShop MAUI OpenId Client | Authorization Code (with PKCE) | モバイルアプリ |
| webapp | WebApp Client | Authorization Code | Webアプリケーション |
| webhooksclient | Webhooks Client | Authorization Code | Webhooksクライアント |
| basketswaggerui | Basket Swagger UI | Implicit | API開発用UI |
| orderingswaggerui | Ordering Swagger UI | Implicit | API開発用UI |
| webhooksswaggerui | WebHooks Service Swagger UI | Implicit | API開発用UI |

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| Cookie有効期限 | 2時間 | `options.Authentication.CookieLifetime = TimeSpan.FromHours(2)` |
| アクセストークン有効期限 | 2時間 (7200秒) | `AccessTokenLifetime = 60*60*2` |
| IDトークン有効期限 | 2時間 (7200秒) | `IdentityTokenLifetime = 60*60*2` |
| セッションCookie有効期限 (WebApp) | 設定可能 (デフォルト60分) | `SessionCookieLifetimeMinutes` 設定 |
| セッション固定化対策 | ASP.NET Core Identity標準機能による対策 | SignInManager使用 |
| SameSite Cookie ポリシー | Lax | `MinimumSameSitePolicy = SameSiteMode.Lax` |

### パスワードポリシー

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| パスワード要件 | ASP.NET Core Identity標準 | 大文字、小文字、数字、特殊文字を含む |
| シードユーザーパスワード例 | Pass123$ | 開発用サンプルパスワード |
| アカウントロックアウト | 有効 | `lockoutOnFailure: true` |

## 認可設計

### 権限体系

| ロール/スコープ | 権限 | 説明 |
| --- | --- | --- |
| openid | ユーザー識別子アクセス | OpenID Connect標準スコープ |
| profile | プロフィール情報アクセス | ユーザー名等の基本情報 |
| offline_access | リフレッシュトークン取得 | 長期アクセス用 |
| orders | 注文サービスアクセス | 注文API操作権限 |
| basket | カートサービスアクセス | カートAPI操作権限 |
| webhooks | Webhooksサービスアクセス | Webhooks登録・管理権限 |

### アクセス制御

#### API レベル認可

各マイクロサービスは JWT Bearer 認証を使用し、トークンの有効性とオーディエンス（Audience）を検証する。

```
// 認証設定 (AuthenticationExtensions.cs)
services.AddAuthentication().AddJwtBearer(options =>
{
    options.Authority = identityUrl;
    options.RequireHttpsMetadata = false;  // 開発環境向け設定
    options.Audience = audience;
});
```

#### リソース所有者ベースアクセス制御

- Basket API: ユーザーIDに基づくカートアクセス制御 (`context.GetUserIdentity()`)
- Ordering API: ユーザーIDに基づく注文アクセス制御 (`GetUserIdentity()`)
- Webhooks API: ユーザーIDに基づくサブスクリプションアクセス制御 (`user.GetUserId()`)

## 通信セキュリティ

| 項目 | 対策 | 備考 |
| --- | --- | --- |
| HTTPS | サービス間通信でサポート | `RequireHttpsMetadata = false` は開発環境向け |
| gRPC通信 | Basket APIでgRPCを使用 | TLS/SSL対応可能 |
| サービスディスカバリ | .NET Aspireによるサービスメッシュ | 内部通信の保護 |

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 | 備考 |
| --- | --- | --- |
| 通信 | TLS (HTTPSおよびgRPC TLS) | サービス間通信の暗号化 |
| パスワード保存 | ASP.NET Core Identity (PBKDF2) | ハッシュ化して保存 |
| クライアントシークレット | SHA256ハッシュ | `new Secret("secret".Sha256())` |
| トークン署名 | 開発者署名キー | 本番環境では適切なキー管理が必要 |

### 機密情報管理

#### クレジットカード番号マスキング

注文作成時にクレジットカード番号をマスキングして保存・ログ出力する。

```csharp
// OrdersApi.cs
var maskedCCNumber = request.CardNumber.Substring(request.CardNumber.Length - 4).PadLeft(request.CardNumber.Length, 'X');
```

#### ログ出力時の機密情報保護

```csharp
// クレジットカード情報を含むリクエストはログに出力しない
services.Logger.LogInformation(
    "Sending command: {CommandName} - {IdProperty}: {CommandId}",
    request.GetGenericTypeName(),
    nameof(request.UserId),
    request.UserId); //don't log the request as it has CC number
```

#### シードデータのカード番号

```csharp
// 保存時はマスク済み形式
CardNumber = "XXXXXXXXXXXX1881"
```

## 入出力対策

| 脅威 | 対策 | 実装箇所 |
| --- | --- | --- |
| XSS | Content-Security-Policy ヘッダー | `SecurityHeadersAttribute.cs` |
| クリックジャッキング | X-Frame-Options: SAMEORIGIN | `SecurityHeadersAttribute.cs` |
| MIMEタイプ混同 | X-Content-Type-Options: nosniff | `SecurityHeadersAttribute.cs` |
| CSRF | ValidateAntiForgeryToken 属性 | `AccountController.cs` |
| リファラー漏洩 | Referrer-Policy: no-referrer | `SecurityHeadersAttribute.cs` |
| SQLインジェクション | Entity Framework Core (パラメータ化クエリ) | 全データアクセス層 |
| 入力検証 | FluentValidation | `Ordering.API/Application/Validations/` |
| オープンリダイレクト | URL検証 | `Url.IsLocalUrl()` による検証 |

### FluentValidation による入力検証

注文作成コマンドの検証ルール:

| フィールド | 検証ルール |
| --- | --- |
| City, Street, State, Country, ZipCode | 必須 |
| CardNumber | 必須、12-19文字 |
| CardHolderName | 必須 |
| CardExpiration | 必須、有効な日付（現在日時以降） |
| CardSecurityNumber | 必須、3文字 |
| CardTypeId | 必須 |
| OrderItems | 1件以上必須 |

### セキュリティヘッダー (SecurityHeadersAttribute)

```csharp
// Content-Security-Policy
var csp = "default-src 'self'; object-src 'none'; frame-ancestors 'none'; sandbox allow-forms allow-same-origin allow-scripts; base-uri 'self';";
```

| ヘッダー | 値 | 目的 |
| --- | --- | --- |
| X-Content-Type-Options | nosniff | MIMEタイプスニッフィング防止 |
| X-Frame-Options | SAMEORIGIN | クリックジャッキング防止 |
| Content-Security-Policy | 上記CSPルール | XSS等の防止 |
| X-Content-Security-Policy | 上記CSPルール | IE互換 |
| Referrer-Policy | no-referrer | リファラー漏洩防止 |

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| 認証成功イベント | UserLoginSuccessEvent (ユーザー名、ユーザーID、クライアントID) | IdentityServer設定による |
| 認証失敗イベント | UserLoginFailureEvent (ユーザー名、失敗理由、クライアントID) | IdentityServer設定による |
| ログアウトイベント | UserLogoutSuccessEvent (Subject ID、表示名) | IdentityServer設定による |
| コマンド実行ログ | LoggingBehavior (コマンド名、リクエスト内容、レスポンス) | アプリケーション設定による |
| バリデーションエラー | ValidatorBehavior (コマンド名、リクエスト、エラー内容) | アプリケーション設定による |

### IdentityServer イベント設定

```csharp
// Program.cs (Identity.API)
options.Events.RaiseErrorEvents = true;
options.Events.RaiseInformationEvents = true;
options.Events.RaiseFailureEvents = true;
options.Events.RaiseSuccessEvents = true;
```

## 認証・認可フロー

### WebApp (Blazor Server) 認証フロー

1. ユーザーがWebAppにアクセス
2. OpenID Connect認証チャレンジを開始
3. Identity APIにリダイレクト
4. ユーザー認証（ユーザー名/パスワード）
5. 認可コード発行
6. WebAppがトークンエンドポイントでアクセストークン取得
7. Cookieにセッション情報保存
8. 以降のAPI呼び出しでアクセストークンを使用

### API間通信の認証トークン伝播

```csharp
// Extensions.cs (WebApp)
builder.Services.AddGrpcClient<Basket.BasketClient>(o => o.Address = new("http://basket-api"))
    .AddAuthToken();  // 認証トークンを伝播
```

## 備考

### 開発環境向け設定（本番環境では変更が必要）

1. **HTTPS必須設定**: `RequireHttpsMetadata = false` を `true` に変更
2. **署名キー**: `AddDeveloperSigningCredential()` を適切なキー管理に変更
3. **キー管理**: `options.KeyManagement.Enabled = false` を `true` に変更
4. **クライアントシークレット**: `"secret"` をセキュアな値に変更
5. **オーディエンス検証**: `ValidateAudience = false` を `true` に変更（必要に応じて）

### セキュリティ強化推奨事項

1. Rate Limiting の実装
2. WAF (Web Application Firewall) の導入
3. セキュリティ監査ログの外部保存
4. 定期的なセキュリティ脆弱性スキャン
5. シークレット管理サービス (Azure Key Vault等) の利用
6. HSTS (HTTP Strict Transport Security) の有効化
7. CSP の `upgrade-insecure-requests` ディレクティブの追加
