# データベース設計書

## 概要

本ドキュメントは、eShopアプリケーションのデータベース設計を定義する。eShopはマイクロサービスアーキテクチャを採用しており、各サービスが独自のデータベースコンテキストを持つ。本システムはPostgreSQLをデータベースとして使用し、Entity Framework Coreを通じてデータアクセスを行う。

### データベースコンテキスト一覧

| コンテキスト名 | スキーマ | 用途 |
| --- | --- | --- |
| OrderingContext | ordering | 注文管理サービス用データベース |
| CatalogContext | (デフォルト) | 商品カタログ管理サービス用データベース |
| ApplicationDbContext | (デフォルト) | ユーザー認証・認可管理用データベース |
| WebhooksContext | (デフォルト) | Webhook購読管理用データベース |

---

## テーブル一覧

### Ordering (注文管理)

| テーブル名 | 対応エンティティ | 説明 |
| --- | --- | --- |
| orders | Order | 注文情報を管理するテーブル |
| orderItems | OrderItem | 注文明細（注文に含まれる商品）を管理するテーブル |
| buyers | Buyer | 購入者情報を管理するテーブル |
| paymentmethods | PaymentMethod | 支払い方法情報を管理するテーブル |
| cardtypes | CardType | クレジットカード種別のマスタテーブル |
| requests | ClientRequest | 冪等性を担保するためのリクエスト管理テーブル |
| IntegrationEventLog | IntegrationEventLogEntry | 統合イベントログを管理するテーブル |

### Catalog (商品カタログ)

| テーブル名 | 対応エンティティ | 説明 |
| --- | --- | --- |
| Catalog | CatalogItem | 商品情報を管理するテーブル |
| CatalogBrand | CatalogBrand | 商品ブランドのマスタテーブル |
| CatalogType | CatalogType | 商品タイプのマスタテーブル |
| IntegrationEventLog | IntegrationEventLogEntry | 統合イベントログを管理するテーブル |

### Identity (認証・認可)

| テーブル名 | 対応エンティティ | 説明 |
| --- | --- | --- |
| AspNetUsers | ApplicationUser | ユーザー情報を管理するテーブル |
| AspNetRoles | IdentityRole | ロール情報を管理するテーブル |
| AspNetUserRoles | IdentityUserRole | ユーザーとロールの関連を管理する中間テーブル |
| AspNetUserClaims | IdentityUserClaim | ユーザークレーム情報を管理するテーブル |
| AspNetRoleClaims | IdentityRoleClaim | ロールクレーム情報を管理するテーブル |
| AspNetUserLogins | IdentityUserLogin | 外部ログインプロバイダー情報を管理するテーブル |
| AspNetUserTokens | IdentityUserToken | ユーザートークン情報を管理するテーブル |

### Webhooks (Webhook管理)

| テーブル名 | 対応エンティティ | 説明 |
| --- | --- | --- |
| Subscriptions | WebhookSubscription | Webhook購読情報を管理するテーブル |

---

## 各テーブル定義

### Ordering スキーマ

#### 1. orders

注文のヘッダー情報を管理するテーブル。DDDパターンに基づくAggregateRootとして設計されている。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| Id | integer | NO | 主キー（HiLoシーケンス: orderseq） |
| OrderDate | timestamp with time zone | NO | 注文日時 |
| OrderStatus | character varying(30) | NO | 注文ステータス（Submitted/AwaitingValidation/StockConfirmed/Paid/Shipped/Cancelled） |
| Description | text | YES | 注文の説明・ステータス変更理由 |
| BuyerId | integer | YES | 購入者ID（buyersテーブルへの外部キー） |
| PaymentMethodId | integer | YES | 支払い方法ID（paymentmethodsテーブルへの外部キー） |
| Address_Street | text | YES | 配送先住所（番地） - Owned Entity |
| Address_City | text | YES | 配送先住所（市区町村） - Owned Entity |
| Address_State | text | YES | 配送先住所（都道府県） - Owned Entity |
| Address_Country | text | YES | 配送先住所（国） - Owned Entity |
| Address_ZipCode | text | YES | 配送先住所（郵便番号） - Owned Entity |

**インデックス:**
- PRIMARY KEY (Id)
- INDEX (BuyerId)
- INDEX (PaymentMethodId)

**外部キー:**
- BuyerId -> buyers(Id)
- PaymentMethodId -> paymentmethods(Id) ON DELETE RESTRICT

---

#### 2. orderItems

注文明細情報を管理するテーブル。各注文に含まれる商品の詳細を保持する。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| Id | integer | NO | 主キー（HiLoシーケンス: orderitemseq） |
| OrderId | integer | NO | 注文ID（ordersテーブルへの外部キー） |
| ProductId | integer | NO | 商品ID（Catalogサービスの商品への参照） |
| ProductName | text | NO | 商品名（スナップショット） |
| UnitPrice | numeric | NO | 単価 |
| Discount | numeric | NO | 割引額 |
| Units | integer | NO | 数量 |
| PictureUrl | text | YES | 商品画像URL |

**インデックス:**
- PRIMARY KEY (Id)
- INDEX (OrderId)

**外部キー:**
- OrderId -> orders(Id) ON DELETE CASCADE

---

#### 3. buyers

購入者情報を管理するテーブル。DDDパターンに基づくAggregateRootとして設計されている。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| Id | integer | NO | 主キー（HiLoシーケンス: buyerseq） |
| IdentityGuid | character varying(200) | NO | 認証サービスのユーザーID |
| Name | text | YES | 購入者名 |

**インデックス:**
- PRIMARY KEY (Id)
- UNIQUE INDEX (IdentityGuid)

---

#### 4. paymentmethods

支払い方法（クレジットカード）情報を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| Id | integer | NO | 主キー（HiLoシーケンス: paymentseq） |
| BuyerId | integer | NO | 購入者ID（buyersテーブルへの外部キー） |
| Alias | character varying(200) | NO | 支払い方法の別名 |
| CardNumber | character varying(25) | NO | カード番号 |
| CardHolderName | character varying(200) | NO | カード名義人 |
| Expiration | timestamp with time zone | NO | 有効期限 |
| CardTypeId | integer | NO | カード種別ID（cardtypesテーブルへの外部キー） |

**インデックス:**
- PRIMARY KEY (Id)
- INDEX (BuyerId)
- INDEX (CardTypeId)

**外部キー:**
- BuyerId -> buyers(Id) ON DELETE CASCADE
- CardTypeId -> cardtypes(Id) ON DELETE CASCADE

---

#### 5. cardtypes

クレジットカード種別のマスタテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| Id | integer | NO | 主キー（手動設定、自動採番なし） |
| Name | character varying(200) | NO | カード種別名（Visa, MasterCard等） |

**インデックス:**
- PRIMARY KEY (Id)

---

#### 6. requests

コマンドの冪等性を担保するためのリクエスト管理テーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| Id | uuid | NO | 主キー（リクエストの一意識別子） |
| Name | text | NO | コマンド名 |
| Time | timestamp with time zone | NO | リクエスト時刻 |

**インデックス:**
- PRIMARY KEY (Id)

---

#### 7. IntegrationEventLog (Ordering)

統合イベントのログを管理するテーブル。マイクロサービス間の非同期通信で使用。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| EventId | uuid | NO | 主キー（イベントID） |
| EventTypeName | text | NO | イベント種別の完全修飾名 |
| Content | text | NO | イベント内容（JSON形式） |
| State | integer | NO | イベント状態（NotPublished/InProgress/Published/PublishedFailed） |
| TimesSent | integer | NO | 送信試行回数 |
| CreationTime | timestamp with time zone | NO | 作成日時 |
| TransactionId | uuid | NO | トランザクションID |

**インデックス:**
- PRIMARY KEY (EventId)

---

### Catalog スキーマ

#### 8. Catalog

商品情報を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| Id | integer | NO | 主キー（Identity） |
| Name | character varying(50) | NO | 商品名 |
| Description | text | YES | 商品説明 |
| Price | numeric | NO | 価格 |
| PictureFileName | text | YES | 商品画像ファイル名 |
| CatalogTypeId | integer | NO | 商品タイプID（CatalogTypeテーブルへの外部キー） |
| CatalogBrandId | integer | NO | 商品ブランドID（CatalogBrandテーブルへの外部キー） |
| AvailableStock | integer | NO | 在庫数 |
| RestockThreshold | integer | NO | 再発注閾値 |
| MaxStockThreshold | integer | NO | 最大在庫数 |
| OnReorder | boolean | NO | 再発注中フラグ |
| Embedding | vector(384) | YES | 商品説明のベクトル埋め込み（AI検索用） |

**インデックス:**
- PRIMARY KEY (Id)
- INDEX (Name)
- INDEX (CatalogBrandId)
- INDEX (CatalogTypeId)

**外部キー:**
- CatalogTypeId -> CatalogType(Id) ON DELETE CASCADE
- CatalogBrandId -> CatalogBrand(Id) ON DELETE CASCADE

---

#### 9. CatalogBrand

商品ブランドのマスタテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| Id | integer | NO | 主キー（Identity） |
| Brand | character varying(100) | NO | ブランド名 |

**インデックス:**
- PRIMARY KEY (Id)

---

#### 10. CatalogType

商品タイプのマスタテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| Id | integer | NO | 主キー（Identity） |
| Type | character varying(100) | NO | タイプ名 |

**インデックス:**
- PRIMARY KEY (Id)

---

#### 11. IntegrationEventLog (Catalog)

統合イベントのログを管理するテーブル（Catalogコンテキスト用）。構造はOrderingスキーマと同一。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| EventId | uuid | NO | 主キー（イベントID） |
| EventTypeName | text | NO | イベント種別の完全修飾名 |
| Content | text | NO | イベント内容（JSON形式） |
| State | integer | NO | イベント状態 |
| TimesSent | integer | NO | 送信試行回数 |
| CreationTime | timestamp with time zone | NO | 作成日時 |
| TransactionId | uuid | NO | トランザクションID |

**インデックス:**
- PRIMARY KEY (EventId)

---

### Identity スキーマ

#### 12. AspNetUsers

ユーザー情報を管理するテーブル。ASP.NET Core Identityの標準テーブルを拡張。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| Id | text | NO | 主キー（GUID文字列） |
| UserName | character varying(256) | YES | ユーザー名 |
| NormalizedUserName | character varying(256) | YES | 正規化されたユーザー名 |
| Email | character varying(256) | YES | メールアドレス |
| NormalizedEmail | character varying(256) | YES | 正規化されたメールアドレス |
| EmailConfirmed | boolean | NO | メール確認済みフラグ |
| PasswordHash | text | YES | パスワードハッシュ |
| SecurityStamp | text | YES | セキュリティスタンプ |
| ConcurrencyStamp | text | YES | 同時実行制御用スタンプ |
| PhoneNumber | text | YES | 電話番号 |
| PhoneNumberConfirmed | boolean | NO | 電話番号確認済みフラグ |
| TwoFactorEnabled | boolean | NO | 二要素認証有効フラグ |
| LockoutEnd | timestamp with time zone | YES | ロックアウト終了日時 |
| LockoutEnabled | boolean | NO | ロックアウト有効フラグ |
| AccessFailedCount | integer | NO | アクセス失敗回数 |
| Name | text | NO | 名前（拡張プロパティ） |
| LastName | text | NO | 姓（拡張プロパティ） |
| CardNumber | text | NO | カード番号（拡張プロパティ） |
| SecurityNumber | text | NO | セキュリティ番号（拡張プロパティ） |
| Expiration | text | NO | 有効期限（MM/YY形式、拡張プロパティ） |
| CardHolderName | text | NO | カード名義人（拡張プロパティ） |
| CardType | integer | NO | カード種別（拡張プロパティ） |
| Street | text | NO | 住所（番地、拡張プロパティ） |
| City | text | NO | 住所（市区町村、拡張プロパティ） |
| State | text | NO | 住所（都道府県、拡張プロパティ） |
| Country | text | NO | 住所（国、拡張プロパティ） |
| ZipCode | text | NO | 住所（郵便番号、拡張プロパティ） |

**インデックス:**
- PRIMARY KEY (Id)
- UNIQUE INDEX (NormalizedUserName) - UserNameIndex
- INDEX (NormalizedEmail) - EmailIndex

---

#### 13. AspNetRoles

ロール情報を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| Id | text | NO | 主キー（GUID文字列） |
| Name | character varying(256) | YES | ロール名 |
| NormalizedName | character varying(256) | YES | 正規化されたロール名 |
| ConcurrencyStamp | text | YES | 同時実行制御用スタンプ |

**インデックス:**
- PRIMARY KEY (Id)
- UNIQUE INDEX (NormalizedName) - RoleNameIndex

---

#### 14. AspNetUserRoles

ユーザーとロールの関連を管理する中間テーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| UserId | text | NO | ユーザーID（主キー、外部キー） |
| RoleId | text | NO | ロールID（主キー、外部キー） |

**インデックス:**
- PRIMARY KEY (UserId, RoleId)
- INDEX (RoleId)

**外部キー:**
- UserId -> AspNetUsers(Id) ON DELETE CASCADE
- RoleId -> AspNetRoles(Id) ON DELETE CASCADE

---

#### 15. AspNetUserClaims

ユーザークレーム情報を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| Id | integer | NO | 主キー（Identity） |
| UserId | text | NO | ユーザーID（外部キー） |
| ClaimType | text | YES | クレームタイプ |
| ClaimValue | text | YES | クレーム値 |

**インデックス:**
- PRIMARY KEY (Id)
- INDEX (UserId)

**外部キー:**
- UserId -> AspNetUsers(Id) ON DELETE CASCADE

---

#### 16. AspNetRoleClaims

ロールクレーム情報を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| Id | integer | NO | 主キー（Identity） |
| RoleId | text | NO | ロールID（外部キー） |
| ClaimType | text | YES | クレームタイプ |
| ClaimValue | text | YES | クレーム値 |

**インデックス:**
- PRIMARY KEY (Id)
- INDEX (RoleId)

**外部キー:**
- RoleId -> AspNetRoles(Id) ON DELETE CASCADE

---

#### 17. AspNetUserLogins

外部ログインプロバイダー情報を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| LoginProvider | text | NO | ログインプロバイダー名（主キー） |
| ProviderKey | text | NO | プロバイダーキー（主キー） |
| ProviderDisplayName | text | YES | プロバイダー表示名 |
| UserId | text | NO | ユーザーID（外部キー） |

**インデックス:**
- PRIMARY KEY (LoginProvider, ProviderKey)
- INDEX (UserId)

**外部キー:**
- UserId -> AspNetUsers(Id) ON DELETE CASCADE

---

#### 18. AspNetUserTokens

ユーザートークン情報を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| UserId | text | NO | ユーザーID（主キー、外部キー） |
| LoginProvider | text | NO | ログインプロバイダー名（主キー） |
| Name | text | NO | トークン名（主キー） |
| Value | text | YES | トークン値 |

**インデックス:**
- PRIMARY KEY (UserId, LoginProvider, Name)

**外部キー:**
- UserId -> AspNetUsers(Id) ON DELETE CASCADE

---

### Webhooks スキーマ

#### 19. Subscriptions

Webhook購読情報を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| Id | integer | NO | 主キー（Identity） |
| Type | integer | NO | Webhook種別（OrderPaid/OrderShipped等） |
| Date | timestamp with time zone | NO | 登録日時 |
| DestUrl | text | NO | 通知先URL |
| Token | text | YES | 認証トークン |
| UserId | text | NO | ユーザーID |

**インデックス:**
- PRIMARY KEY (Id)
- INDEX (Type)
- INDEX (UserId)

---

## シーケンス定義

### Ordering スキーマ

| シーケンス名 | 初期値 | 増分 | 用途 |
| --- | --- | --- | --- |
| buyerseq | 1 | 10 | buyersテーブルのId生成（HiLo） |
| orderseq | 1 | 10 | ordersテーブルのId生成（HiLo） |
| orderitemseq | 1 | 10 | orderItemsテーブルのId生成（HiLo） |
| paymentseq | 1 | 10 | paymentmethodsテーブルのId生成（HiLo） |

---

## 備考

### アーキテクチャ特性

1. **マイクロサービスパターン**: 各サービス（Ordering, Catalog, Identity, Webhooks）は独自のデータベースコンテキストを持ち、サービス間でデータベースを共有しない

2. **DDDパターンの適用**: OrderingドメインではDomain-Driven Designパターンを採用
   - AggregateRoot: Order, Buyer
   - Value Object: Address（Ordersテーブルに埋め込み）
   - Entity: OrderItem, PaymentMethod, CardType

3. **イベント駆動アーキテクチャ**: IntegrationEventLogテーブルを使用したOutboxパターンにより、マイクロサービス間の信頼性の高い非同期通信を実現

4. **冪等性の確保**: requestsテーブルにより、コマンドの重複実行を防止

### データベース固有機能

1. **PostgreSQL拡張**: Catalogテーブルでpgvector拡張を使用し、AI検索のためのベクトル埋め込み（vector(384)型）をサポート

2. **HiLoアルゴリズム**: Orderingスキーマでは効率的なID生成のためにHiLoアルゴリズムを使用

3. **Owned Entity**: AddressはOrderのOwned Entityとして同一テーブルに保存される

### セキュリティ考慮事項

1. クレジットカード情報（CardNumber, SecurityNumber）はプレーンテキストで保存されているため、本番環境では暗号化が必要

2. ASP.NET Core Identityによるパスワードハッシュ化を使用
