# バッチ設計書 3-MigrationHostedService

## 概要

本ドキュメントは、eShopシステムにおけるデータベースマイグレーションおよびシードデータ投入を行うバックグラウンドサービス（MigrationHostedService）の設計を記載する。

### 本バッチの処理概要

このバッチは、アプリケーション起動時にEntity Framework Coreを使用してデータベースマイグレーションを自動実行し、必要に応じてシードデータを投入するバックグラウンドサービスである。各DbContext単位で登録・実行が可能な汎用的な仕組みを提供する。

**業務上の目的・背景**：マイクロサービスアーキテクチャにおいて、各サービスが独自のデータベーススキーマを管理する必要がある。アプリケーション起動時に自動的にマイグレーションを適用することで、デプロイメントプロセスを簡素化し、スキーマの一貫性を保証する。また、初期データ（マスタデータ等）の投入も同時に行うことで、サービスの初期セットアップを自動化する。開発環境やテスト環境での迅速なセットアップに特に有効である。

**バッチの実行タイミング**：アプリケーション起動時に1回のみ実行される。StartAsyncメソッドで同期的にマイグレーションが完了するまで待機し、その後ExecuteAsyncで即座に完了する。

**主要な処理内容**：
1. アプリケーション起動時にStartAsyncが呼び出される
2. DIコンテナから対象のDbContextを取得
3. ExecutionStrategyを使用してリトライ可能な実行コンテキストを作成
4. Database.MigrateAsyncでマイグレーションを適用
5. 登録されたSeederのSeedAsyncメソッドでシードデータを投入
6. OpenTelemetryを通じてトレーシング情報を出力

**前後の処理との関連**：本バッチはアプリケーションの起動フェーズで最初に実行される処理の一つである。データベーススキーマが準備されることで、後続のすべてのデータベース操作が可能になる。Catalog.API、Ordering.API、Identity.API、Webhooks.APIなど、複数のサービスで共通的に使用される。

**影響範囲**：対象のDbContextに紐づくデータベースのスキーマ全体に影響を与える。テーブル作成、カラム追加、インデックス作成などのDDL操作およびシードデータのINSERT操作が実行される。

## バッチ種別

データベース移行 / 初期データ投入

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | アプリケーション起動時に1回 |
| 実行時刻 | アプリケーション起動時 |
| 実行曜日 | - |
| 実行日 | - |
| トリガー | アプリケーション起動（HostedService開始） |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| データベース接続 | 対象のデータベースサーバーへの接続が可能であること |
| DbContext登録 | 対象のDbContextがDIコンテナに登録されていること |
| マイグレーションファイル | 適用するマイグレーションファイルが存在すること |

### 実行可否判定

- データベースサーバーが起動していること
- 接続文字列が正しく設定されていること
- 必要な権限（DDL実行権限）が付与されていること

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| DbContext | TContext | Yes | - | マイグレーション対象のEntity Framework DbContext |
| Seeder | Func<TContext, IServiceProvider, Task> | No | 空の処理 | シードデータ投入処理を行うデリゲート |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| Migrationsフォルダ | C#クラス | Entity Framework Coreのマイグレーションファイル |
| appsettings.json | JSON | データベース接続文字列 |
| Seederクラス | C#クラス | IDbSeeder<TContext>を実装したシードクラス |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| データベース | DDL/DML | スキーマ変更およびシードデータ |
| __EFMigrationsHistory | DB | 適用済みマイグレーションの履歴 |
| ログ | テキスト | マイグレーション実行状況をILoggerで出力 |
| OpenTelemetry | Trace | トレーシング情報を出力 |

### 出力ファイル仕様

ファイル出力なし（データベースへの直接出力のみ）

| 項目 | 内容 |
|-----|------|
| ファイル名 | - |
| 出力先 | - |
| 文字コード | - |
| 区切り文字 | - |

## 処理フロー

### 処理シーケンス

```
1. サービス開始（StartAsync）
   └─ HostedServiceがアプリケーション起動時に呼び出される
2. スコープ作成
   └─ IServiceProviderからスコープを作成
3. DbContext取得
   └─ スコープからDbContextを解決
4. Activity開始
   └─ OpenTelemetryのActivityを開始（トレーシング用）
5. ExecutionStrategy作成
   └─ Database.CreateExecutionStrategyでリトライ可能な実行コンテキストを取得
6. マイグレーション実行
   └─ Database.MigrateAsyncで未適用のマイグレーションを適用
7. シードデータ投入
   └─ Seeder関数を実行してシードデータを投入
8. 完了ログ出力
   └─ マイグレーション完了を記録
9. ExecuteAsync
   └─ Task.CompletedTaskを返却（即座に完了）
```

### フローチャート

```mermaid
flowchart TD
    A[アプリケーション起動] --> B[StartAsync 呼び出し]
    B --> C[IServiceProvider.CreateScope]
    C --> D[DbContext 取得]
    D --> E[Activity 開始 - OpenTelemetry]
    E --> F[ExecutionStrategy 作成]
    F --> G[Database.MigrateAsync 実行]
    G --> H{Seeder 登録あり?}
    H -->|Yes| I[Seeder.SeedAsync 実行]
    H -->|No| J[完了ログ出力]
    I --> J
    J --> K[ExecuteAsync - 即座に完了]
    K --> L[サービス起動完了]

    G --> M{エラー発生?}
    M -->|Yes| N[エラーログ出力]
    N --> O[Activity に例外タグ設定]
    O --> P[例外を再スロー]
```

## データベース操作仕様

### 操作別データベース影響一覧

| 処理 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| マイグレーション適用 | 全テーブル | DDL | テーブル作成、カラム追加、インデックス作成等 |
| マイグレーション履歴記録 | __EFMigrationsHistory | INSERT | 適用したマイグレーションIDを記録 |
| シードデータ投入 | 各マスタテーブル | INSERT/UPDATE | 初期データの投入 |

### テーブル別操作詳細

#### __EFMigrationsHistory

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | MigrationId | マイグレーションファイルの識別子 | 適用済みマイグレーションを追跡 |
| INSERT | ProductVersion | EF Coreのバージョン | 適用時のEF Coreバージョンを記録 |

#### Seederによるテーブル操作（例：CatalogContext）

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | CatalogBrands | catalog.jsonから読み込んだブランド情報 | 初回のみ |
| INSERT | CatalogTypes | catalog.jsonから読み込んだタイプ情報 | 初回のみ |
| INSERT | CatalogItems | catalog.jsonから読み込んだ商品情報 | 初回のみ |

#### Seederによるテーブル操作（例：OrderingContext）

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | CardTypes | Amex, Visa, MasterCard等のカード種別 | 初回のみ |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | DbException | データベース接続失敗 | エラーログ出力後、例外を再スロー。アプリケーション起動失敗 |
| - | InvalidOperationException | マイグレーション競合 | エラーログ出力後、例外を再スロー。手動でのマイグレーション解決が必要 |
| - | Exception | その他のエラー | エラーログ出力、Activityに例外タグを設定後、例外を再スロー |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | ExecutionStrategyに依存（デフォルトで一時的エラーに対してリトライ） |
| リトライ間隔 | ExecutionStrategyに依存（指数バックオフ） |
| リトライ対象エラー | 一時的なデータベース接続エラー |

### 障害時対応

- マイグレーション失敗時はアプリケーション起動が中断される
- ログおよびOpenTelemetryトレースでエラー詳細を確認
- 手動でのデータベース状態確認およびマイグレーション適用が必要

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | マイグレーションはEF Coreが自動管理、Seederは明示的なSaveChanges単位 |
| コミットタイミング | 各マイグレーション適用後、SaveChangesAsync呼び出し時 |
| ロールバック条件 | マイグレーション中のエラー発生時（部分適用の可能性あり） |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | マイグレーション数およびシードデータ量に依存 |
| 目標処理時間 | 数秒〜数分（データ量に依存） |
| メモリ使用量上限 | シードデータの読み込みサイズに依存 |

## 排他制御

- データベースレベルでのマイグレーションロックが適用される
- 複数インスタンスの同時起動時は、1つのインスタンスのみがマイグレーションを実行
- __EFMigrationsHistoryテーブルで適用済みマイグレーションを追跡

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 情報ログ | マイグレーション開始時 | "Migrating database associated with context {DbContextName}" |
| 情報ログ | シードデータ投入時 | シーダー固有のログ（例："Seeded catalog with {NumItems} items"） |
| エラーログ | エラー発生時 | "An error occurred while migrating the database used on context {DbContextName}" |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| マイグレーション失敗 | 発生時 | アプリケーションログ、OpenTelemetry |
| マイグレーション所要時間 | 閾値超過時 | OpenTelemetryトレース |

## 備考

- 本番環境では自動マイグレーションは推奨されない（コード内コメント参照）
- 本番環境ではSQLスクリプトを生成して手動適用することを推奨
- OpenTelemetryによるトレーシングが有効化されており、分散トレーシングで監視可能
- ActivitySource名は"DbMigrations"で固定
- 各サービス（Catalog.API, Ordering.API, Identity.API, Webhooks.API）で個別に使用
- IDbSeeder<TContext>インターフェースを実装することでカスタムシーダーを作成可能
