# 機能設計書 1-商品一覧取得

## 概要

本ドキュメントは、eShopアプリケーションにおける商品一覧取得機能の設計仕様を定義する。Catalog.APIサービスが提供するREST APIエンドポイントにより、カタログ内の商品をページネーション付きで取得し、ブランド・タイプ・名前によるフィルタリングを可能にする。

### 本機能の処理概要

**業務上の目的・背景**：ECサイトにおいて、ユーザーが商品を閲覧・比較し、購入対象を選定するための基盤機能である。大量の商品データを効率的にページ分割して表示することで、ユーザーエクスペリエンスとシステム性能を両立させる。また、ブランドやタイプでのフィルタリングにより、ユーザーが目的の商品を素早く見つけられるようにする。

**機能の利用シーン**：カタログ画面（WebApp/HybridApp）での商品一覧表示、検索結果の表示、カテゴリ別商品表示など、商品を一覧で表示するあらゆる場面で利用される。AIチャットボット機能からも商品検索のために呼び出される。

**主要な処理内容**：
1. ページネーションパラメータ（PageIndex, PageSize）を受け取り、取得範囲を決定
2. フィルタ条件（name, type, brand）が指定された場合はWHERE句を動的に構築
3. 商品名でソートした上で、指定ページの商品データをデータベースから取得
4. 総件数と取得データをPaginatedItems形式でレスポンスとして返却

**関連システム・外部連携**：PostgreSQLデータベースからEntity Framework Coreを通じてデータを取得する。フロントエンド（WebApp、HybridApp）およびAIチャットボット機能と連携する。

**権限による制御**：本APIエンドポイントは認証不要で公開されており、誰でもアクセス可能である。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | カタログ画面 | 主画面 | CatalogServiceを使用してページネーション付き商品一覧を取得・表示 |
| 8 | チャットボット | 参照画面 | CatalogServiceで商品検索・一覧取得 |
| 25 | カタログ画面（HybridApp） | 主画面 | CatalogServiceで商品一覧を取得・表示 |

## 機能種別

CRUD操作（Read）/ データ検索

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| PageSize | int | No | 1ページあたりの取得件数 | デフォルト値: 10 |
| PageIndex | int | No | 取得するページのインデックス（0始まり） | デフォルト値: 0 |
| name | string | No | 商品名フィルタ（前方一致） | V2のみ。minlength制限なし |
| type | int | No | 商品タイプIDフィルタ | V2のみ。CatalogTypeIdとして存在するID |
| brand | int | No | ブランドIDフィルタ | V2のみ。CatalogBrandIdとして存在するID |

### 入力データソース

- HTTPクエリパラメータ経由でPaginationRequestおよびフィルタ条件を受け取る
- V1 API: `/api/catalog/items?pageSize={pageSize}&pageIndex={pageIndex}`
- V2 API: `/api/catalog/items?pageSize={pageSize}&pageIndex={pageIndex}&name={name}&type={type}&brand={brand}`

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| PageIndex | int | 現在のページインデックス |
| PageSize | int | 1ページあたりの件数 |
| Count | long | 総件数（フィルタ適用後） |
| Data | IEnumerable&lt;CatalogItem&gt; | 商品データの配列 |

**CatalogItem構造**

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Id | int | 商品ID |
| Name | string | 商品名 |
| Description | string? | 商品説明 |
| Price | decimal | 価格 |
| PictureFileName | string? | 商品画像ファイル名 |
| CatalogTypeId | int | 商品タイプID |
| CatalogBrandId | int | ブランドID |
| AvailableStock | int | 在庫数 |
| RestockThreshold | int | 再発注閾値 |
| MaxStockThreshold | int | 最大在庫閾値 |
| OnReorder | bool | 再発注フラグ |

### 出力先

- HTTP Response Body（JSON形式）
- Content-Type: application/json

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエスト受信
   └─ PaginationRequestパラメータをバインド

2. クエリベース構築
   └─ CatalogItemsテーブルをIQueryableとして取得

3. フィルタ条件適用（V2のみ）
   ├─ name指定時: Where(c => c.Name.StartsWith(name))
   ├─ type指定時: Where(c => c.CatalogTypeId == type)
   └─ brand指定時: Where(c => c.CatalogBrandId == brand)

4. 総件数取得
   └─ LongCountAsync()で総件数をカウント

5. ページデータ取得
   ├─ OrderBy(c => c.Name)でソート
   ├─ Skip(pageSize * pageIndex)でオフセット
   ├─ Take(pageSize)で件数制限
   └─ ToListAsync()でデータ取得

6. レスポンス生成
   └─ PaginatedItems<CatalogItem>オブジェクトを生成して返却
```

### フローチャート

```mermaid
flowchart TD
    A[HTTPリクエスト受信] --> B[PaginationRequest取得]
    B --> C{V2 API?}
    C -->|Yes| D[フィルタパラメータ取得]
    C -->|No| E[クエリベース構築]
    D --> E
    E --> F{nameフィルタ?}
    F -->|Yes| G[Name.StartsWith条件追加]
    F -->|No| H{typeフィルタ?}
    G --> H
    H -->|Yes| I[CatalogTypeId条件追加]
    H -->|No| J{brandフィルタ?}
    I --> J
    J -->|Yes| K[CatalogBrandId条件追加]
    J -->|No| L[総件数カウント]
    K --> L
    L --> M[名前順ソート]
    M --> N[Skip/Take適用]
    N --> O[データ取得]
    O --> P[PaginatedItems生成]
    P --> Q[JSONレスポンス返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | デフォルトページサイズ | PageSizeが指定されない場合は10件をデフォルトとする | 常時 |
| BR-002 | ページインデックス初期値 | PageIndexが指定されない場合は0をデフォルトとする | 常時 |
| BR-003 | 名前フィルタ方式 | 名前フィルタは前方一致で検索する | nameパラメータ指定時 |
| BR-004 | ソート順 | 商品名の昇順でソートする | 常時 |

### 計算ロジック

- オフセット計算: `offset = pageSize * pageIndex`
- 取得範囲: `offset` から `offset + pageSize - 1` までのレコード

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 商品一覧取得 | Catalog | SELECT | フィルタ条件に合致する商品を取得 |

### テーブル別操作詳細

#### Catalog

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id, Name, Description, Price, PictureFileName, CatalogTypeId, CatalogBrandId, AvailableStock, RestockThreshold, MaxStockThreshold, OnReorder | フィルタ条件、ページネーション条件に基づく | Embeddingカラムは除外（JsonIgnore） |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | パラメータ形式が不正 | ProblemDetails形式でエラー詳細を返却 |
| 500 | InternalServerError | データベース接続エラー等 | サーバーエラーとして処理 |

### リトライ仕様

本機能では明示的なリトライ処理は実装されていない。データベース接続のリトライはEntity Framework Core/Npgsqlの組み込み機能に依存する。

## トランザクション仕様

読み取り専用操作のため、明示的なトランザクション制御は行わない。Entity Framework Coreのデフォルトの読み取り一貫性が適用される。

## パフォーマンス要件

- Nameカラムにはインデックスが設定されており、名前フィルタおよびソートの性能を確保
- ページネーションによりメモリ使用量を制限し、大量データでも安定したレスポンスを実現

## セキュリティ考慮事項

- 認証不要のパブリックAPI
- SQLインジェクション対策: Entity Framework CoreのLINQクエリによりパラメータ化されたクエリを生成
- 在庫情報（AvailableStock等）が公開されるが、ECサイトでは一般的な仕様

## 備考

- V1とV2の両方のAPIバージョンが存在し、V2ではフィルタパラメータがクエリパラメータとして追加されている
- V1では、フィルタなしの一覧取得のみサポート

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、APIで返却されるデータ構造と、データベースエンティティの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | 商品エンティティの構造、プロパティ定義、Embeddingの存在 |
| 1-2 | PaginatedItems.cs | `src/Catalog.API/Model/PaginatedItems.cs` | ページネーション結果のラッパークラス構造 |
| 1-3 | PaginationRequest.cs | `src/Catalog.API/Model/PaginationRequest.cs` | ページネーションリクエストパラメータの定義 |

**読解のコツ**: CatalogItem.csでは`[JsonIgnore]`属性が付いたEmbeddingプロパティに注目。これはAI検索用のベクトルデータで、通常のJSON応答には含まれない。

#### Step 2: エントリーポイントを理解する

処理の起点となるAPIエンドポイント定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | Minimal APIのエンドポイント定義と処理ハンドラ |

**主要処理フロー**:
1. **21-30行目**: V1/V2の`/items`エンドポイント定義
2. **116-121行目**: V1用ハンドラ`GetAllItemsV1` - V2メソッドをフィルタなしで呼び出し
3. **124-159行目**: V2用ハンドラ`GetAllItems` - フィルタ条件の構築からデータ取得まで
4. **134-147行目**: フィルタ条件の動的構築（name, type, brand）
5. **149-158行目**: 総件数カウントとページデータ取得

#### Step 3: サービス層を理解する

依存性注入されるサービスクラスの構造を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CatalogServices.cs | `src/Catalog.API/Model/CatalogServices.cs` | APIハンドラに注入されるサービス群のコンテナ |

**主要処理フロー**:
- **4-16行目**: Primary Constructorでサービスを受け取り、プロパティとして公開

#### Step 4: データアクセス層を理解する

Entity Framework CoreによるDBアクセスの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | CatalogContext.cs | `src/Catalog.API/Infrastructure/CatalogContext.cs` | DbContextとDbSet定義 |
| 4-2 | CatalogItemEntityTypeConfiguration.cs | `src/Catalog.API/Infrastructure/EntityConfigurations/CatalogItemEntityTypeConfiguration.cs` | テーブルマッピングとインデックス設定 |

**主要処理フロー**:
- **CatalogContext.cs 14行目**: `CatalogItems`プロパティ - 商品テーブルへのアクセスポイント
- **CatalogItemEntityTypeConfiguration.cs 8行目**: テーブル名"Catalog"へのマッピング
- **CatalogItemEntityTypeConfiguration.cs 22行目**: Name列へのインデックス設定

### プログラム呼び出し階層図

```
HTTP Request (GET /api/catalog/items)
    │
    ├─ CatalogApi.GetAllItemsV1() [V1の場合]
    │      └─ CatalogApi.GetAllItems()
    │
    └─ CatalogApi.GetAllItems() [V2の場合]
           │
           ├─ CatalogServices.Context (CatalogContext)
           │      └─ DbSet<CatalogItem>.CatalogItems
           │             ├─ Where() - フィルタ条件適用
           │             ├─ LongCountAsync() - 総件数取得
           │             ├─ OrderBy() - ソート
           │             ├─ Skip() - オフセット
           │             ├─ Take() - 件数制限
           │             └─ ToListAsync() - データ取得
           │
           └─ PaginatedItems<CatalogItem> - 結果ラッピング
```

### データフロー図

```
[入力]                       [処理]                              [出力]

HTTPリクエスト               CatalogApi.GetAllItems()            HTTPレスポンス
  │                              │                                  │
  ├─ PageSize ─────────────────▶│                                  │
  ├─ PageIndex ────────────────▶│                                  │
  ├─ name ─────────────────────▶├─▶ IQueryable<CatalogItem>       │
  ├─ type ─────────────────────▶│      │                          │
  └─ brand ────────────────────▶│      ▼                          │
                                 │   PostgreSQL                     │
                                 │   (Catalog table)                │
                                 │      │                          │
                                 │      ▼                          │
                                 │   List<CatalogItem>             │
                                 │      │                          │
                                 │      ▼                          │
                                 └─▶ PaginatedItems ───────────────▶ JSON
                                      ├─ PageIndex
                                      ├─ PageSize
                                      ├─ Count
                                      └─ Data[]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | ソース | APIエンドポイント定義とハンドラ |
| CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | ソース | 商品エンティティモデル |
| CatalogType.cs | `src/Catalog.API/Model/CatalogType.cs` | ソース | 商品タイプエンティティ |
| CatalogBrand.cs | `src/Catalog.API/Model/CatalogBrand.cs` | ソース | ブランドエンティティ |
| PaginatedItems.cs | `src/Catalog.API/Model/PaginatedItems.cs` | ソース | ページネーション結果モデル |
| PaginationRequest.cs | `src/Catalog.API/Model/PaginationRequest.cs` | ソース | ページネーションリクエストモデル |
| CatalogServices.cs | `src/Catalog.API/Model/CatalogServices.cs` | ソース | サービスコンテナ |
| CatalogContext.cs | `src/Catalog.API/Infrastructure/CatalogContext.cs` | ソース | Entity Framework DbContext |
| CatalogItemEntityTypeConfiguration.cs | `src/Catalog.API/Infrastructure/EntityConfigurations/CatalogItemEntityTypeConfiguration.cs` | ソース | エンティティ設定 |
