# 機能設計書 11-商品登録

## 概要

本ドキュメントは、eShopシステムにおける商品登録機能の詳細設計を記載する。

### 本機能の処理概要

本機能は、新しい商品をカタログに登録するためのREST APIエンドポイントを提供する。商品情報の登録と同時に、AI機能が有効な場合は商品の名前と説明からベクトル埋め込み（Embedding）を自動生成し、セマンティック検索機能に対応させる。

**業務上の目的・背景**：EC事業において、新商品の追加は継続的な売上拡大に不可欠である。本機能により、管理者やバックオフィス担当者は新たな商品をシステムに登録し、顧客に販売可能な状態にできる。また、AI埋め込みベクトルの自動生成により、登録直後からセマンティック検索（意味的な類似性に基づく検索）の対象となり、顧客体験が向上する。

**機能の利用シーン**：
- 新商品の発売時に商品情報を登録する場合
- 仕入れた商品をカタログに追加する場合
- 季節商品やキャンペーン商品を新規追加する場合
- 他システムからのデータ移行時

**主要な処理内容**：
1. HTTPリクエストボディから商品情報（名前、説明、価格、在庫数、ブランドID、タイプID等）を受け取る
2. 商品情報からCatalogItemエンティティを生成する
3. AI機能が有効な場合、商品名と説明からベクトル埋め込みを生成する
4. CatalogContextを通じてデータベースに商品を永続化する
5. 作成された商品のURIを含むCreated (201)レスポンスを返却する

**関連システム・外部連携**：
- AI埋め込みベクトル生成（Microsoft.Extensions.AI経由でのEmbeddingGenerator）
- PostgreSQLデータベース（pgvector拡張によるベクトル格納）

**権限による制御**：本APIエンドポイントは認可設定により保護される可能性があるが、現在のコードでは明示的な認可属性は設定されていない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は管理API経由で使用され、直接関連する画面は存在しない |

## 機能種別

CRUD操作（Create） / データ連携（AI埋め込み生成）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Id | int | No | 商品ID（指定しない場合は自動採番） | - |
| Name | string | Yes | 商品名 | Required属性により必須 |
| Description | string | No | 商品説明 | - |
| Price | decimal | No | 価格 | - |
| PictureFileName | string | No | 商品画像ファイル名 | - |
| CatalogTypeId | int | Yes | 商品タイプID | 存在するCatalogTypeを参照 |
| CatalogBrandId | int | Yes | ブランドID | 存在するCatalogBrandを参照 |
| AvailableStock | int | No | 在庫数 | - |
| RestockThreshold | int | No | 再発注閾値 | - |
| MaxStockThreshold | int | No | 最大在庫閾値 | - |

### 入力データソース

HTTP POSTリクエストボディ（JSON形式）

```json
{
  "name": "商品名",
  "description": "商品説明",
  "price": 100.00,
  "pictureFileName": "product.png",
  "catalogTypeId": 1,
  "catalogBrandId": 1,
  "availableStock": 100,
  "restockThreshold": 10,
  "maxStockThreshold": 200
}
```

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Location | string (Header) | 作成された商品のリソースURI（/api/catalog/items/{id}） |

### 出力先

- HTTPレスポンス（201 Created）
- PostgreSQL CatalogItemsテーブル

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエスト受信
   └─ POST /api/catalog/items でリクエストボディからCatalogItemを受け取る

2. CatalogItemエンティティ生成
   └─ 入力データからCatalogItemの新規インスタンスを作成

3. AI埋め込みベクトル生成
   └─ CatalogAI.GetEmbeddingAsync(item)を呼び出し
   └─ 商品名と説明を連結したテキストからベクトルを生成
   └─ AI機能が無効の場合はnullを設定

4. データベース保存
   └─ CatalogContext.CatalogItems.Add(item)でエンティティ追加
   └─ SaveChangesAsync()でトランザクションコミット

5. レスポンス返却
   └─ Created (201)ステータスと作成リソースURIを返却
```

### フローチャート

```mermaid
flowchart TD
    A[POST /api/catalog/items] --> B[CatalogItem生成]
    B --> C{AI機能有効?}
    C -->|Yes| D[埋め込みベクトル生成]
    C -->|No| E[Embedding = null]
    D --> F[DBに保存]
    E --> F
    F --> G[Created 201 返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-11-01 | 商品名必須 | 商品名は必須項目である | 商品登録時 |
| BR-11-02 | 埋め込み自動生成 | AI機能有効時は埋め込みベクトルを自動生成する | IEmbeddingGeneratorがDI登録されている場合 |
| BR-11-03 | 在庫閾値管理 | MaxStockThreshold以上の在庫は登録できない | AddStock処理時 |

### 計算ロジック

埋め込みベクトル生成：
- 入力テキスト = 商品名 + " " + 商品説明
- 出力ベクトル次元 = 384（EmbeddingDimensions定数）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 商品登録 | CatalogItems | INSERT | 新規商品レコードを挿入 |

### テーブル別操作詳細

#### CatalogItems

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | Id | 自動採番（指定時はその値） | 主キー |
| INSERT | Name | 入力値 | 必須 |
| INSERT | Description | 入力値 | 任意 |
| INSERT | Price | 入力値 | - |
| INSERT | PictureFileName | 入力値 | - |
| INSERT | CatalogTypeId | 入力値 | 外部キー |
| INSERT | CatalogBrandId | 入力値 | 外部キー |
| INSERT | AvailableStock | 入力値 | - |
| INSERT | RestockThreshold | 入力値 | - |
| INSERT | MaxStockThreshold | 入力値 | - |
| INSERT | Embedding | AI生成値 or null | pgvector型 |
| INSERT | OnReorder | false（デフォルト） | - |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | Bad Request | リクエストボディが不正な場合 | リクエストボディを正しいJSON形式に修正 |
| 500 | Internal Server Error | DB保存失敗、AI生成失敗時 | サーバーログを確認し再試行 |

### リトライ仕様

本機能にリトライ機能は実装されていない。クライアント側で再試行を実装する。

## トランザクション仕様

- CatalogContext.SaveChangesAsync()により単一トランザクションでコミット
- エラー発生時は自動ロールバック

## パフォーマンス要件

- AI埋め込み生成には外部APIコールが発生するため、数百ミリ秒〜数秒の処理時間を要する可能性がある
- AI機能が無効な場合は高速に処理される

## セキュリティ考慮事項

- 本APIは管理用途のため、本番環境では認証・認可の設定が必要
- 入力値に対するサニタイジングはEntity Frameworkにより実施

## 備考

- AI埋め込み機能は設定により有効/無効を切り替え可能
- 商品画像はPictureFileNameで参照されるが、画像ファイル自体のアップロードは別処理

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、商品エンティティの構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | 商品エンティティの全プロパティと在庫管理メソッドを確認 |
| 1-2 | CatalogBrand.cs | `src/Catalog.API/Model/CatalogBrand.cs` | ブランドマスタの構造 |
| 1-3 | CatalogType.cs | `src/Catalog.API/Model/CatalogType.cs` | 商品タイプマスタの構造 |

**読解のコツ**: CatalogItemの`Embedding`プロパティは`[JsonIgnore]`属性が付与されており、APIレスポンスには含まれない。pgvector型（`Vector`クラス）で定義されている。

#### Step 2: エントリーポイントを理解する

処理の起点となるAPIエンドポイントを特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | CreateItem静的メソッドが商品登録のエントリーポイント |

**主要処理フロー**:
1. **103行目**: `api.MapPost("/items", CreateItem)` - エンドポイント定義
2. **366-388行目**: `CreateItem`メソッド - 商品登録処理本体
3. **370-381行目**: CatalogItemインスタンス生成
4. **382行目**: AI埋め込みベクトル生成
5. **384-385行目**: DB保存

#### Step 3: AI埋め込み生成を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ICatalogAI.cs | `src/Catalog.API/Services/ICatalogAI.cs` | AI機能のインターフェース定義 |
| 3-2 | CatalogAI.cs | `src/Catalog.API/Services/CatalogAI.cs` | 埋め込み生成の実装詳細 |

**主要処理フロー**:
- **25行目**: `IsEnabled`プロパティ - AI機能有効判定
- **28-31行目**: `GetEmbeddingAsync(CatalogItem)` - 商品からの埋め込み生成
- **75行目**: `CatalogItemToString` - 名前と説明の連結

#### Step 4: データベースコンテキストを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | CatalogContext.cs | `src/Catalog.API/Infrastructure/CatalogContext.cs` | EFCoreコンテキストとDbSet定義 |
| 4-2 | CatalogServices.cs | `src/Catalog.API/Model/CatalogServices.cs` | DIコンテナからの依存性注入 |

### プログラム呼び出し階層図

```
CatalogApi.CreateItem (エントリーポイント)
    │
    ├─ new CatalogItem() - エンティティ生成
    │
    ├─ CatalogAI.GetEmbeddingAsync(item)
    │      │
    │      └─ IEmbeddingGenerator.GenerateVectorAsync()
    │             └─ 外部AI API呼び出し
    │
    ├─ CatalogContext.CatalogItems.Add(item)
    │
    └─ CatalogContext.SaveChangesAsync()
           └─ PostgreSQL INSERT
```

### データフロー図

```
[入力]               [処理]                    [出力]

HTTP POST        CatalogApi.CreateItem      PostgreSQL
  Body  ──────▶    CatalogItem生成  ──────▶  CatalogItems
(JSON)              ↓                        テーブル
                CatalogAI                       │
                  ↓                            ▼
              Embedding生成              HTTP 201 Created
              (AI有効時)                  Location Header
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | ソース | APIエンドポイント定義・商品登録処理 |
| CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | ソース | 商品エンティティモデル |
| CatalogServices.cs | `src/Catalog.API/Model/CatalogServices.cs` | ソース | 依存性注入用サービスクラス |
| ICatalogAI.cs | `src/Catalog.API/Services/ICatalogAI.cs` | ソース | AIサービスインターフェース |
| CatalogAI.cs | `src/Catalog.API/Services/CatalogAI.cs` | ソース | AI埋め込み生成実装 |
| CatalogContext.cs | `src/Catalog.API/Infrastructure/CatalogContext.cs` | ソース | EF Coreデータベースコンテキスト |
| CatalogBrand.cs | `src/Catalog.API/Model/CatalogBrand.cs` | ソース | ブランドマスタエンティティ |
| CatalogType.cs | `src/Catalog.API/Model/CatalogType.cs` | ソース | 商品タイプマスタエンティティ |
