# 機能設計書 12-商品更新

## 概要

本ドキュメントは、eShopシステムにおける商品更新機能の詳細設計を記載する。

### 本機能の処理概要

本機能は、既存の商品情報を更新するためのREST APIエンドポイントを提供する。商品情報の更新に伴い、AI埋め込みベクトルも再生成される。特に価格変更が発生した場合は、統合イベント（ProductPriceChangedIntegrationEvent）を発行し、他のマイクロサービス（買い物かご等）に価格変更を通知する。

**業務上の目的・背景**：EC運営において、商品の価格改定、在庫数調整、説明文の修正などは日常的に発生する業務である。本機能により、管理者は既存商品の情報をリアルタイムに更新できる。価格変更時の統合イベント発行により、システム全体での整合性（買い物かご内の価格更新等）が保たれる。

**機能の利用シーン**：
- 商品価格の値上げ・値下げ時
- 在庫数の手動調整時
- 商品説明の修正・改善時
- 商品画像の変更時
- ブランドやカテゴリの変更時

**主要な処理内容**：
1. HTTPリクエストから商品IDと更新内容を受け取る
2. データベースから既存の商品情報を取得する
3. 商品が存在しない場合はNotFound (404)を返却する
4. 入力データで商品情報を更新する
5. AI埋め込みベクトルを再生成する
6. 価格変更がある場合は統合イベントを発行する
7. データベースに変更を保存する
8. Created (201)レスポンスを返却する

**関連システム・外部連携**：
- AI埋め込みベクトル生成（Microsoft.Extensions.AI経由）
- イベントバス（RabbitMQ）への統合イベント発行
- 統合イベントログへの永続化

**権限による制御**：本APIエンドポイントは管理者向けであり、本番環境では適切な認可設定が必要である。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は管理API経由で使用され、直接関連する画面は存在しない |

## 機能種別

CRUD操作（Update） / データ連携（AI埋め込み生成、統合イベント発行）

## 入力仕様

### 入力パラメータ

**パスパラメータ（V2 API）**

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| id | int | Yes | 更新対象の商品ID | 正の整数 |

**リクエストボディ**

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Name | string | Yes | 商品名 | Required属性により必須 |
| Description | string | No | 商品説明 | - |
| Price | decimal | No | 価格 | - |
| PictureFileName | string | No | 商品画像ファイル名 | - |
| CatalogTypeId | int | Yes | 商品タイプID | 存在するCatalogTypeを参照 |
| CatalogBrandId | int | Yes | ブランドID | 存在するCatalogBrandを参照 |
| AvailableStock | int | No | 在庫数 | - |
| RestockThreshold | int | No | 再発注閾値 | - |
| MaxStockThreshold | int | No | 最大在庫閾値 | - |

### 入力データソース

- パスパラメータ: URL（V2 API: `/api/catalog/items/{id}`）
- リクエストボディ: HTTP PUTリクエストボディ（JSON形式）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Location | string (Header) | 更新された商品のリソースURI（/api/catalog/items/{id}） |

### 出力先

- HTTPレスポンス（201 Created / 404 NotFound / 400 BadRequest）
- PostgreSQL CatalogItemsテーブル（UPDATE）
- 統合イベントログテーブル（価格変更時）
- イベントバス（価格変更時）

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエスト受信
   └─ PUT /api/catalog/items/{id} または PUT /api/catalog/items（V1）

2. 既存商品の取得
   └─ CatalogContext.CatalogItems.SingleOrDefaultAsync(i => i.Id == id)

3. 存在チェック
   └─ 商品が存在しない場合は NotFound (404) を返却

4. 商品情報更新
   └─ catalogEntry.CurrentValues.SetValues(productToUpdate) で一括更新

5. AI埋め込みベクトル再生成
   └─ CatalogAI.GetEmbeddingAsync(catalogItem)

6. 価格変更チェック
   └─ priceEntry.IsModified で価格変更を判定

7A. 価格変更あり
    └─ ProductPriceChangedIntegrationEvent を生成
    └─ EventService.SaveEventAndCatalogContextChangesAsync でアトミックに保存
    └─ EventService.PublishThroughEventBusAsync でイベント発行

7B. 価格変更なし
    └─ SaveChangesAsync() で通常保存

8. レスポンス返却
   └─ Created (201) と Location ヘッダー
```

### フローチャート

```mermaid
flowchart TD
    A[PUT /api/catalog/items/id] --> B[既存商品取得]
    B --> C{商品存在?}
    C -->|No| D[NotFound 404]
    C -->|Yes| E[商品情報更新]
    E --> F[埋め込みベクトル再生成]
    F --> G{価格変更?}
    G -->|Yes| H[統合イベント生成]
    H --> I[イベント+商品をアトミック保存]
    I --> J[イベントバスに発行]
    G -->|No| K[通常保存]
    J --> L[Created 201]
    K --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-12-01 | 存在確認 | 更新対象の商品が存在しない場合は404エラーを返す | 商品更新時 |
| BR-12-02 | 価格変更通知 | 価格変更時は統合イベントを発行して他サービスに通知する | priceEntry.IsModified == true |
| BR-12-03 | 埋め込み再生成 | 更新時は常に埋め込みベクトルを再生成する | 商品更新時 |
| BR-12-04 | アトミック保存 | 価格変更時はイベントログと商品更新をアトミックに保存する | 価格変更時 |

### 計算ロジック

統合イベント（ProductPriceChangedIntegrationEvent）のペイロード：
- ProductId: 商品ID
- NewPrice: 新価格
- OldPrice: 旧価格（変更前の価格）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 商品更新 | CatalogItems | UPDATE | 商品レコードを更新 |
| イベント保存（価格変更時） | IntegrationEventLog | INSERT | 統合イベントを永続化 |

### テーブル別操作詳細

#### CatalogItems

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id | WHERE Id = @id | 更新対象取得 |
| UPDATE | 全カラム | 入力値で上書き | SetValues使用 |
| UPDATE | Embedding | AI再生成値 | 更新時常に再生成 |

#### IntegrationEventLog（価格変更時のみ）

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | EventId | GUID | イベント識別子 |
| INSERT | EventTypeName | ProductPriceChangedIntegrationEvent | イベント型名 |
| INSERT | Content | JSON | イベントペイロード |
| INSERT | State | NotPublished | 発行状態 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | Bad Request | V1 APIでリクエストボディにIdがない場合 | リクエストボディにIdを含める |
| 404 | Not Found | 指定IDの商品が存在しない場合 | 正しい商品IDを指定 |
| 500 | Internal Server Error | DB保存失敗、イベント発行失敗時 | サーバーログを確認し再試行 |

### リトライ仕様

本機能にリトライ機能は実装されていない。イベント発行失敗時はイベントログに保存された状態で残り、別途再発行処理が可能。

## トランザクション仕様

- 価格変更時：SaveEventAndCatalogContextChangesAsyncにより、商品更新とイベントログ保存がアトミックに実行される
- 価格変更なし：SaveChangesAsync()による単一トランザクション
- イベント発行：別トランザクション（発行失敗時もDB更新はコミット済み）

## パフォーマンス要件

- AI埋め込み再生成により、外部APIコールが発生する
- イベント発行はRabbitMQへの非同期送信

## セキュリティ考慮事項

- 本APIは管理用途のため、本番環境では認証・認可の設定が必要
- 価格変更履歴は統合イベントログに残る（監査証跡）

## 備考

- V1 APIはリクエストボディにIDを含める（PUT /api/catalog/items）
- V2 APIはパスパラメータでIDを指定（PUT /api/catalog/items/{id}）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | 更新対象となる商品エンティティの全プロパティ |
| 1-2 | ProductPriceChangedIntegrationEvent.cs | `src/Catalog.API/IntegrationEvents/Events/ProductPriceChangedIntegrationEvent.cs` | 価格変更時に発行されるイベントの構造 |

**読解のコツ**: Entity Framework CoreのChangeTrackerを使用して変更検知を行っている。`priceEntry.IsModified`で価格変更を判定し、`priceEntry.OriginalValue`で変更前の価格を取得している。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | UpdateItem/UpdateItemV1が更新処理のエントリーポイント |

**主要処理フロー**:
1. **93行目**: `v1.MapPut("/items", UpdateItemV1)` - V1エンドポイント定義
2. **98行目**: `v2.MapPut("/items/{id:int}", UpdateItem)` - V2エンドポイント定義
3. **310-322行目**: `UpdateItemV1`メソッド - V1ラッパー
4. **324-363行目**: `UpdateItem`メソッド - 更新処理本体
5. **330行目**: 既存商品取得
6. **340-341行目**: SetValuesによる一括更新
7. **343行目**: 埋め込みベクトル再生成
8. **345-361行目**: 価格変更判定と分岐処理

#### Step 3: 統合イベントサービスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ICatalogIntegrationEventService.cs | `src/Catalog.API/IntegrationEvents/ICatalogIntegrationEventService.cs` | イベントサービスインターフェース |
| 3-2 | CatalogIntegrationEventService.cs | `src/Catalog.API/IntegrationEvents/CatalogIntegrationEventService.cs` | アトミック保存と発行の実装 |

**主要処理フロー**:
- `SaveEventAndCatalogContextChangesAsync`: トランザクション内でイベントと商品を同時保存
- `PublishThroughEventBusAsync`: イベントバスへの発行

### プログラム呼び出し階層図

```
CatalogApi.UpdateItem (エントリーポイント)
    │
    ├─ CatalogContext.CatalogItems.SingleOrDefaultAsync()
    │      └─ 既存商品取得
    │
    ├─ catalogEntry.CurrentValues.SetValues()
    │      └─ 一括更新
    │
    ├─ CatalogAI.GetEmbeddingAsync(catalogItem)
    │      └─ 埋め込み再生成
    │
    └─ [価格変更時]
           │
           ├─ new ProductPriceChangedIntegrationEvent()
           │
           ├─ EventService.SaveEventAndCatalogContextChangesAsync()
           │      └─ アトミック保存
           │
           └─ EventService.PublishThroughEventBusAsync()
                  └─ RabbitMQ発行
```

### データフロー図

```
[入力]               [処理]                    [出力]

HTTP PUT         CatalogApi.UpdateItem      PostgreSQL
  Body  ──────▶    商品情報更新  ───────────▶ CatalogItems
(JSON)              ↓
  +             埋め込み再生成
Path ID              ↓
                価格変更判定
                     │
               ┌─────┴─────┐
               ↓           ↓
            変更あり    変更なし
               ↓           ↓
         イベント発行   通常保存
               │           │
               ▼           │
         RabbitMQ          │
               │           │
               └─────┬─────┘
                     ▼
              HTTP 201 Created
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | ソース | APIエンドポイント定義・更新処理 |
| CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | ソース | 商品エンティティモデル |
| ICatalogIntegrationEventService.cs | `src/Catalog.API/IntegrationEvents/ICatalogIntegrationEventService.cs` | ソース | 統合イベントサービスIF |
| CatalogIntegrationEventService.cs | `src/Catalog.API/IntegrationEvents/CatalogIntegrationEventService.cs` | ソース | 統合イベントサービス実装 |
| ProductPriceChangedIntegrationEvent.cs | `src/Catalog.API/IntegrationEvents/Events/ProductPriceChangedIntegrationEvent.cs` | ソース | 価格変更イベント定義 |
| CatalogAI.cs | `src/Catalog.API/Services/CatalogAI.cs` | ソース | AI埋め込み生成 |
| CatalogContext.cs | `src/Catalog.API/Infrastructure/CatalogContext.cs` | ソース | DBコンテキスト |
