# 機能設計書 13-商品削除

## 概要

本ドキュメントは、eShopシステムにおける商品削除機能の詳細設計を記載する。

### 本機能の処理概要

本機能は、指定されたIDの商品をカタログから削除するためのREST APIエンドポイントを提供する。物理削除（データベースからの完全削除）を行い、削除成功時はNoContent (204)を返却する。

**業務上の目的・背景**：EC運営において、取り扱い終了商品、誤って登録した商品、販売停止商品などを削除する必要がある。本機能により、管理者は不要な商品をカタログから完全に削除できる。削除された商品はセマンティック検索の対象からも除外される。

**機能の利用シーン**：
- 取り扱い終了商品をカタログから削除する場合
- 誤って登録した商品を削除する場合
- 季節商品やキャンペーン終了商品を削除する場合
- テストデータをクリーンアップする場合

**主要な処理内容**：
1. HTTPリクエストから削除対象の商品IDを受け取る
2. データベースから対象商品を検索する
3. 商品が存在しない場合はNotFound (404)を返却する
4. 商品をデータベースから削除する
5. NoContent (204)レスポンスを返却する

**関連システム・外部連携**：特になし（削除は単一サービス内で完結）

**権限による制御**：本APIエンドポイントは管理者向けであり、本番環境では適切な認可設定が必要である。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は管理API経由で使用され、直接関連する画面は存在しない |

## 機能種別

CRUD操作（Delete）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| id | int | Yes | 削除対象の商品ID | パスパラメータとして指定 |

### 入力データソース

パスパラメータ: URL（`/api/catalog/items/{id}`）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| - | - | 成功時はレスポンスボディなし |

### 出力先

- HTTPレスポンス（204 NoContent / 404 NotFound）
- PostgreSQL CatalogItemsテーブル（DELETE）

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエスト受信
   └─ DELETE /api/catalog/items/{id}

2. 対象商品の取得
   └─ CatalogContext.CatalogItems.SingleOrDefault(x => x.Id == id)

3. 存在チェック
   └─ 商品が存在しない場合は NotFound (404) を返却

4. 商品削除
   └─ CatalogContext.CatalogItems.Remove(item)

5. データベース保存
   └─ SaveChangesAsync()

6. レスポンス返却
   └─ NoContent (204)
```

### フローチャート

```mermaid
flowchart TD
    A[DELETE /api/catalog/items/id] --> B[対象商品取得]
    B --> C{商品存在?}
    C -->|No| D[NotFound 404]
    C -->|Yes| E[商品削除]
    E --> F[DB保存]
    F --> G[NoContent 204]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-13-01 | 存在確認 | 削除対象の商品が存在しない場合は404エラーを返す | 商品削除時 |
| BR-13-02 | 物理削除 | 商品は論理削除ではなく物理削除される | 商品削除時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 商品削除 | CatalogItems | DELETE | 商品レコードを物理削除 |

### テーブル別操作詳細

#### CatalogItems

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id | WHERE Id = @id | 削除対象取得 |
| DELETE | - | WHERE Id = @id | 物理削除 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 404 | Not Found | 指定IDの商品が存在しない場合 | 正しい商品IDを指定 |
| 500 | Internal Server Error | DB削除失敗時 | サーバーログを確認し再試行 |

### リトライ仕様

本機能にリトライ機能は実装されていない。

## トランザクション仕様

- SaveChangesAsync()による単一トランザクションで削除をコミット
- エラー発生時は自動ロールバック

## パフォーマンス要件

- 単一レコードの削除のため、高速に処理される
- インデックスが有効に機能する

## セキュリティ考慮事項

- 本APIは管理用途のため、本番環境では認証・認可の設定が必要
- 削除操作は取り消し不可のため、慎重な運用が必要
- 関連する注文履歴等への影響を考慮する必要がある

## 備考

- 物理削除のため、削除後のデータ復旧は不可能
- 削除前に関連データ（注文履歴、買い物かご等）への影響を確認することを推奨
- 本機能は削除イベントを発行しないため、他サービスへの通知は行われない

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | 削除対象となる商品エンティティの構造 |

**読解のコツ**: 削除処理はEntity Framework CoreのRemoveメソッドを使用しており、関連エンティティの削除（Cascade Delete）の設定を確認することが重要。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | DeleteItemByIdが削除処理のエントリーポイント |

**主要処理フロー**:
1. **107-110行目**: `api.MapDelete("/items/{id:int}", DeleteItemById)` - エンドポイント定義
2. **390-404行目**: `DeleteItemById`メソッド - 削除処理本体
3. **394行目**: SingleOrDefaultで対象商品取得
4. **396-399行目**: 存在チェックとNotFound返却
5. **401行目**: Removeで削除マーク
6. **402行目**: SaveChangesAsyncで永続化
7. **403行目**: NoContent返却

#### Step 3: データベースコンテキストを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CatalogContext.cs | `src/Catalog.API/Infrastructure/CatalogContext.cs` | EFCoreのDbSetとRemove操作 |

### プログラム呼び出し階層図

```
CatalogApi.DeleteItemById (エントリーポイント)
    │
    ├─ CatalogContext.CatalogItems.SingleOrDefault()
    │      └─ 対象商品取得
    │
    ├─ CatalogContext.CatalogItems.Remove(item)
    │      └─ 削除マーク
    │
    └─ CatalogContext.SaveChangesAsync()
           └─ PostgreSQL DELETE
```

### データフロー図

```
[入力]               [処理]                    [出力]

HTTP DELETE      CatalogApi.DeleteItemById   PostgreSQL
Path ID ──────▶    商品取得  ───────────────▶ CatalogItems
                      ↓                       (DELETE)
                  存在チェック                    │
                      │                         ▼
                 ┌────┴────┐              HTTP 204
                 ↓         ↓              NoContent
              存在する   存在しない
                 ↓         ↓
              削除実行   NotFound 404
                 ↓
           NoContent 204
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | ソース | APIエンドポイント定義・削除処理 |
| CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | ソース | 商品エンティティモデル |
| CatalogContext.cs | `src/Catalog.API/Infrastructure/CatalogContext.cs` | ソース | EF Coreデータベースコンテキスト |
| CatalogServices.cs | `src/Catalog.API/Model/CatalogServices.cs` | ソース | 依存性注入用サービスクラス |
