# 機能設計書 14-かご取得

## 概要

本ドキュメントは、eShopシステムにおける買い物かご取得機能の詳細設計を記載する。

### 本機能の処理概要

本機能は、ユーザーの買い物かご内容を取得するためのgRPCサービスを提供する。認証されたユーザーのIDに基づいてRedisから買い物かごデータを取得し、商品IDと数量のリストを返却する。未認証ユーザーの場合は空のレスポンスを返却する。

**業務上の目的・背景**：ECサイトにおいて、ユーザーが買い物かごに追加した商品を確認することは購買フローの重要なステップである。本機能により、ユーザーは現在の買い物かごの状態を確認でき、商品詳細画面やカート画面から買い物かごの内容を表示できる。

**機能の利用シーン**：
- カート画面で買い物かごの内容一覧を表示する場合
- 商品詳細画面でカート内商品数を表示する場合
- チェックアウト前にカート内容を確認する場合
- ヘッダー等でカートアイコンのバッジ数を表示する場合

**主要な処理内容**：
1. gRPCリクエストを受信する
2. サーバーコールコンテキストからユーザーIDを取得する
3. ユーザーIDが空の場合は空のレスポンスを返却する
4. RedisからユーザーIDをキーにして買い物かごデータを取得する
5. 取得したデータをgRPCレスポンス形式にマッピングして返却する

**関連システム・外部連携**：
- Redis（買い物かごデータの永続化）
- Identity認証（JWTトークンからのユーザーID取得）

**権限による制御**：`[AllowAnonymous]`属性により未認証アクセスも許可されているが、未認証の場合は空のレスポンスを返却する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 2 | 商品詳細画面 | API連携 | BasketState.GetBasketItemsAsyncでカート内商品数を取得 |
| 3 | カート画面 | 主機能 | BasketState.GetBasketItemsAsyncでカート内容一覧を取得 |
| 4 | チェックアウト画面 | 補助機能 | カート内容の存在確認（空チェック） |

## 機能種別

CRUD操作（Read） / gRPCサービス

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| - | - | - | GetBasketRequestは空のメッセージ | - |

**暗黙的な入力**

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| userId | string | No | 認証コンテキストから取得されるユーザーID（subクレーム） | - |

### 入力データソース

- gRPCリクエスト: GetBasketRequest（Protocol Buffers）
- 認証コンテキスト: JWTトークンのsubクレーム

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| items | repeated BasketItem | 買い物かごアイテムのリスト |
| items[].product_id | int32 | 商品ID |
| items[].quantity | int32 | 数量 |

### 出力先

gRPCレスポンス（CustomerBasketResponse）

## 処理フロー

### 処理シーケンス

```
1. gRPCリクエスト受信
   └─ GetBasket(GetBasketRequest)

2. ユーザーID取得
   └─ context.GetUserIdentity() でJWTのsubクレームを取得

3. 認証チェック
   └─ ユーザーIDが空の場合は空のCustomerBasketResponseを返却

4. Redis取得
   └─ repository.GetBasketAsync(userId)

5. 存在チェック
   └─ データがない場合は空のCustomerBasketResponseを返却

6. レスポンスマッピング
   └─ MapToCustomerBasketResponse(data)で形式変換

7. レスポンス返却
   └─ CustomerBasketResponse
```

### フローチャート

```mermaid
flowchart TD
    A[gRPC GetBasket] --> B[ユーザーID取得]
    B --> C{ユーザーID存在?}
    C -->|No| D[空レスポンス返却]
    C -->|Yes| E[Redis取得]
    E --> F{データ存在?}
    F -->|No| D
    F -->|Yes| G[レスポンスマッピング]
    G --> H[CustomerBasketResponse返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-14-01 | 匿名アクセス許可 | 未認証ユーザーもAPIコール可能だが、空のレスポンスを返す | AllowAnonymous属性 |
| BR-14-02 | ユーザー識別 | JWTのsubクレームでユーザーを識別する | 認証済みリクエスト |
| BR-14-03 | データ不存在時 | 買い物かごが存在しない場合は空のリストを返す | Redis取得結果がnull |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象ストア | 操作種別 | 概要 |
|-----|-------------|---------|------|
| かご取得 | Redis | GET | キー/basket/{userId}からデータ取得 |

### テーブル別操作詳細

#### Redis

| 操作 | キー | 値 | 備考 |
|-----|------|-----|------|
| GET | /basket/{userId} | JSON (CustomerBasket) | StringGetLeaseAsyncで取得 |

```json
// 格納形式
{
  "BuyerId": "user-id",
  "Items": [
    {
      "Id": "item-id",
      "ProductId": 1,
      "ProductName": "商品名",
      "UnitPrice": 100.00,
      "OldUnitPrice": 100.00,
      "Quantity": 2,
      "PictureUrl": "url"
    }
  ]
}
```

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 空レスポンス | ユーザーIDが空の場合 | 認証後に再試行 |
| - | 空レスポンス | Redisにデータがない場合 | 買い物かごに商品を追加 |
| INTERNAL | gRPCエラー | Redis接続失敗時 | サーバーログを確認 |

### リトライ仕様

本機能にリトライ機能は実装されていない。Redis接続失敗時はgRPCエラーとして返却される。

## トランザクション仕様

Redis GETは単一操作のため、トランザクション管理は不要。

## パフォーマンス要件

- Redisへのアクセスのため、ミリ秒単位の高速レスポンスが期待される
- キーによる直接アクセスのため、データ量に依存しない

## セキュリティ考慮事項

- JWTトークンからのユーザーID取得により、他ユーザーの買い物かごにはアクセス不可
- AllowAnonymous属性により未認証アクセスは許可されるが、データは返却されない

## 備考

- gRPCサービスとして実装されており、REST APIは提供されていない
- CustomerBasketResponseにはproduct_idとquantityのみが含まれ、商品名や価格は含まれない
- 詳細な商品情報が必要な場合は、別途Catalog.APIから取得する必要がある

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | basket.proto | `src/Basket.API/Proto/basket.proto` | gRPCメッセージ定義（GetBasketRequest, CustomerBasketResponse, BasketItem） |
| 1-2 | CustomerBasket.cs | `src/Basket.API/Model/CustomerBasket.cs` | 内部データモデル |
| 1-3 | BasketItem.cs | `src/Basket.API/Model/BasketItem.cs` | かごアイテムの詳細プロパティ |

**読解のコツ**: gRPCのBasketItemとモデルのBasketItemは別クラス。gRPC版はproduct_idとquantityのみ、モデル版はProductName、UnitPrice等も含む。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | BasketService.cs | `src/Basket.API/Grpc/BasketService.cs` | gRPCサービス実装、GetBasketメソッド |

**主要処理フロー**:
1. **12行目**: `[AllowAnonymous]` - 匿名アクセス許可
2. **13行目**: `GetBasket`メソッド定義
3. **15行目**: `context.GetUserIdentity()` - ユーザーID取得
4. **16-19行目**: ユーザーID空チェック
5. **26行目**: `repository.GetBasketAsync(userId)` - Redis取得
6. **28-31行目**: 存在チェックとレスポンス変換

#### Step 3: リポジトリ層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | IBasketRepository.cs | `src/Basket.API/Repositories/IBasketRepository.cs` | リポジトリインターフェース |
| 3-2 | RedisBasketRepository.cs | `src/Basket.API/Repositories/RedisBasketRepository.cs` | Redis実装 |

**主要処理フロー**:
- **13行目**: `BasketKeyPrefix = "/basket/"` - キープレフィックス
- **16行目**: `GetBasketKey(string userId)` - キー生成
- **23-32行目**: `GetBasketAsync` - Redis GETとJSON デシリアライズ

#### Step 4: 認証拡張を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | ServerCallContextIdentityExtensions.cs | `src/Basket.API/Extensions/ServerCallContextIdentityExtensions.cs` | ユーザーID取得拡張メソッド |

**主要処理フロー**:
- **7行目**: `GetUserIdentity` - subクレームからユーザーID取得

### プログラム呼び出し階層図

```
BasketService.GetBasket (gRPCエントリーポイント)
    │
    ├─ context.GetUserIdentity()
    │      └─ ServerCallContextIdentityExtensions
    │             └─ JWTのsubクレーム取得
    │
    ├─ repository.GetBasketAsync(userId)
    │      └─ RedisBasketRepository
    │             ├─ GetBasketKey(userId)
    │             ├─ _database.StringGetLeaseAsync()
    │             └─ JsonSerializer.Deserialize()
    │
    └─ MapToCustomerBasketResponse(data)
           └─ モデルからgRPCレスポンスへ変換
```

### データフロー図

```
[入力]               [処理]                    [出力]

gRPC Request     BasketService.GetBasket    gRPC Response
(GetBasketRequest)      ↓                   (CustomerBasketResponse)
       +           ユーザーID取得                 │
JWT Token              ↓                        │
                   Redis GET                    │
                      ↓                         │
                 ┌────┴────┐                    │
                 ↓         ↓                    │
              あり       なし                   │
                 ↓         ↓                    │
            マッピング  空レスポンス            │
                 │         │                    │
                 └────┬────┘                    │
                      ▼                         │
              CustomerBasketResponse ───────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| basket.proto | `src/Basket.API/Proto/basket.proto` | 定義 | gRPCサービス・メッセージ定義 |
| BasketService.cs | `src/Basket.API/Grpc/BasketService.cs` | ソース | gRPCサービス実装 |
| IBasketRepository.cs | `src/Basket.API/Repositories/IBasketRepository.cs` | ソース | リポジトリインターフェース |
| RedisBasketRepository.cs | `src/Basket.API/Repositories/RedisBasketRepository.cs` | ソース | Redis実装 |
| CustomerBasket.cs | `src/Basket.API/Model/CustomerBasket.cs` | ソース | 内部データモデル |
| BasketItem.cs | `src/Basket.API/Model/BasketItem.cs` | ソース | かごアイテムモデル |
| ServerCallContextIdentityExtensions.cs | `src/Basket.API/Extensions/ServerCallContextIdentityExtensions.cs` | ソース | 認証拡張メソッド |
