# 機能設計書 15-かご更新

## 概要

本ドキュメントは、eShopシステムにおける買い物かご更新機能の詳細設計を記載する。

### 本機能の処理概要

本機能は、ユーザーの買い物かごに商品を追加・数量変更するためのgRPCサービスを提供する。認証されたユーザーのIDに基づいてRedisに買い物かごデータを保存し、更新後の買い物かご内容を返却する。

**業務上の目的・背景**：ECサイトにおいて、ユーザーが購入したい商品を一時的に保存しておく買い物かご機能は必須である。本機能により、ユーザーは商品詳細画面から商品をカートに追加したり、カート画面で数量を変更したりできる。AIチャットボットからもカート操作が可能である。

**機能の利用シーン**：
- 商品詳細画面で「カートに追加」ボタンを押した場合
- カート画面で商品の数量を変更した場合
- AIチャットボットの指示でカートに商品を追加する場合
- カート画面で商品を削除（数量を0に）した場合

**主要な処理内容**：
1. gRPCリクエストを受信する
2. サーバーコールコンテキストからユーザーIDを取得する
3. 未認証の場合は認証エラー（UNAUTHENTICATED）を返却する
4. リクエストの商品リストからCustomerBasketオブジェクトを生成する
5. Redisに買い物かごデータを保存する
6. 保存失敗時はNotFound（NOT_FOUND）エラーを返却する
7. 更新後の買い物かごデータをレスポンスとして返却する

**関連システム・外部連携**：
- Redis（買い物かごデータの永続化）
- Identity認証（JWTトークンからのユーザーID取得）

**権限による制御**：認証が必須。未認証アクセスはUNAUTHENTICATEDエラーとなる。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 2 | 商品詳細画面 | API連携 | AddToCartAsyncでBasketState.AddAsyncを呼び出しカートに追加 |
| 3 | カート画面 | API連携 | UpdateQuantityAsyncでBasketState.SetQuantityAsyncを呼び出し数量変更 |
| 8 | チャットボット | API連携 | BasketStateでカート操作支援 |

## 機能種別

CRUD操作（Update/Create） / gRPCサービス

## 入力仕様

### 入力パラメータ

**gRPCメッセージ（UpdateBasketRequest）**

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| items | repeated BasketItem | Yes | 買い物かごアイテムのリスト | - |
| items[].product_id | int32 | Yes | 商品ID | - |
| items[].quantity | int32 | Yes | 数量 | - |

**暗黙的な入力**

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| userId | string | Yes | 認証コンテキストから取得されるユーザーID（subクレーム） | 空の場合UNAUTHENTICATED |

### 入力データソース

- gRPCリクエスト: UpdateBasketRequest（Protocol Buffers）
- 認証コンテキスト: JWTトークンのsubクレーム

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| items | repeated BasketItem | 更新後の買い物かごアイテムのリスト |
| items[].product_id | int32 | 商品ID |
| items[].quantity | int32 | 数量 |

### 出力先

gRPCレスポンス（CustomerBasketResponse）

## 処理フロー

### 処理シーケンス

```
1. gRPCリクエスト受信
   └─ UpdateBasket(UpdateBasketRequest)

2. ユーザーID取得
   └─ context.GetUserIdentity() でJWTのsubクレームを取得

3. 認証チェック
   └─ ユーザーIDが空の場合はUNAUTHENTICATEDエラー

4. データマッピング
   └─ MapToCustomerBasket(userId, request) で内部モデルに変換

5. Redis保存
   └─ repository.UpdateBasketAsync(customerBasket)

6. 保存結果チェック
   └─ 保存失敗時はNOT_FOUNDエラー

7. レスポンスマッピング
   └─ MapToCustomerBasketResponse(response)

8. レスポンス返却
   └─ CustomerBasketResponse
```

### フローチャート

```mermaid
flowchart TD
    A[gRPC UpdateBasket] --> B[ユーザーID取得]
    B --> C{認証済み?}
    C -->|No| D[UNAUTHENTICATED]
    C -->|Yes| E[内部モデル変換]
    E --> F[Redis保存]
    F --> G{保存成功?}
    G -->|No| H[NOT_FOUND]
    G -->|Yes| I[レスポンス変換]
    I --> J[CustomerBasketResponse返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-15-01 | 認証必須 | 未認証ユーザーはUNAUTHENTICATEDエラー | ユーザーID空 |
| BR-15-02 | 全置換方式 | 買い物かごの内容は全て置換される（差分更新ではない） | 更新時 |
| BR-15-03 | 保存失敗時エラー | Redis保存失敗時はNOT_FOUNDエラーを返す | 保存結果null |

### 計算ロジック

特になし（買い物かご内容は全置換）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象ストア | 操作種別 | 概要 |
|-----|-------------|---------|------|
| かご更新 | Redis | SET | キー/basket/{userId}にデータを保存 |
| かご取得 | Redis | GET | 保存後に再取得して返却 |

### テーブル別操作詳細

#### Redis

| 操作 | キー | 値 | 備考 |
|-----|------|-----|------|
| SET | /basket/{userId} | JSON (CustomerBasket) | StringSetAsyncで保存 |
| GET | /basket/{userId} | JSON (CustomerBasket) | 保存後に再取得 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| UNAUTHENTICATED | gRPCエラー | ユーザーIDが空の場合 | 認証後に再試行 |
| NOT_FOUND | gRPCエラー | Redis保存失敗時 | サーバーログを確認 |
| INTERNAL | gRPCエラー | Redis接続失敗時 | サーバーログを確認 |

### リトライ仕様

本機能にリトライ機能は実装されていない。

## トランザクション仕様

Redis SET/GETは単一操作のため、アトミック性は操作単位で保証される。

## パフォーマンス要件

- Redisへのアクセスのため、ミリ秒単位の高速レスポンスが期待される
- JSON シリアライズ/デシリアライズのオーバーヘッドあり

## セキュリティ考慮事項

- 認証必須により、他ユーザーの買い物かごの更新は不可
- JWTトークンの検証により、なりすましを防止

## 備考

- 買い物かごの更新は全置換方式のため、クライアントは現在のかご内容に変更を加えた完全なリストを送信する必要がある
- gRPCのBasketItemにはproduct_idとquantityのみが含まれる
- 商品名や価格などの詳細情報は、クライアント側でCatalog.APIから取得する

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | basket.proto | `src/Basket.API/Proto/basket.proto` | gRPCメッセージ定義（UpdateBasketRequest, CustomerBasketResponse） |
| 1-2 | CustomerBasket.cs | `src/Basket.API/Model/CustomerBasket.cs` | 内部データモデル |
| 1-3 | BasketItem.cs | `src/Basket.API/Model/BasketItem.cs` | かごアイテムモデルとバリデーション |

**読解のコツ**: BasketItem.csにはQuantityのバリデーション（1未満は無効）が含まれている。IValidatableObjectインターフェースを実装。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | BasketService.cs | `src/Basket.API/Grpc/BasketService.cs` | gRPCサービス実装、UpdateBasketメソッド |

**主要処理フロー**:
1. **36行目**: `UpdateBasket`メソッド定義
2. **38行目**: `context.GetUserIdentity()` - ユーザーID取得
3. **39-42行目**: 認証チェック（ThrowNotAuthenticated）
4. **49行目**: `MapToCustomerBasket` - 内部モデル変換
5. **50行目**: `repository.UpdateBasketAsync` - Redis保存
6. **51-54行目**: 保存失敗チェック（ThrowBasketDoesNotExist）
7. **56行目**: `MapToCustomerBasketResponse` - レスポンス変換

#### Step 3: リポジトリ層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | RedisBasketRepository.cs | `src/Basket.API/Repositories/RedisBasketRepository.cs` | Redis保存実装 |

**主要処理フロー**:
- **34-48行目**: `UpdateBasketAsync` - JSON シリアライズとRedis SET
- **36行目**: `JsonSerializer.SerializeToUtf8Bytes` - シリアライズ
- **37行目**: `_database.StringSetAsync` - Redis SET
- **47行目**: `GetBasketAsync` - 保存後の再取得

### プログラム呼び出し階層図

```
BasketService.UpdateBasket (gRPCエントリーポイント)
    │
    ├─ context.GetUserIdentity()
    │      └─ ServerCallContextIdentityExtensions
    │
    ├─ MapToCustomerBasket(userId, request)
    │      └─ gRPCメッセージから内部モデルへ変換
    │
    ├─ repository.UpdateBasketAsync(customerBasket)
    │      └─ RedisBasketRepository
    │             ├─ JsonSerializer.SerializeToUtf8Bytes()
    │             ├─ _database.StringSetAsync()
    │             └─ GetBasketAsync()
    │
    └─ MapToCustomerBasketResponse(response)
           └─ 内部モデルからgRPCレスポンスへ変換
```

### データフロー図

```
[入力]               [処理]                    [出力]

gRPC Request     BasketService.UpdateBasket  gRPC Response
(UpdateBasketRequest)  ↓                     (CustomerBasketResponse)
       +           認証チェック                   │
JWT Token              ↓                        │
                 内部モデル変換                  │
                      ↓                         │
                   Redis SET                    │
                      ↓                         │
                   Redis GET                    │
                      ↓                         │
                レスポンス変換 ─────────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| basket.proto | `src/Basket.API/Proto/basket.proto` | 定義 | gRPCサービス・メッセージ定義 |
| BasketService.cs | `src/Basket.API/Grpc/BasketService.cs` | ソース | gRPCサービス実装 |
| IBasketRepository.cs | `src/Basket.API/Repositories/IBasketRepository.cs` | ソース | リポジトリインターフェース |
| RedisBasketRepository.cs | `src/Basket.API/Repositories/RedisBasketRepository.cs` | ソース | Redis実装 |
| CustomerBasket.cs | `src/Basket.API/Model/CustomerBasket.cs` | ソース | 内部データモデル |
| BasketItem.cs | `src/Basket.API/Model/BasketItem.cs` | ソース | かごアイテムモデル |
| ServerCallContextIdentityExtensions.cs | `src/Basket.API/Extensions/ServerCallContextIdentityExtensions.cs` | ソース | 認証拡張メソッド |
