# 機能設計書 16-かご削除

## 概要

本ドキュメントは、eShopシステムにおける買い物かご削除機能の詳細設計を記載する。

### 本機能の処理概要

本機能は、ユーザーの買い物かごの内容を全て削除するためのgRPCサービスを提供する。認証されたユーザーのIDに基づいてRedisから買い物かごデータを削除する。チェックアウト完了後のカートクリアや、ユーザーがカートを空にしたい場合に使用される。

**業務上の目的・背景**：ECサイトにおいて、注文確定後に買い物かごをクリアすることは、重複注文を防ぎ、次回の買い物のために空のカートを用意するために必要である。また、ユーザーが不要になった商品をまとめて削除したい場合にも使用される。

**機能の利用シーン**：
- チェックアウト完了後にカートをクリアする場合
- ユーザーが「カートを空にする」操作を行った場合
- 注文開始時の統合イベント処理で自動的にカートをクリアする場合

**主要な処理内容**：
1. gRPCリクエストを受信する
2. サーバーコールコンテキストからユーザーIDを取得する
3. 未認証の場合は認証エラー（UNAUTHENTICATED）を返却する
4. Redisから該当ユーザーの買い物かごデータを削除する
5. 空のDeleteBasketResponseを返却する

**関連システム・外部連携**：
- Redis（買い物かごデータの削除）
- Identity認証（JWTトークンからのユーザーID取得）
- 統合イベント（OrderStartedIntegrationEvent受信時に自動実行）

**権限による制御**：認証が必須。未認証アクセスはUNAUTHENTICATEDエラーとなる。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 4 | チェックアウト画面 | 補助機能 | 注文完了後にBasketState.DeleteBasketAsyncでカート削除 |

## 機能種別

CRUD操作（Delete） / gRPCサービス

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| - | - | - | DeleteBasketRequestは空のメッセージ | - |

**暗黙的な入力**

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| userId | string | Yes | 認証コンテキストから取得されるユーザーID（subクレーム） | 空の場合UNAUTHENTICATED |

### 入力データソース

- gRPCリクエスト: DeleteBasketRequest（Protocol Buffers）
- 認証コンテキスト: JWTトークンのsubクレーム

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| - | - | DeleteBasketResponseは空のメッセージ |

### 出力先

gRPCレスポンス（DeleteBasketResponse）

## 処理フロー

### 処理シーケンス

```
1. gRPCリクエスト受信
   └─ DeleteBasket(DeleteBasketRequest)

2. ユーザーID取得
   └─ context.GetUserIdentity() でJWTのsubクレームを取得

3. 認証チェック
   └─ ユーザーIDが空の場合はUNAUTHENTICATEDエラー

4. Redis削除
   └─ repository.DeleteBasketAsync(userId)

5. レスポンス返却
   └─ 空のDeleteBasketResponse
```

### フローチャート

```mermaid
flowchart TD
    A[gRPC DeleteBasket] --> B[ユーザーID取得]
    B --> C{認証済み?}
    C -->|No| D[UNAUTHENTICATED]
    C -->|Yes| E[Redis削除]
    E --> F[空のDeleteBasketResponse返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-16-01 | 認証必須 | 未認証ユーザーはUNAUTHENTICATEDエラー | ユーザーID空 |
| BR-16-02 | 冪等性 | 既に削除済み（存在しない）でもエラーにならない | 削除時 |
| BR-16-03 | 全削除 | 買い物かご内の全商品が削除される | 削除時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象ストア | 操作種別 | 概要 |
|-----|-------------|---------|------|
| かご削除 | Redis | DEL | キー/basket/{userId}を削除 |

### テーブル別操作詳細

#### Redis

| 操作 | キー | 値 | 備考 |
|-----|------|-----|------|
| DEL | /basket/{userId} | - | KeyDeleteAsyncで削除 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| UNAUTHENTICATED | gRPCエラー | ユーザーIDが空の場合 | 認証後に再試行 |
| INTERNAL | gRPCエラー | Redis接続失敗時 | サーバーログを確認 |

### リトライ仕様

本機能にリトライ機能は実装されていない。

## トランザクション仕様

Redis DELは単一操作のため、トランザクション管理は不要。

## パフォーマンス要件

- Redisへのアクセスのため、ミリ秒単位の高速レスポンスが期待される
- キーによる直接削除のため、データ量に依存しない

## セキュリティ考慮事項

- 認証必須により、他ユーザーの買い物かごの削除は不可
- JWTトークンの検証により、なりすましを防止

## 備考

- 削除対象のカートが存在しない場合でもエラーにはならない（冪等性）
- 注文開始時にOrderStartedIntegrationEventが発行されると、Basket.API側のイベントハンドラがこの機能を呼び出してカートをクリアする

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | basket.proto | `src/Basket.API/Proto/basket.proto` | gRPCメッセージ定義（DeleteBasketRequest, DeleteBasketResponse） |

**読解のコツ**: DeleteBasketRequestとDeleteBasketResponseは共に空のメッセージ。ユーザーIDは認証コンテキストから取得される。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | BasketService.cs | `src/Basket.API/Grpc/BasketService.cs` | gRPCサービス実装、DeleteBasketメソッド |

**主要処理フロー**:
1. **59行目**: `DeleteBasket`メソッド定義
2. **61行目**: `context.GetUserIdentity()` - ユーザーID取得
3. **62-65行目**: 認証チェック（ThrowNotAuthenticated）
4. **67行目**: `repository.DeleteBasketAsync(userId)` - Redis削除
5. **68行目**: 空のDeleteBasketResponse返却

#### Step 3: リポジトリ層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | RedisBasketRepository.cs | `src/Basket.API/Repositories/RedisBasketRepository.cs` | Redis削除実装 |

**主要処理フロー**:
- **18-21行目**: `DeleteBasketAsync` - Redis DEL実行
- **20行目**: `_database.KeyDeleteAsync` - キー削除

#### Step 4: 統合イベントハンドラを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | OrderStartedIntegrationEventHandler.cs | `src/Basket.API/IntegrationEvents/EventHandling/OrderStartedIntegrationEventHandler.cs` | 注文開始時の自動削除 |

**読解のコツ**: 注文作成時にOrdering.APIからOrderStartedIntegrationEventが発行され、このハンドラがDeleteBasketAsyncを呼び出してカートをクリアする。

### プログラム呼び出し階層図

```
BasketService.DeleteBasket (gRPCエントリーポイント)
    │
    ├─ context.GetUserIdentity()
    │      └─ ServerCallContextIdentityExtensions
    │
    └─ repository.DeleteBasketAsync(userId)
           └─ RedisBasketRepository
                  └─ _database.KeyDeleteAsync()

--- 別経路（統合イベント経由）---

OrderStartedIntegrationEventHandler.Handle
    │
    └─ _repository.DeleteBasketAsync(userId)
           └─ RedisBasketRepository
                  └─ _database.KeyDeleteAsync()
```

### データフロー図

```
[入力]               [処理]                    [出力]

gRPC Request     BasketService.DeleteBasket  gRPC Response
(DeleteBasketRequest)  ↓                     (DeleteBasketResponse)
       +           認証チェック                   │
JWT Token              ↓                        │
                   Redis DEL                    │
                      ↓                         │
                空レスポンス ───────────────────┘

--- 別経路 ---

統合イベント     OrderStartedIntegrationEventHandler
(OrderStarted)         ↓
                   Redis DEL
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| basket.proto | `src/Basket.API/Proto/basket.proto` | 定義 | gRPCサービス・メッセージ定義 |
| BasketService.cs | `src/Basket.API/Grpc/BasketService.cs` | ソース | gRPCサービス実装 |
| IBasketRepository.cs | `src/Basket.API/Repositories/IBasketRepository.cs` | ソース | リポジトリインターフェース |
| RedisBasketRepository.cs | `src/Basket.API/Repositories/RedisBasketRepository.cs` | ソース | Redis実装 |
| OrderStartedIntegrationEventHandler.cs | `src/Basket.API/IntegrationEvents/EventHandling/OrderStartedIntegrationEventHandler.cs` | ソース | 統合イベントハンドラ |
| ServerCallContextIdentityExtensions.cs | `src/Basket.API/Extensions/ServerCallContextIdentityExtensions.cs` | ソース | 認証拡張メソッド |
