# 機能設計書 18-注文下書き作成

## 概要

本ドキュメントは、eShopシステムにおける注文下書き作成機能の詳細設計を記載する。

### 本機能の処理概要

本機能は、注文の下書き（Draft）を作成するためのREST APIエンドポイントを提供する。買い物かごの商品リストを受け取り、注文明細と合計金額を算出した下書きオブジェクトを返却する。この処理はデータベースに永続化せず、注文プレビューや確認画面での表示に使用される。

**業務上の目的・背景**：ユーザーがチェックアウトを行う前に、注文の詳細（商品名、数量、単価、合計金額など）を確認できるようにするための機能である。買い物かごからカート情報を受け取り、正式な注文データ形式に変換して表示用の情報を提供する。

**機能の利用シーン**：
- チェックアウト画面で注文内容をプレビュー表示する場合
- 注文確定前に合計金額を確認する場合
- カート内容から注文明細形式に変換する場合

**主要な処理内容**：
1. HTTPリクエストから購入者IDと商品リストを受け取る
2. Order.NewDraft()でドラフト注文を生成する
3. 各BasketItemをOrderItemに変換し追加する
4. OrderDraftDTOに変換して返却する（合計金額計算を含む）

**関連システム・外部連携**：特になし（メモリ内処理のみ）

**権限による制御**：特になし（明示的な認可属性なし）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | チェックアウト前のプレビュー表示に使用される可能性があるが、直接関連する画面は特定されていない |

## 機能種別

計算処理 / データ変換

## 入力仕様

### 入力パラメータ

**リクエストボディ（CreateOrderDraftCommand）**

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| BuyerId | string | Yes | 購入者ID | - |
| Items | IEnumerable<BasketItem> | Yes | 買い物かごアイテムリスト | - |

**BasketItem構造**

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Id | string | No | アイテムID | - |
| ProductId | int | Yes | 商品ID | - |
| ProductName | string | Yes | 商品名 | - |
| UnitPrice | decimal | Yes | 単価 | - |
| OldUnitPrice | decimal | No | 旧単価 | - |
| Quantity | int | Yes | 数量 | - |
| PictureUrl | string | No | 商品画像URL | - |

### 入力データソース

リクエストボディ: HTTP POSTリクエストボディ（JSON形式）

## 出力仕様

### 出力データ

**OrderDraftDTO**

| 項目名 | 型 | 説明 |
|--------|-----|------|
| OrderItems | IEnumerable<OrderItemDTO> | 注文明細リスト |
| Total | decimal | 合計金額 |

**OrderItemDTO構造**

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ProductId | int | 商品ID |
| ProductName | string | 商品名 |
| UnitPrice | decimal | 単価 |
| Discount | decimal | 割引額 |
| Units | int | 数量 |
| PictureUrl | string | 商品画像URL |

### 出力先

HTTPレスポンス（200 OK + OrderDraftDTO）

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエスト受信
   └─ POST /api/orders/draft

2. CreateOrderDraftCommand生成
   └─ リクエストボディからコマンドオブジェクト生成

3. MediatR経由でハンドラ実行
   └─ CreateOrderDraftCommandHandler.Handle()

4. ドラフト注文生成
   └─ Order.NewDraft() でドラフトフラグ付きの空注文を生成

5. BasketItemからOrderItemへ変換
   └─ 各アイテムを order.AddOrderItem() で追加

6. OrderDraftDTOに変換
   └─ OrderDraftDTO.FromOrder(order) で合計金額計算を含めて変換

7. レスポンス返却
   └─ OrderDraftDTO
```

### フローチャート

```mermaid
flowchart TD
    A[POST /api/orders/draft] --> B[CreateOrderDraftCommand生成]
    B --> C[MediatR送信]
    C --> D[Order.NewDraft生成]
    D --> E[BasketItem変換]
    E --> F[order.AddOrderItem × N]
    F --> G[OrderDraftDTO.FromOrder]
    G --> H[合計金額計算]
    H --> I[OrderDraftDTO返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-18-01 | 非永続化 | 下書き注文はデータベースに保存されない | 常に |
| BR-18-02 | ドラフトフラグ | Order._isDraftがtrueに設定される | NewDraft時 |
| BR-18-03 | 合計金額計算 | 合計 = Σ(単価 × 数量) | DTO変換時 |

### 計算ロジック

注文合計金額:
```
Total = Σ (OrderItem.UnitPrice × OrderItem.Units)
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし（メモリ内処理のみ） |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | Bad Request | リクエストボディが不正な場合 | リクエストを修正 |
| 500 | Internal Server Error | 内部エラー発生時 | サーバーログを確認 |

### リトライ仕様

状態を持たないため、リトライは安全に実行可能。

## トランザクション仕様

データベース操作がないため、トランザクション管理は不要。

## パフォーマンス要件

- メモリ内処理のみのため、非常に高速（ミリ秒以下）
- 外部リソースアクセスなし

## セキュリティ考慮事項

- 特になし（データの永続化なし）
- 価格情報はクライアントから受け取るため、サーバー側での再検証が推奨される

## 備考

- 本機能は注文のプレビュー目的であり、実際の注文作成は別機能（No.17 注文作成）で行う
- Order._isDraftフラグは設定されるが、現在のコードでは参照されていない

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CreateOrderDraftCommand.cs | `src/Ordering.API/Application/Commands/CreateOrderDraftCommand.cs` | コマンドオブジェクト定義（record型） |
| 1-2 | CreateOrderDraftCommandHandler.cs | `src/Ordering.API/Application/Commands/CreateOrderDraftCommandHandler.cs` | OrderDraftDTO、OrderItemDTO定義 |
| 1-3 | BasketItem.cs | `src/Ordering.API/Application/Models/BasketItem.cs` | 入力データモデル |

**読解のコツ**: CreateOrderDraftCommandはrecord型で定義されており、イミュータブルなコマンドオブジェクト。OrderDraftDTOとOrderItemDTOも同ファイルに定義されている。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | OrdersApi.cs | `src/Ordering.API/Apis/OrdersApi.cs` | APIエンドポイント、CreateOrderDraftAsync |

**主要処理フロー**:
1. **16行目**: `api.MapPost("/draft", CreateOrderDraftAsync)` - エンドポイント定義
2. **106-116行目**: `CreateOrderDraftAsync` - API処理
3. **115行目**: `services.Mediator.Send(command)` - MediatRにコマンド送信

#### Step 3: コマンドハンドラを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CreateOrderDraftCommandHandler.cs | `src/Ordering.API/Application/Commands/CreateOrderDraftCommandHandler.cs` | ハンドラ実装とDTO定義 |

**主要処理フロー**:
- **10-19行目**: `Handle`メソッド
- **12行目**: `Order.NewDraft()` - ドラフト注文生成
- **13行目**: `i.ToOrderItemDTO()` - BasketItemからOrderItemDTOへ変換
- **14-17行目**: 各アイテムをAddOrderItemで追加
- **19行目**: `OrderDraftDTO.FromOrder(order)` - DTO変換

#### Step 4: ドメインモデルを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Order.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | NewDraft静的メソッド |

**主要処理フロー**:
- **37-44行目**: `NewDraft()` - ドラフトフラグ付きの空注文を生成
- **185行目**: `GetTotal()` - 合計金額計算

### プログラム呼び出し階層図

```
OrdersApi.CreateOrderDraftAsync (APIエントリーポイント)
    │
    └─ Mediator.Send(CreateOrderDraftCommand)
           │
           └─ CreateOrderDraftCommandHandler.Handle()
                  │
                  ├─ Order.NewDraft()
                  │      └─ _isDraft = true
                  │
                  ├─ BasketItem.ToOrderItemDTO() × N
                  │
                  ├─ order.AddOrderItem() × N
                  │
                  └─ OrderDraftDTO.FromOrder(order)
                         │
                         ├─ OrderItems変換
                         └─ order.GetTotal()
```

### データフロー図

```
[入力]               [処理]                    [出力]

HTTP POST        CreateOrderDraftCommandHandler  HTTP Response
(CreateOrder     ┌────────────────────┐         (OrderDraftDTO)
 DraftCommand)   │ Order.NewDraft()   │              │
       │         │      ↓             │              │
       ├───────▶ │ AddOrderItem × N   │              │
       │         │      ↓             │              │
       │         │ OrderDraftDTO.     │              │
       │         │   FromOrder()      │──────────────┘
       │         │      ↓             │
       │         │ GetTotal()        │
       │         └────────────────────┘
       │
       │         ※ DB操作なし
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| OrdersApi.cs | `src/Ordering.API/Apis/OrdersApi.cs` | ソース | APIエンドポイント |
| CreateOrderDraftCommand.cs | `src/Ordering.API/Application/Commands/CreateOrderDraftCommand.cs` | ソース | コマンドオブジェクト |
| CreateOrderDraftCommandHandler.cs | `src/Ordering.API/Application/Commands/CreateOrderDraftCommandHandler.cs` | ソース | コマンドハンドラ・DTO定義 |
| BasketItem.cs | `src/Ordering.API/Application/Models/BasketItem.cs` | ソース | 入力データモデル |
| Order.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | ソース | 注文エンティティ（NewDraft） |
| BasketItemExtensions.cs | `src/Ordering.API/Extensions/BasketItemExtensions.cs` | ソース | BasketItem変換拡張メソッド |
