# 機能設計書 19-注文詳細取得

## 概要

本ドキュメントは、eShopシステムにおける注文詳細取得機能の詳細設計を記載する。

### 本機能の処理概要

本機能は、指定されたIDの注文詳細を取得するためのREST APIエンドポイントを提供する。注文の基本情報（注文番号、日付、ステータス、配送先住所）と注文明細（商品名、数量、単価、画像URL）を含む完全な注文情報を返却する。

**業務上の目的・背景**：ユーザーや管理者が特定の注文の詳細情報を確認するための機能である。注文ステータスの確認、配送先住所の確認、注文商品の確認など、注文に関するあらゆる情報を取得できる。

**機能の利用シーン**：
- 注文履歴画面から特定の注文の詳細を表示する場合
- 注文確認メールに記載するための注文情報を取得する場合
- カスタマーサポートが注文内容を確認する場合

**主要な処理内容**：
1. HTTPリクエストからorderIdを受け取る
2. OrderQueriesを通じてデータベースから注文情報を取得する
3. 注文が存在しない場合はNotFound (404)を返却する
4. 注文情報をOrder DTOに変換して返却する

**関連システム・外部連携**：特になし

**権限による制御**：特になし（認可は上位で制御される想定）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 注文履歴画面から特定注文の詳細表示に使用される可能性があるが、画面機能マッピングに直接定義はない |

## 機能種別

CRUD操作（Read）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| orderId | int | Yes | 取得対象の注文ID | パスパラメータとして指定 |

### 入力データソース

パスパラメータ: URL（`/api/orders/{orderId}`）

## 出力仕様

### 出力データ

**Order DTO**

| 項目名 | 型 | 説明 |
|--------|-----|------|
| OrderNumber | int | 注文番号（注文ID） |
| Date | DateTime | 注文日時 |
| Status | string | 注文ステータス |
| Description | string | 注文の説明（ステータス変更理由等） |
| Street | string | 配送先通り |
| City | string | 配送先都市 |
| State | string | 配送先州 |
| Zipcode | string | 配送先郵便番号 |
| Country | string | 配送先国 |
| OrderItems | List<Orderitem> | 注文明細リスト |
| Total | decimal | 合計金額 |

**Orderitem構造**

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ProductName | string | 商品名 |
| Units | int | 数量 |
| UnitPrice | double | 単価 |
| PictureUrl | string | 商品画像URL |

### 出力先

HTTPレスポンス（200 OK + Order / 404 NotFound）

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエスト受信
   └─ GET /api/orders/{orderId}

2. OrderQueries.GetOrderAsync実行
   └─ Entity Framework経由でDB取得

3. 存在チェック
   └─ 注文が存在しない場合はKeyNotFoundExceptionがスロー

4. DTO変換
   └─ ドメインエンティティからOrder DTOに変換

5. レスポンス返却
   └─ Ok(Order) または NotFound
```

### フローチャート

```mermaid
flowchart TD
    A[GET /api/orders/orderId] --> B[OrderQueries.GetOrderAsync]
    B --> C{注文存在?}
    C -->|No| D[NotFound 404]
    C -->|Yes| E[Order DTO変換]
    E --> F[合計金額計算]
    F --> G[OK 200 + Order]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-19-01 | 存在確認 | 注文が存在しない場合は404エラーを返す | 取得時 |
| BR-19-02 | 合計金額計算 | 合計金額は明細の単価×数量の合計 | DTO変換時 |
| BR-19-03 | ステータス文字列化 | OrderStatusは文字列に変換して返却 | DTO変換時 |

### 計算ロジック

注文合計金額:
```
Total = order.GetTotal() // ドメインメソッドで計算
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 注文取得 | Orders | SELECT | 注文レコード取得 |
| 明細取得 | OrderItems | SELECT | 注文明細取得（Include） |

### テーブル別操作詳細

#### Orders + OrderItems

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | 全カラム | WHERE Id = @orderId | Include(OrderItems)で一括取得 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 404 | Not Found | 指定IDの注文が存在しない場合 | 正しい注文IDを指定 |
| 500 | Internal Server Error | DB接続エラー時 | サーバーログを確認 |

### リトライ仕様

本機能にリトライ機能は実装されていない。

## トランザクション仕様

読み取り専用のため、トランザクション管理は特に必要ない。

## パフォーマンス要件

- Include(OrderItems)により、N+1問題を回避
- 単一注文の取得のため、高速に処理される

## セキュリティ考慮事項

- 現在の実装では任意の注文IDでアクセス可能
- 本番環境では、ユーザー自身の注文のみアクセス可能にする認可チェックが必要

## 備考

- CQRSパターンにより、クエリは専用のIOrderQueriesインターフェースで処理
- ドメインエンティティからDTOへの変換はクエリクラス内で実施

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | OrderViewModel.cs | `src/Ordering.API/Application/Queries/OrderViewModel.cs` | Order, Orderitemレコード定義 |
| 1-2 | Order.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | ドメインエンティティ |

**読解のコツ**: CQRSパターンにより、読み取り用のDTO（OrderViewModel.cs）と書き込み用のドメインモデル（Order.cs）が分離されている。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | OrdersApi.cs | `src/Ordering.API/Apis/OrdersApi.cs` | APIエンドポイント、GetOrderAsync |

**主要処理フロー**:
1. **13行目**: `api.MapGet("{orderId:int}", GetOrderAsync)` - エンドポイント定義
2. **80-91行目**: `GetOrderAsync` - API処理
3. **84行目**: `services.Queries.GetOrderAsync(orderId)` - クエリ実行
4. **85行目**: `TypedResults.Ok(order)` - 成功時レスポンス
5. **89行目**: `TypedResults.NotFound()` - 失敗時レスポンス

#### Step 3: クエリ実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | IOrderQueries.cs | `src/Ordering.API/Application/Queries/IOrderQueries.cs` | クエリインターフェース |
| 3-2 | OrderQueries.cs | `src/Ordering.API/Application/Queries/OrderQueries.cs` | クエリ実装 |

**主要処理フロー**:
- **6-35行目**: `GetOrderAsync` - 注文取得実装
- **8-10行目**: Include(OrderItems)でEager Loading
- **12-13行目**: null チェックとKeyNotFoundException
- **15-34行目**: ドメインエンティティからDTOへ変換

### プログラム呼び出し階層図

```
OrdersApi.GetOrderAsync (APIエントリーポイント)
    │
    └─ services.Queries.GetOrderAsync(orderId)
           │
           └─ OrderQueries.GetOrderAsync()
                  │
                  ├─ context.Orders.Include(OrderItems)
                  │      .FirstOrDefaultAsync()
                  │
                  └─ Order DTO変換
                         ├─ 基本情報コピー
                         ├─ OrderItems変換
                         └─ order.GetTotal()
```

### データフロー図

```
[入力]               [処理]                    [出力]

HTTP GET         OrdersApi.GetOrderAsync     HTTP Response
Path ID ──────▶   OrderQueries実行  ─────────▶ (Order DTO)
                      ↓                           │
                 DB SELECT                        │
                 (Orders +                        │
                  OrderItems)                     │
                      ↓                           │
                 DTO変換                          │
                      │                           │
                 ┌────┴────┐                      │
                 ↓         ↓                      │
              存在する   存在しない               │
                 ↓         ↓                      │
              OK 200   NotFound 404               │
                 │                                │
                 └────────────────────────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| OrdersApi.cs | `src/Ordering.API/Apis/OrdersApi.cs` | ソース | APIエンドポイント |
| IOrderQueries.cs | `src/Ordering.API/Application/Queries/IOrderQueries.cs` | ソース | クエリインターフェース |
| OrderQueries.cs | `src/Ordering.API/Application/Queries/OrderQueries.cs` | ソース | クエリ実装 |
| OrderViewModel.cs | `src/Ordering.API/Application/Queries/OrderViewModel.cs` | ソース | Order, Orderitem DTO定義 |
| Order.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | ソース | ドメインエンティティ |
| OrderServices.cs | `src/Ordering.API/Apis/OrderServices.cs` | ソース | DI用サービスクラス |
