# 機能設計書 2-商品詳細取得

## 概要

本ドキュメントは、eShopアプリケーションにおける商品詳細取得機能の設計仕様を定義する。指定された商品IDに基づいて、商品の詳細情報（名前、説明、価格、在庫情報、ブランド情報等）を取得するREST APIエンドポイントを提供する。

### 本機能の処理概要

**業務上の目的・背景**：ECサイトにおいて、ユーザーが特定の商品を選択した際に、その商品の詳細情報を表示するための基盤機能である。商品の購入判断に必要な情報（価格、説明、ブランド等）を提供し、ユーザーの購買意思決定を支援する。

**機能の利用シーン**：商品詳細画面（WebApp/HybridApp）での商品情報表示、買い物かごへの追加前の商品確認、AIチャットボットでの商品推薦時の詳細情報取得など、単一商品の詳細を表示するあらゆる場面で利用される。

**主要な処理内容**：
1. パスパラメータから商品IDを取得
2. IDの妥当性検証（0より大きいこと）
3. データベースからブランド情報を含む商品データを取得
4. 商品が存在しない場合は404 NotFound、存在する場合は商品情報を返却

**関連システム・外部連携**：PostgreSQLデータベースからEntity Framework Coreを通じてデータを取得。CatalogBrandテーブルとのJOINによりブランド情報を含めて取得する。

**権限による制御**：本APIエンドポイントは認証不要で公開されており、誰でもアクセス可能である。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 2 | 商品詳細画面 | 主画面 | CatalogService.GetCatalogItemで商品詳細情報を取得 |
| 26 | 商品詳細画面（HybridApp） | 主画面 | CatalogService.GetCatalogItemで商品詳細情報を取得 |

## 機能種別

CRUD操作（Read）/ 単一レコード取得

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| id | int | Yes | 商品ID | 1以上の正の整数であること |

### 入力データソース

- URLパスパラメータ: `/api/catalog/items/{id}`

## 出力仕様

### 出力データ

**正常時（200 OK）: CatalogItem**

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Id | int | 商品ID |
| Name | string | 商品名 |
| Description | string? | 商品説明 |
| Price | decimal | 価格 |
| PictureFileName | string? | 商品画像ファイル名 |
| CatalogTypeId | int | 商品タイプID |
| CatalogBrandId | int | ブランドID |
| CatalogBrand | CatalogBrand | ブランド情報（Include） |
| AvailableStock | int | 在庫数 |
| RestockThreshold | int | 再発注閾値 |
| MaxStockThreshold | int | 最大在庫閾値 |
| OnReorder | bool | 再発注フラグ |

**CatalogBrand構造**

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Id | int | ブランドID |
| Brand | string | ブランド名 |

### 出力先

- HTTP Response Body（JSON形式）
- Content-Type: application/json

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエスト受信
   └─ パスパラメータからidを取得

2. IDバリデーション
   └─ id <= 0 の場合、BadRequest(400)を返却

3. データベースクエリ実行
   ├─ CatalogItemsテーブルからid指定で検索
   └─ CatalogBrandをIncludeでJOIN

4. 存在チェック
   ├─ 商品が存在しない場合: NotFound(404)を返却
   └─ 商品が存在する場合: 次のステップへ

5. レスポンス生成
   └─ CatalogItemオブジェクトをJSONで返却
```

### フローチャート

```mermaid
flowchart TD
    A[HTTPリクエスト受信] --> B[id取得]
    B --> C{id > 0 ?}
    C -->|No| D[BadRequest 400]
    C -->|Yes| E[DB検索実行]
    E --> F[Include CatalogBrand]
    F --> G{商品存在?}
    G -->|No| H[NotFound 404]
    G -->|Yes| I[CatalogItem返却]
    I --> J[200 OK]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | ID正値チェック | 商品IDは1以上の正の整数でなければならない | 常時 |
| BR-002 | ブランド情報付加 | 商品詳細にはブランド情報を含めて返却する | 常時 |
| BR-003 | 単一商品取得 | SingleOrDefaultAsyncにより単一レコードを取得 | 常時 |

### 計算ロジック

本機能に計算ロジックはない。

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 商品詳細取得 | Catalog | SELECT | 指定IDの商品を取得 |
| ブランド取得 | CatalogBrand | SELECT | 商品に紐づくブランド情報を取得（JOIN） |

### テーブル別操作詳細

#### Catalog

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | 全カラム | WHERE Id = {id} | Embeddingは除外（JsonIgnore） |

#### CatalogBrand

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id, Brand | INNER JOIN via CatalogBrandId | Include句で結合 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | id <= 0 の場合 | ProblemDetails形式で "Id is not valid" を返却 |
| 404 | NotFound | 指定IDの商品が存在しない | NotFound応答を返却 |
| 500 | InternalServerError | データベース接続エラー等 | サーバーエラーとして処理 |

### リトライ仕様

本機能では明示的なリトライ処理は実装されていない。

## トランザクション仕様

読み取り専用操作のため、明示的なトランザクション制御は行わない。

## パフォーマンス要件

- 主キー（Id）による検索のため、インデックスを活用した高速な取得が可能
- SingleOrDefaultAsyncにより1件のみ取得するため、メモリ効率が良い

## セキュリティ考慮事項

- 認証不要のパブリックAPI
- SQLインジェクション対策: Entity Framework CoreのLINQクエリによりパラメータ化
- 在庫情報（AvailableStock等）が公開される

## 備考

- CatalogTypeはIncludeされていないため、レスポンスには含まれない
- ブランド情報のみが関連データとして取得される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | 商品エンティティ、CatalogBrandナビゲーションプロパティ |
| 1-2 | CatalogBrand.cs | `src/Catalog.API/Model/CatalogBrand.cs` | ブランドエンティティの構造 |

**読解のコツ**: CatalogItem.csの26行目にある`CatalogBrand?`プロパティがナビゲーションプロパティ。Include句で結合取得される。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | GetItemByIdメソッドの実装 |

**主要処理フロー**:
1. **36-40行目**: `/items/{id:int}` エンドポイント定義
2. **171-191行目**: `GetItemById` メソッド実装
3. **176-180行目**: IDバリデーション（id <= 0 でBadRequest）
4. **183行目**: Include(ci => ci.CatalogBrand)でJOIN
5. **185-188行目**: 存在チェックとNotFound返却
6. **190行目**: 正常時のOk返却

#### Step 3: データアクセス層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CatalogContext.cs | `src/Catalog.API/Infrastructure/CatalogContext.cs` | DbSet定義 |
| 3-2 | CatalogItemEntityTypeConfiguration.cs | `src/Catalog.API/Infrastructure/EntityConfigurations/CatalogItemEntityTypeConfiguration.cs` | リレーション設定 |

**主要処理フロー**:
- **CatalogItemEntityTypeConfiguration.cs 16-17行目**: CatalogBrandとのリレーション定義

### プログラム呼び出し階層図

```
HTTP Request (GET /api/catalog/items/{id})
    │
    └─ CatalogApi.GetItemById()
           │
           ├─ IDバリデーション (id <= 0)
           │      └─ TypedResults.BadRequest
           │
           └─ CatalogServices.Context (CatalogContext)
                  └─ DbSet<CatalogItem>.CatalogItems
                         ├─ Include(ci => ci.CatalogBrand)
                         └─ SingleOrDefaultAsync(ci => ci.Id == id)
                                │
                                ├─ null → TypedResults.NotFound
                                └─ item → TypedResults.Ok(item)
```

### データフロー図

```
[入力]                [処理]                              [出力]

HTTPリクエスト         CatalogApi.GetItemById()           HTTPレスポンス
  │                        │                                │
  └─ id ──────────────────▶├─▶ IDバリデーション            │
                           │      │                        │
                           │      ├─ NG → BadRequest ─────▶ 400
                           │      │                        │
                           │      ▼                        │
                           │   PostgreSQL                  │
                           │   (Catalog + CatalogBrand)    │
                           │      │                        │
                           │      ├─ null → NotFound ─────▶ 404
                           │      │                        │
                           │      ▼                        │
                           └─▶ CatalogItem ───────────────▶ 200 OK
                                 ├─ Id
                                 ├─ Name
                                 ├─ Price
                                 ├─ CatalogBrand
                                 │    ├─ Id
                                 │    └─ Brand
                                 └─ ...
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CatalogApi.cs | `src/Catalog.API/Apis/CatalogApi.cs` | ソース | APIエンドポイント定義 |
| CatalogItem.cs | `src/Catalog.API/Model/CatalogItem.cs` | ソース | 商品エンティティモデル |
| CatalogBrand.cs | `src/Catalog.API/Model/CatalogBrand.cs` | ソース | ブランドエンティティモデル |
| CatalogServices.cs | `src/Catalog.API/Model/CatalogServices.cs` | ソース | サービスコンテナ |
| CatalogContext.cs | `src/Catalog.API/Infrastructure/CatalogContext.cs` | ソース | DbContext |
| CatalogItemEntityTypeConfiguration.cs | `src/Catalog.API/Infrastructure/EntityConfigurations/CatalogItemEntityTypeConfiguration.cs` | ソース | エンティティ設定 |
