# 機能設計書 20-ユーザー注文履歴取得

## 概要

本ドキュメントは、eShopシステムにおけるユーザー注文履歴取得機能の詳細設計を記載する。

### 本機能の処理概要

本機能は、ログインユーザーの注文履歴一覧を取得するためのREST APIエンドポイントを提供する。認証されたユーザーのIDに基づいて、そのユーザーが過去に行った全ての注文のサマリー情報（注文番号、日付、ステータス、合計金額）を返却する。

**業務上の目的・背景**：ユーザーが自身の購入履歴を確認できることは、ECサイトにおける基本機能である。過去の注文状況の確認、再購入の検討、注文のトラッキングなど、様々な目的で使用される。

**機能の利用シーン**：
- 注文履歴画面でユーザーの過去の注文一覧を表示する場合
- マイページで最近の注文を表示する場合
- 注文ステータスの確認を行う場合

**主要な処理内容**：
1. HTTPリクエストを受信する
2. IIdentityServiceからログインユーザーのIDを取得する
3. OrderQueriesを通じてユーザーの注文履歴を取得する
4. OrderSummary DTOのリストを返却する

**関連システム・外部連携**：
- Identity認証（ユーザーID取得）

**権限による制御**：認証が必須。自分自身の注文履歴のみ取得可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 5 | 注文履歴画面 | 主機能 | OrderingService.GetOrdersでユーザーの注文履歴を取得 |

## 機能種別

CRUD操作（Read）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| - | - | - | パスパラメータ・ボディなし | - |

**暗黙的な入力**

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| userId | string | Yes | IIdentityServiceから取得されるユーザーID | - |

### 入力データソース

- 認証コンテキスト: IIdentityService経由でのユーザーID

## 出力仕様

### 出力データ

**IEnumerable<OrderSummary>**

| 項目名 | 型 | 説明 |
|--------|-----|------|
| OrderNumber | int | 注文番号（注文ID） |
| Date | DateTime | 注文日時 |
| Status | string | 注文ステータス |
| Total | double | 合計金額 |

### 出力先

HTTPレスポンス（200 OK + IEnumerable<OrderSummary>）

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエスト受信
   └─ GET /api/orders

2. ユーザーID取得
   └─ services.IdentityService.GetUserIdentity()

3. 注文履歴取得
   └─ services.Queries.GetOrdersFromUserAsync(userId)

4. レスポンス返却
   └─ Ok(IEnumerable<OrderSummary>)
```

### フローチャート

```mermaid
flowchart TD
    A[GET /api/orders] --> B[ユーザーID取得]
    B --> C[GetOrdersFromUserAsync]
    C --> D[OrderSummaryリスト返却]
    D --> E[OK 200]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-20-01 | 自己制限 | ユーザーは自分自身の注文履歴のみ取得可能 | 認証済みユーザー |
| BR-20-02 | Buyer紐付け | 注文はBuyer.IdentityGuidでユーザーと紐付け | クエリ条件 |
| BR-20-03 | 合計金額集計 | 合計金額は明細の単価×数量の合計をクエリで計算 | 取得時 |

### 計算ロジック

注文合計金額（クエリ内計算）:
```
Total = Sum(OrderItems.UnitPrice * OrderItems.Units)
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 注文履歴取得 | Orders | SELECT | ユーザーの注文レコード取得 |
| 合計金額計算 | OrderItems | SELECT (サブクエリ) | Sum集計 |
| Buyer参照 | Buyers | JOIN | ユーザーID照合 |

### テーブル別操作詳細

#### Orders + Buyers + OrderItems

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id, OrderDate, OrderStatus | WHERE Buyer.IdentityGuid = @userId | Buyers経由でフィルタ |
| SELECT (Sum) | UnitPrice * Units | OrderItemsのSum集計 | Total計算 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 401 | Unauthorized | 未認証アクセスの場合 | 認証後に再試行 |
| 500 | Internal Server Error | DB接続エラー時 | サーバーログを確認 |

### リトライ仕様

本機能にリトライ機能は実装されていない。

## トランザクション仕様

読み取り専用のため、トランザクション管理は特に必要ない。

## パフォーマンス要件

- ユーザー毎の注文数が多い場合、ページネーションの検討が必要
- 現在の実装では全件取得

## セキュリティ考慮事項

- IIdentityServiceによりログインユーザーのIDのみが使用される
- 他ユーザーの注文履歴にはアクセス不可

## 備考

- 現在の実装にはページネーションがないため、注文数が多いユーザーではパフォーマンス問題が発生する可能性がある
- 注文履歴画面ではSignalRによるリアルタイム更新に対応している（画面側機能）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | OrderViewModel.cs | `src/Ordering.API/Application/Queries/OrderViewModel.cs` | OrderSummary record定義 |

**読解のコツ**: OrderSummaryはOrder DTOよりも軽量で、一覧表示に必要な最小限の情報のみを含む。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | OrdersApi.cs | `src/Ordering.API/Apis/OrdersApi.cs` | APIエンドポイント、GetOrdersByUserAsync |

**主要処理フロー**:
1. **14行目**: `api.MapGet("/", GetOrdersByUserAsync)` - エンドポイント定義
2. **93-98行目**: `GetOrdersByUserAsync` - API処理
3. **95行目**: `services.IdentityService.GetUserIdentity()` - ユーザーID取得
4. **96行目**: `services.Queries.GetOrdersFromUserAsync(userId)` - クエリ実行
5. **97行目**: `TypedResults.Ok(orders)` - レスポンス返却

#### Step 3: クエリ実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | OrderQueries.cs | `src/Ordering.API/Application/Queries/OrderQueries.cs` | GetOrdersFromUserAsync実装 |

**主要処理フロー**:
- **37-49行目**: `GetOrdersFromUserAsync` - 注文履歴取得実装
- **40行目**: `Where(o => o.Buyer.IdentityGuid == userId)` - ユーザーフィルタ
- **41-47行目**: `Select(o => new OrderSummary {...})` - DTOへのプロジェクション
- **46行目**: `Sum(oi => oi.UnitPrice * oi.Units)` - 合計金額計算

#### Step 4: IdentityServiceを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | IIdentityService.cs | `src/Ordering.API/Infrastructure/Services/IIdentityService.cs` | インターフェース定義 |
| 4-2 | IdentityService.cs | `src/Ordering.API/Infrastructure/Services/IdentityService.cs` | 実装 |

**読解のコツ**: HttpContextのClaimsPrincipalからユーザーIDを取得している。

### プログラム呼び出し階層図

```
OrdersApi.GetOrdersByUserAsync (APIエントリーポイント)
    │
    ├─ services.IdentityService.GetUserIdentity()
    │      └─ HttpContext.User.Claimsから取得
    │
    └─ services.Queries.GetOrdersFromUserAsync(userId)
           │
           └─ OrderQueries.GetOrdersFromUserAsync()
                  │
                  └─ context.Orders
                         .Where(Buyer.IdentityGuid == userId)
                         .Select(OrderSummary)
                         .ToListAsync()
```

### データフロー図

```
[入力]               [処理]                    [出力]

HTTP GET         OrdersApi.GetOrdersByUserAsync  HTTP Response
(認証コンテキスト)      ↓                       (IEnumerable<
       │         ユーザーID取得                   OrderSummary>)
       │              ↓                              │
       └───────▶ OrderQueries実行                    │
                      ↓                              │
                 DB SELECT                           │
                 (Orders JOIN                        │
                  Buyers WHERE                       │
                  IdentityGuid)                      │
                      ↓                              │
                 Sum集計（Total）                    │
                      ↓                              │
                 OrderSummaryリスト ─────────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| OrdersApi.cs | `src/Ordering.API/Apis/OrdersApi.cs` | ソース | APIエンドポイント |
| IOrderQueries.cs | `src/Ordering.API/Application/Queries/IOrderQueries.cs` | ソース | クエリインターフェース |
| OrderQueries.cs | `src/Ordering.API/Application/Queries/OrderQueries.cs` | ソース | クエリ実装 |
| OrderViewModel.cs | `src/Ordering.API/Application/Queries/OrderViewModel.cs` | ソース | OrderSummary DTO定義 |
| IIdentityService.cs | `src/Ordering.API/Infrastructure/Services/IIdentityService.cs` | ソース | Identity取得IF |
| IdentityService.cs | `src/Ordering.API/Infrastructure/Services/IdentityService.cs` | ソース | Identity取得実装 |
| OrderServices.cs | `src/Ordering.API/Apis/OrderServices.cs` | ソース | DI用サービスクラス |
