# 機能設計書 21-注文キャンセル

## 概要

本ドキュメントは、eShopシステムにおける注文キャンセル機能の設計を記述するものである。ユーザーまたはシステムが既存の注文を取り消し、注文ステータスをキャンセル状態に変更するための機能仕様を定義する。

### 本機能の処理概要

注文キャンセル機能は、顧客が注文確定後にキャンセルを希望した場合、または在庫不足などのシステム判定により注文を無効化する必要がある場合に、注文のステータスを「Cancelled」に変更する機能である。

**業務上の目的・背景**：ECサイトにおいて、顧客は様々な理由（間違い注文、気が変わった、支払い情報の問題など）で注文をキャンセルする必要がある。また、システム側でも在庫不足や決済失敗時に注文を自動キャンセルする必要がある。本機能は、注文ライフサイクル管理の重要な要素として、適切なタイミングでのキャンセル処理とその結果の通知を実現する。

**機能の利用シーン**：
- 顧客がWeb画面から注文履歴を確認し、キャンセルボタンを押下した場合
- 支払い処理が失敗し、システムが自動的に注文をキャンセルする場合
- 在庫確認の結果、必要な商品の在庫がない場合のシステム自動キャンセル

**主要な処理内容**：
1. APIエンドポイント（PUT /api/orders/cancel）でキャンセルリクエストを受け付ける
2. 冪等性を保証するためのIdentifiedCommandパターンでリクエストを処理する
3. 注文ステータスが「Paid」または「Shipped」でないことを検証する
4. 注文ステータスを「Cancelled」に変更し、ドメインイベントを発行する
5. ドメインイベントハンドラが統合イベントを発行し、他のマイクロサービスに通知する

**関連システム・外部連携**：
- イベントバス（RabbitMQ）を通じて`OrderStatusChangedToCancelledIntegrationEvent`を発行
- 購買者情報の取得のためBuyerリポジトリと連携

**権限による制御**：
- 認証されたユーザーのみがキャンセルリクエストを送信可能
- 自身の注文のみキャンセル可能（将来的な拡張として管理者による強制キャンセルも考慮）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 5 | 注文履歴画面 | 主画面 | 注文一覧からキャンセルボタンを押下してキャンセルリクエストを送信 |

## 機能種別

CRUD操作（Update） / ステータス変更処理 / イベント発行

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| OrderNumber | int | Yes | キャンセル対象の注文番号 | 空でないこと（NotEmpty） |
| x-requestid | Guid | Yes | リクエストID（冪等性保証用） | 空のGUIDでないこと |

### 入力データソース

- HTTPリクエストボディ（JSON形式）から`CancelOrderCommand`を取得
- HTTPヘッダー（x-requestid）からリクエストIDを取得

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 成功時 | HTTP 200 OK | キャンセル処理成功 |
| リクエストID不正 | HTTP 400 BadRequest | リクエストIDが空の場合 |
| 処理失敗 | HTTP 500 ProblemDetails | キャンセル処理が失敗した場合 |

### 出力先

- HTTPレスポンス
- イベントバス（OrderStatusChangedToCancelledIntegrationEvent）

## 処理フロー

### 処理シーケンス

```
1. APIエンドポイントでリクエスト受信
   └─ PUT /api/orders/cancel でCancelOrderCommandを受け取る

2. リクエストIDの検証
   └─ x-requestidヘッダーが空でないことを確認

3. IdentifiedCommandの作成
   └─ 冪等性保証のためIdentifiedCommand<CancelOrderCommand, bool>でラップ

4. CancelOrderIdentifiedCommandHandlerの実行
   └─ 重複リクエストチェック（RequestManagerで管理）
   └─ 重複の場合はtrueを返却し処理終了

5. CancelOrderCommandHandlerの実行
   └─ IOrderRepositoryから注文を取得
   └─ 注文が存在しない場合はfalseを返却

6. Order.SetCancelledStatus()の実行
   └─ ステータスが「Paid」または「Shipped」でないことを確認
   └─ 条件違反時はOrderingDomainExceptionをスロー
   └─ ステータスを「Cancelled」に変更
   └─ OrderCancelledDomainEventを発行

7. UnitOfWorkでの永続化
   └─ SaveEntitiesAsync()でデータベースに保存
   └─ ドメインイベントをディスパッチ

8. OrderCancelledDomainEventHandlerの実行
   └─ 注文情報と購買者情報を取得
   └─ OrderStatusChangedToCancelledIntegrationEventを作成
   └─ イベントバスに発行
```

### フローチャート

```mermaid
flowchart TD
    A[PUT /api/orders/cancel] --> B{x-requestid有効?}
    B -->|No| C[400 BadRequest]
    B -->|Yes| D[IdentifiedCommand作成]
    D --> E{重複リクエスト?}
    E -->|Yes| F[true返却]
    F --> G[200 OK]
    E -->|No| H[注文取得]
    H --> I{注文存在?}
    I -->|No| J[false返却]
    J --> K[500 Problem]
    I -->|Yes| L{Paid/Shipped?}
    L -->|Yes| M[例外スロー]
    M --> K
    L -->|No| N[ステータスをCancelledに変更]
    N --> O[OrderCancelledDomainEvent発行]
    O --> P[データベース保存]
    P --> Q[統合イベント発行]
    Q --> G
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-21-01 | キャンセル可能ステータス制限 | 支払い済み（Paid）または発送済み（Shipped）の注文はキャンセル不可 | SetCancelledStatus()呼び出し時 |
| BR-21-02 | 冪等性保証 | 同一リクエストIDの処理は1回のみ実行 | すべてのキャンセルリクエスト |
| BR-21-03 | キャンセル理由記録 | キャンセル時に説明文を設定 | ステータス変更時 |

### 計算ロジック

特になし（ステータス変更処理のみ）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 注文取得 | ordering.orders | SELECT | 注文IDで注文を取得 |
| ステータス更新 | ordering.orders | UPDATE | OrderStatusをCancelledに更新、Descriptionを更新 |
| 購買者取得 | ordering.buyers | SELECT | 統合イベント発行用に購買者情報を取得 |
| リクエスト記録 | ordering.requests | INSERT | 冪等性保証のためリクエストIDを記録 |

### テーブル別操作詳細

#### ordering.orders

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id | WHERE Id = @OrderNumber | 注文取得 |
| UPDATE | OrderStatus | 'Cancelled' (6) | 列挙型の値 |
| UPDATE | Description | 'The order was cancelled.' | 固定メッセージ |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | リクエストIDが空のGUID | クライアントに有効なリクエストIDを要求 |
| 500 | ProblemHttpResult | 注文が存在しない | エラーメッセージを返却 |
| 500 | OrderingDomainException | Paid/Shippedステータスの注文をキャンセル | 業務ルール違反として処理 |

### リトライ仕様

- 冪等性が保証されているため、クライアント側でのリトライは安全
- 同一リクエストIDでのリトライはスキップされる

## トランザクション仕様

- UnitOfWorkパターンによりトランザクション管理
- SaveEntitiesAsync()でコミット
- 例外発生時は自動ロールバック
- ドメインイベントはトランザクションコミット後に発行

## パフォーマンス要件

- レスポンス時間: 1秒以内（通常時）
- 同時実行: 複数ユーザーからの同時キャンセルリクエストに対応

## セキュリティ考慮事項

- 認証済みユーザーのみアクセス可能
- 注文の所有者確認（将来的な拡張）
- リクエストIDによるリプレイ攻撃防止

## 備考

- IdentifiedCommandパターンにより冪等性を保証
- CQRSパターンに基づきコマンドとクエリを分離
- ドメインイベントを通じてマイクロサービス間の整合性を維持

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、注文のステータスと注文エンティティの構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | OrderStatus.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/OrderStatus.cs` | 注文ステータスの列挙型（Submitted=1〜Cancelled=6）を確認 |
| 1-2 | Order.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | 注文エンティティの構造、特にOrderStatusプロパティとSetCancelledStatus()メソッドを理解 |
| 1-3 | CancelOrderCommand.cs | `src/Ordering.API/Application/Commands/CancelOrderCommand.cs` | コマンドの構造（OrderNumberのみを持つrecord型）を確認 |

**読解のコツ**: C#のrecord型はイミュータブルなデータ型で、`record CancelOrderCommand(int OrderNumber)`は自動的にコンストラクタとプロパティを生成する。

#### Step 2: エントリーポイントを理解する

APIエンドポイントから処理がどのように開始されるかを把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | OrdersApi.cs | `src/Ordering.API/Apis/OrdersApi.cs` | CancelOrderAsync()メソッドがエントリーポイント |

**主要処理フロー**:
1. **22-49行目**: CancelOrderAsyncメソッド全体
2. **27-30行目**: リクエストIDの検証（空のGUIDチェック）
3. **32行目**: IdentifiedCommand<CancelOrderCommand, bool>でラップ
4. **41行目**: MediatorでコマンドをSend

#### Step 3: コマンドハンドラを理解する

コマンドがどのように処理されるかを追跡する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CancelOrderCommandHandler.cs | `src/Ordering.API/Application/Commands/CancelOrderCommandHandler.cs` | Handleメソッドでの処理フロー |

**主要処理フロー**:
- **19-29行目**: Handle()メソッド
- **21行目**: リポジトリから注文を取得
- **22-25行目**: 注文が存在しない場合はfalseを返却
- **27行目**: Order.SetCancelledStatus()を呼び出し
- **28行目**: UnitOfWorkで永続化

#### Step 4: ドメインロジックを理解する

注文エンティティ内でのキャンセル処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Order.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | SetCancelledStatus()メソッドのビジネスルール |

**主要処理フロー**:
- **142-153行目**: SetCancelledStatus()メソッド
- **144-148行目**: Paid/Shippedステータスのチェック、違反時は例外スロー
- **150行目**: ステータスをCancelledに変更
- **151行目**: 説明文を設定
- **152行目**: OrderCancelledDomainEventを発行

#### Step 5: ドメインイベントハンドラを理解する

ドメインイベントがどのように統合イベントに変換されるかを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | OrderCancelledDomainEventHandler.cs | `src/Ordering.API/Application/DomainEventHandlers/OrderCancelledDomainEventHandler.cs` | 統合イベント発行処理 |

**主要処理フロー**:
- **23-32行目**: Handle()メソッド
- **27-28行目**: 注文と購買者情報を取得
- **30行目**: OrderStatusChangedToCancelledIntegrationEventを作成
- **31行目**: イベントバスに発行

### プログラム呼び出し階層図

```
OrdersApi.CancelOrderAsync()
    │
    ├─ IdentifiedCommand<CancelOrderCommand, bool> 作成
    │
    └─ IMediator.Send()
           │
           ├─ CancelOrderIdentifiedCommandHandler.Handle()
           │      │
           │      ├─ IRequestManager (重複チェック)
           │      │
           │      └─ IMediator.Send() (内部コマンド)
           │             │
           │             └─ CancelOrderCommandHandler.Handle()
           │                    │
           │                    ├─ IOrderRepository.GetAsync()
           │                    │
           │                    ├─ Order.SetCancelledStatus()
           │                    │      │
           │                    │      └─ AddDomainEvent(OrderCancelledDomainEvent)
           │                    │
           │                    └─ IUnitOfWork.SaveEntitiesAsync()
           │                           │
           │                           └─ MediatR Dispatch
           │                                  │
           │                                  └─ OrderCancelledDomainEventHandler.Handle()
           │                                         │
           │                                         ├─ IOrderRepository.GetAsync()
           │                                         │
           │                                         ├─ IBuyerRepository.FindByIdAsync()
           │                                         │
           │                                         └─ IOrderingIntegrationEventService.AddAndSaveEventAsync()
           │
           └─ 結果返却 (bool)
```

### データフロー図

```
[入力]                        [処理]                              [出力]

HTTP PUT Request ───────▶ OrdersApi.CancelOrderAsync() ──────▶ HTTP Response
  │                              │
  ├─ x-requestid (Header)        │
  │                              ▼
  └─ CancelOrderCommand    IdentifiedCommandHandler ──────────▶ RequestManager
       (OrderNumber)             │                                (重複記録)
                                 ▼
                          CancelOrderCommandHandler
                                 │
                                 ▼
                          IOrderRepository.GetAsync() ◀────────── Orders Table
                                 │
                                 ▼
                          Order.SetCancelledStatus()
                                 │
                                 ├─ OrderStatus = Cancelled
                                 │
                                 └─ OrderCancelledDomainEvent
                                          │
                                          ▼
                          OrderCancelledDomainEventHandler
                                          │
                                          ▼
                          OrderStatusChangedToCancelledIntegrationEvent
                                          │
                                          ▼
                                    EventBus (RabbitMQ)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| OrdersApi.cs | `src/Ordering.API/Apis/OrdersApi.cs` | ソース | APIエンドポイント定義 |
| CancelOrderCommand.cs | `src/Ordering.API/Application/Commands/CancelOrderCommand.cs` | ソース | キャンセルコマンド定義 |
| CancelOrderCommandHandler.cs | `src/Ordering.API/Application/Commands/CancelOrderCommandHandler.cs` | ソース | コマンドハンドラ |
| CancelOrderCommandValidator.cs | `src/Ordering.API/Application/Validations/CancelOrderCommandValidator.cs` | ソース | バリデーション |
| Order.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | ソース | 注文エンティティ |
| OrderStatus.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/OrderStatus.cs` | ソース | ステータス列挙型 |
| IOrderRepository.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/IOrderRepository.cs` | ソース | リポジトリインターフェース |
| OrderCancelledDomainEvent.cs | `src/Ordering.Domain/Events/OrderCancelledDomainEvent.cs` | ソース | ドメインイベント |
| OrderCancelledDomainEventHandler.cs | `src/Ordering.API/Application/DomainEventHandlers/OrderCancelledDomainEventHandler.cs` | ソース | ドメインイベントハンドラ |
| OrderStatusChangedToCancelledIntegrationEvent.cs | `src/Ordering.API/Application/IntegrationEvents/Events/OrderStatusChangedToCancelledIntegrationEvent.cs` | ソース | 統合イベント |
| IdentifiedCommandHandler.cs | `src/Ordering.API/Application/Commands/IdentifiedCommandHandler.cs` | ソース | 冪等性ハンドラ |
