# 機能設計書 22-注文発送処理

## 概要

本ドキュメントは、eShopシステムにおける注文発送処理機能の設計を記述するものである。管理者が支払い完了した注文を発送済みステータスに変更するための機能仕様を定義する。

### 本機能の処理概要

注文発送処理機能は、管理者が注文の発送手続きを完了した際に、注文ステータスを「Shipped」に変更する機能である。支払いが完了（Paid）している注文のみが発送可能であり、それ以外のステータスからの発送処理は業務ルール違反としてエラーとなる。

**業務上の目的・背景**：ECサイトにおいて、注文の発送は物流プロセスの重要なマイルストーンである。顧客への通知、在庫管理、売上計上などの後続処理のトリガーとなるため、発送ステータスの正確な管理が求められる。本機能は、発送完了の記録と関連システムへの通知を一貫して行う。

**機能の利用シーン**：
- 倉庫スタッフが商品を梱包・発送した後、管理画面から発送完了を登録する場合
- 物流システムからの自動連携により発送完了を記録する場合
- バッチ処理により複数注文の発送処理を一括実行する場合

**主要な処理内容**：
1. APIエンドポイント（PUT /api/orders/ship）で発送リクエストを受け付ける
2. 冪等性を保証するためのIdentifiedCommandパターンでリクエストを処理する
3. 注文ステータスが「Paid」であることを検証する
4. 注文ステータスを「Shipped」に変更し、ドメインイベントを発行する
5. ドメインイベントハンドラが統合イベントを発行し、他のマイクロサービスに通知する

**関連システム・外部連携**：
- イベントバス（RabbitMQ）を通じて`OrderStatusChangedToShippedIntegrationEvent`を発行
- Webhooks APIを通じて外部システム（配送追跡など）に通知可能

**権限による制御**：
- 管理者権限を持つユーザーのみが発送処理を実行可能
- 一般顧客は発送処理を実行できない

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | 管理画面（未実装） | 主画面 | 注文一覧から発送ボタンを押下して発送リクエストを送信 |
| 5 | 注文履歴画面 | 結果表示画面 | 発送済みステータスの注文を表示 |

## 機能種別

CRUD操作（Update） / ステータス変更処理 / イベント発行

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| OrderNumber | int | Yes | 発送対象の注文番号 | 空でないこと（NotEmpty） |
| x-requestid | Guid | Yes | リクエストID（冪等性保証用） | 空のGUIDでないこと |

### 入力データソース

- HTTPリクエストボディ（JSON形式）から`ShipOrderCommand`を取得
- HTTPヘッダー（x-requestid）からリクエストIDを取得

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 成功時 | HTTP 200 OK | 発送処理成功 |
| リクエストID不正 | HTTP 400 BadRequest | リクエストIDが空の場合 |
| 処理失敗 | HTTP 500 ProblemDetails | 発送処理が失敗した場合 |

### 出力先

- HTTPレスポンス
- イベントバス（OrderStatusChangedToShippedIntegrationEvent）

## 処理フロー

### 処理シーケンス

```
1. APIエンドポイントでリクエスト受信
   └─ PUT /api/orders/ship でShipOrderCommandを受け取る

2. リクエストIDの検証
   └─ x-requestidヘッダーが空でないことを確認

3. IdentifiedCommandの作成
   └─ 冪等性保証のためIdentifiedCommand<ShipOrderCommand, bool>でラップ

4. ShipOrderIdentifiedCommandHandlerの実行
   └─ 重複リクエストチェック（RequestManagerで管理）
   └─ 重複の場合はtrueを返却し処理終了

5. ShipOrderCommandHandlerの実行
   └─ IOrderRepositoryから注文を取得
   └─ 注文が存在しない場合はfalseを返却

6. Order.SetShippedStatus()の実行
   └─ ステータスが「Paid」であることを確認
   └─ 条件違反時はOrderingDomainExceptionをスロー
   └─ ステータスを「Shipped」に変更
   └─ OrderShippedDomainEventを発行

7. UnitOfWorkでの永続化
   └─ SaveEntitiesAsync()でデータベースに保存
   └─ ドメインイベントをディスパッチ

8. OrderShippedDomainEventHandlerの実行
   └─ 注文情報と購買者情報を取得
   └─ OrderStatusChangedToShippedIntegrationEventを作成
   └─ イベントバスに発行
```

### フローチャート

```mermaid
flowchart TD
    A[PUT /api/orders/ship] --> B{x-requestid有効?}
    B -->|No| C[400 BadRequest]
    B -->|Yes| D[IdentifiedCommand作成]
    D --> E{重複リクエスト?}
    E -->|Yes| F[true返却]
    F --> G[200 OK]
    E -->|No| H[注文取得]
    H --> I{注文存在?}
    I -->|No| J[false返却]
    J --> K[500 Problem]
    I -->|Yes| L{Paidステータス?}
    L -->|No| M[例外スロー]
    M --> K
    L -->|Yes| N[ステータスをShippedに変更]
    N --> O[OrderShippedDomainEvent発行]
    O --> P[データベース保存]
    P --> Q[統合イベント発行]
    Q --> G
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-22-01 | 発送可能ステータス制限 | 支払い済み（Paid）の注文のみ発送可能 | SetShippedStatus()呼び出し時 |
| BR-22-02 | 冪等性保証 | 同一リクエストIDの処理は1回のみ実行 | すべての発送リクエスト |
| BR-22-03 | 発送メッセージ記録 | 発送時に説明文を設定 | ステータス変更時 |

### 計算ロジック

特になし（ステータス変更処理のみ）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 注文取得 | ordering.orders | SELECT | 注文IDで注文を取得 |
| ステータス更新 | ordering.orders | UPDATE | OrderStatusをShippedに更新、Descriptionを更新 |
| 購買者取得 | ordering.buyers | SELECT | 統合イベント発行用に購買者情報を取得 |
| リクエスト記録 | ordering.requests | INSERT | 冪等性保証のためリクエストIDを記録 |

### テーブル別操作詳細

#### ordering.orders

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id | WHERE Id = @OrderNumber | 注文取得 |
| UPDATE | OrderStatus | 'Shipped' (5) | 列挙型の値 |
| UPDATE | Description | 'The order was shipped.' | 固定メッセージ |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | リクエストIDが空のGUID | クライアントに有効なリクエストIDを要求 |
| 500 | ProblemHttpResult | 注文が存在しない | エラーメッセージを返却 |
| 500 | OrderingDomainException | Paid以外のステータスの注文を発送 | 業務ルール違反として処理 |

### リトライ仕様

- 冪等性が保証されているため、クライアント側でのリトライは安全
- 同一リクエストIDでのリトライはスキップされる

## トランザクション仕様

- UnitOfWorkパターンによりトランザクション管理
- SaveEntitiesAsync()でコミット
- 例外発生時は自動ロールバック
- ドメインイベントはトランザクションコミット後に発行

## パフォーマンス要件

- レスポンス時間: 1秒以内（通常時）
- 同時実行: 複数管理者からの同時発送リクエストに対応

## セキュリティ考慮事項

- 管理者権限を持つユーザーのみアクセス可能
- リクエストIDによるリプレイ攻撃防止
- 監査ログへの発送操作記録（将来的な拡張）

## 備考

- IdentifiedCommandパターンにより冪等性を保証
- CQRSパターンに基づきコマンドとクエリを分離
- ドメインイベントを通じてマイクロサービス間の整合性を維持
- 発送完了後はキャンセル不可（Shippedステータスからの遷移制限）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、注文のステータスと注文エンティティの構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | OrderStatus.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/OrderStatus.cs` | 注文ステータスの列挙型（Paid=4, Shipped=5）を確認 |
| 1-2 | Order.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | 注文エンティティの構造、特にSetShippedStatus()メソッドを理解 |
| 1-3 | ShipOrderCommand.cs | `src/Ordering.API/Application/Commands/ShipOrderCommand.cs` | コマンドの構造（OrderNumberのみを持つrecord型）を確認 |

**読解のコツ**: SetShippedStatus()はPaid以外のステータスから呼び出すと例外をスローする。この制約がビジネスルールとしてドメインモデルに組み込まれている。

#### Step 2: エントリーポイントを理解する

APIエンドポイントから処理がどのように開始されるかを把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | OrdersApi.cs | `src/Ordering.API/Apis/OrdersApi.cs` | ShipOrderAsync()メソッドがエントリーポイント |

**主要処理フロー**:
1. **51-78行目**: ShipOrderAsyncメソッド全体
2. **56-59行目**: リクエストIDの検証（空のGUIDチェック）
3. **61行目**: IdentifiedCommand<ShipOrderCommand, bool>でラップ
4. **70行目**: MediatorでコマンドをSend

#### Step 3: コマンドハンドラを理解する

コマンドがどのように処理されるかを追跡する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ShipOrderCommandHandler.cs | `src/Ordering.API/Application/Commands/ShipOrderCommandHandler.cs` | Handleメソッドでの処理フロー |

**主要処理フロー**:
- **19-29行目**: Handle()メソッド
- **21行目**: リポジトリから注文を取得
- **22-25行目**: 注文が存在しない場合はfalseを返却
- **27行目**: Order.SetShippedStatus()を呼び出し
- **28行目**: UnitOfWorkで永続化

#### Step 4: ドメインロジックを理解する

注文エンティティ内での発送処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Order.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | SetShippedStatus()メソッドのビジネスルール |

**主要処理フロー**:
- **130-140行目**: SetShippedStatus()メソッド
- **132-135行目**: Paidステータスでないことをチェック、違反時は例外スロー
- **137行目**: ステータスをShippedに変更
- **138行目**: 説明文を設定
- **139行目**: OrderShippedDomainEventを発行

#### Step 5: ドメインイベントハンドラを理解する

ドメインイベントがどのように統合イベントに変換されるかを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | OrderShippedDomainEventHandler.cs | `src/Ordering.API/Application/DomainEventHandlers/OrderShippedDomainEventHandler.cs` | 統合イベント発行処理 |

**主要処理フロー**:
- **23-32行目**: Handle()メソッド
- **27-28行目**: 注文と購買者情報を取得
- **30行目**: OrderStatusChangedToShippedIntegrationEventを作成
- **31行目**: イベントバスに発行

### プログラム呼び出し階層図

```
OrdersApi.ShipOrderAsync()
    │
    ├─ IdentifiedCommand<ShipOrderCommand, bool> 作成
    │
    └─ IMediator.Send()
           │
           ├─ ShipOrderIdentifiedCommandHandler.Handle()
           │      │
           │      ├─ IRequestManager (重複チェック)
           │      │
           │      └─ IMediator.Send() (内部コマンド)
           │             │
           │             └─ ShipOrderCommandHandler.Handle()
           │                    │
           │                    ├─ IOrderRepository.GetAsync()
           │                    │
           │                    ├─ Order.SetShippedStatus()
           │                    │      │
           │                    │      └─ AddDomainEvent(OrderShippedDomainEvent)
           │                    │
           │                    └─ IUnitOfWork.SaveEntitiesAsync()
           │                           │
           │                           └─ MediatR Dispatch
           │                                  │
           │                                  └─ OrderShippedDomainEventHandler.Handle()
           │                                         │
           │                                         ├─ IOrderRepository.GetAsync()
           │                                         │
           │                                         ├─ IBuyerRepository.FindByIdAsync()
           │                                         │
           │                                         └─ IOrderingIntegrationEventService.AddAndSaveEventAsync()
           │
           └─ 結果返却 (bool)
```

### データフロー図

```
[入力]                        [処理]                              [出力]

HTTP PUT Request ───────▶ OrdersApi.ShipOrderAsync() ──────▶ HTTP Response
  │                              │
  ├─ x-requestid (Header)        │
  │                              ▼
  └─ ShipOrderCommand      IdentifiedCommandHandler ──────────▶ RequestManager
       (OrderNumber)             │                                (重複記録)
                                 ▼
                          ShipOrderCommandHandler
                                 │
                                 ▼
                          IOrderRepository.GetAsync() ◀────────── Orders Table
                                 │
                                 ▼
                          Order.SetShippedStatus()
                                 │
                                 ├─ OrderStatus = Shipped
                                 │
                                 └─ OrderShippedDomainEvent
                                          │
                                          ▼
                          OrderShippedDomainEventHandler
                                          │
                                          ▼
                          OrderStatusChangedToShippedIntegrationEvent
                                          │
                                          ▼
                                    EventBus (RabbitMQ)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| OrdersApi.cs | `src/Ordering.API/Apis/OrdersApi.cs` | ソース | APIエンドポイント定義 |
| ShipOrderCommand.cs | `src/Ordering.API/Application/Commands/ShipOrderCommand.cs` | ソース | 発送コマンド定義 |
| ShipOrderCommandHandler.cs | `src/Ordering.API/Application/Commands/ShipOrderCommandHandler.cs` | ソース | コマンドハンドラ |
| ShipOrderCommandValidator.cs | `src/Ordering.API/Application/Validations/ShipOrderCommandValidator.cs` | ソース | バリデーション |
| Order.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | ソース | 注文エンティティ |
| OrderStatus.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/OrderStatus.cs` | ソース | ステータス列挙型 |
| IOrderRepository.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/IOrderRepository.cs` | ソース | リポジトリインターフェース |
| OrderShippedDomainEvent.cs | `src/Ordering.Domain/Events/OrderShippedDomainEvent.cs` | ソース | ドメインイベント |
| OrderShippedDomainEventHandler.cs | `src/Ordering.API/Application/DomainEventHandlers/OrderShippedDomainEventHandler.cs` | ソース | ドメインイベントハンドラ |
| OrderStatusChangedToShippedIntegrationEvent.cs | `src/Ordering.API/Application/IntegrationEvents/Events/OrderStatusChangedToShippedIntegrationEvent.cs` | ソース | 統合イベント |
| IdentifiedCommandHandler.cs | `src/Ordering.API/Application/Commands/IdentifiedCommandHandler.cs` | ソース | 冪等性ハンドラ |
