# 機能設計書 23-カードタイプ取得

## 概要

本ドキュメントは、eShopシステムにおけるカードタイプ取得機能の設計を記述するものである。注文時に利用可能なクレジットカードタイプ（Visa、MasterCard、Amexなど）の一覧を取得するための機能仕様を定義する。

### 本機能の処理概要

カードタイプ取得機能は、ECサイトのチェックアウト画面で決済に使用可能なカードブランドをユーザーに表示するために、システムに登録されているカードタイプの一覧を取得する機能である。

**業務上の目的・背景**：ECサイトでの決済において、ユーザーは自身のクレジットカードのブランド（Visa、MasterCard、Amexなど）を選択する必要がある。利用可能なカードタイプをマスターデータとして管理し、APIで取得することで、フロントエンドでの選択肢の表示を動的に制御できる。カードタイプをハードコードせずにデータベースで管理することで、新しいカードブランドの追加や無効化が容易になる。

**機能の利用シーン**：
- チェックアウト画面で決済情報を入力する際に、カードタイプのドロップダウンリストを表示する場合
- 注文作成時にカードタイプIDの妥当性を検証するためのマスターデータ参照
- 管理画面で利用可能なカードタイプを確認する場合

**主要な処理内容**：
1. APIエンドポイント（GET /api/orders/cardtypes）でリクエストを受け付ける
2. OrderQueriesを通じてCardTypesテーブルからデータを取得
3. カードタイプID・名前のリストをJSON形式で返却

**関連システム・外部連携**：
- 特になし（内部マスターデータの参照のみ）

**権限による制御**：
- 認証されたユーザーのみアクセス可能（チェックアウト前提のため）
- 特別な権限は不要

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 4 | チェックアウト画面 | 主画面 | 決済情報入力時にカードタイプを選択 |

## 機能種別

CRUD操作（Read） / マスターデータ参照

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| なし | - | - | パラメータ不要 | - |

### 入力データソース

- HTTPリクエスト（パラメータなし）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Id | int | カードタイプID |
| Name | string | カードタイプ名（Visa、MasterCard、Amexなど） |

### 出力先

- HTTPレスポンス（JSON配列形式）

## 処理フロー

### 処理シーケンス

```
1. APIエンドポイントでリクエスト受信
   └─ GET /api/orders/cardtypes でリクエストを受け取る

2. IOrderQueriesの呼び出し
   └─ GetCardTypesAsync()メソッドを実行

3. データベースクエリの実行
   └─ CardTypesテーブルから全件取得
   └─ Id, Nameのみを選択

4. 結果の返却
   └─ CardTypeオブジェクトのリストをJSON形式で返却
```

### フローチャート

```mermaid
flowchart TD
    A[GET /api/orders/cardtypes] --> B[IOrderQueries.GetCardTypesAsync]
    B --> C[DbContext.CardTypes.Select]
    C --> D[CardType リスト]
    D --> E[200 OK + JSON配列]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-23-01 | 全件取得 | フィルタリングなしで全カードタイプを返却 | 常に適用 |
| BR-23-02 | 最小限のデータ返却 | Id, Nameのみを返却（セキュリティ上不要なデータを含めない） | 常に適用 |

### 計算ロジック

特になし（単純なデータ取得）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| カードタイプ取得 | ordering.cardtypes | SELECT | 全カードタイプを取得 |

### テーブル別操作詳細

#### ordering.cardtypes

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id | 全件 | カードタイプID |
| SELECT | Name | 全件 | カードタイプ名 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 401 | Unauthorized | 認証されていない場合 | ログインを促す |
| 500 | InternalServerError | データベース接続エラー | エラーログ出力、リトライ |

### リトライ仕様

- クライアント側でのリトライは安全（参照のみのため）

## トランザクション仕様

- 読み取り専用操作のためトランザクション不要
- 暗黙的な読み取りコミットレベル

## パフォーマンス要件

- レスポンス時間: 100ms以内（通常時）
- キャッシュ: マスターデータのため、クライアント側でのキャッシュ推奨

## セキュリティ考慮事項

- 認証済みユーザーのみアクセス可能
- センシティブなデータを含まない

## 備考

- CQRSパターンに基づきクエリ専用のOrderQueriesで実装
- データ量が少ないためページネーション不要
- マスターデータのため更新頻度は低い

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、カードタイプのドメインモデルとクエリ用DTOの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CardType.cs（Domain） | `src/Ordering.Domain/AggregatesModel/BuyerAggregate/CardType.cs` | ドメインモデルの構造（Id, Nameプロパティ）を確認 |
| 1-2 | OrderViewModel.cs | `src/Ordering.API/Application/Queries/OrderViewModel.cs` | クエリ用CardType recordの定義を確認 |

**読解のコツ**: ドメインモデル（Ordering.Domain）とクエリ用DTO（Ordering.API）で同名のCardTypeクラスが存在する。APIではクエリ用のシンプルなrecordを使用し、ドメインモデルから必要な項目のみをマッピングしている。

#### Step 2: エントリーポイントを理解する

APIエンドポイントから処理がどのように開始されるかを把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | OrdersApi.cs | `src/Ordering.API/Apis/OrdersApi.cs` | GetCardTypesAsyncメソッドがエントリーポイント |

**主要処理フロー**:
1. **15行目**: ルート定義 `api.MapGet("/cardtypes", GetCardTypesAsync)`
2. **100-104行目**: GetCardTypesAsyncメソッド全体
3. **102行目**: IOrderQueries.GetCardTypesAsync()を呼び出し
4. **103行目**: TypedResults.Ok()でレスポンス返却

#### Step 3: クエリ実装を理解する

クエリがどのように実装されているかを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | IOrderQueries.cs | `src/Ordering.API/Application/Queries/IOrderQueries.cs` | インターフェース定義を確認 |
| 3-2 | OrderQueries.cs | `src/Ordering.API/Application/Queries/OrderQueries.cs` | 具体的なクエリ実装を確認 |

**主要処理フロー**:
- **51-52行目**: GetCardTypesAsync()メソッド
- **52行目**: context.CardTypes.Select()でデータ取得
- **52行目**: Id, NameのみをマッピングしてDTOを作成

### プログラム呼び出し階層図

```
OrdersApi.GetCardTypesAsync()
    │
    └─ IOrderQueries.GetCardTypesAsync()
           │
           └─ OrderQueries.GetCardTypesAsync()
                  │
                  └─ OrderingContext.CardTypes.Select()
                         │
                         └─ ToListAsync()
```

### データフロー図

```
[入力]                        [処理]                              [出力]

HTTP GET Request ───────▶ OrdersApi.GetCardTypesAsync() ──────▶ HTTP Response
                                 │                                    │
                                 ▼                                    │
                         IOrderQueries.GetCardTypesAsync()            │
                                 │                                    │
                                 ▼                                    │
                         OrderingContext.CardTypes ◀────── CardTypes Table
                                 │                                    │
                                 ▼                                    │
                         Select(c => new CardType                     │
                           { Id = c.Id, Name = c.Name })              │
                                 │                                    │
                                 ▼                                    ▼
                         List<CardType> ─────────────────▶ JSON Array Response
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| OrdersApi.cs | `src/Ordering.API/Apis/OrdersApi.cs` | ソース | APIエンドポイント定義 |
| IOrderQueries.cs | `src/Ordering.API/Application/Queries/IOrderQueries.cs` | ソース | クエリインターフェース |
| OrderQueries.cs | `src/Ordering.API/Application/Queries/OrderQueries.cs` | ソース | クエリ実装 |
| OrderViewModel.cs | `src/Ordering.API/Application/Queries/OrderViewModel.cs` | ソース | クエリ用DTO（CardType record） |
| CardType.cs | `src/Ordering.Domain/AggregatesModel/BuyerAggregate/CardType.cs` | ソース | ドメインモデル |
