# 機能設計書 25-在庫検証待ち処理

## 概要

本ドキュメントは、eShopシステムにおける在庫検証待ち処理機能の設計を記述するものである。注文が猶予期間を経過した後、在庫確認プロセスを開始するためにステータスを「AwaitingValidation」に遷移させる機能仕様を定義する。

### 本機能の処理概要

在庫検証待ち処理機能は、注文作成後の猶予期間（Grace Period）が経過したタイミングで、注文ステータスを「Submitted」から「AwaitingValidation」に変更する機能である。この遷移により、在庫確認サービスへの検証依頼が開始される。

**業務上の目的・背景**：ECサイトでは、顧客が注文を確定した直後にキャンセルする可能性がある。猶予期間を設けることで、不要な在庫ロックや決済処理を回避し、システムリソースを効率的に活用できる。猶予期間経過後に初めて在庫確認・決済処理を開始することで、実際に処理が必要な注文のみを対象とした効率的なワークフローを実現する。

**機能の利用シーン**：
- OrderProcessorバックグラウンドサービスが定期的に猶予期間経過注文を検出し、本機能を呼び出す
- GracePeriodConfirmedIntegrationEventを受信した際に実行される

**主要な処理内容**：
1. GracePeriodConfirmedIntegrationEventを受信
2. SetAwaitingValidationOrderStatusCommandを生成
3. 注文ステータスをAwaitingValidationに変更
4. OrderStatusChangedToAwaitingValidationDomainEventを発行
5. ドメインイベントハンドラが統合イベントを発行し、在庫確認サービスに通知

**関連システム・外部連携**：
- OrderProcessor（GracePeriodManagerService）から統合イベントを受信
- イベントバスを通じてOrderStatusChangedToAwaitingValidationIntegrationEventを発行
- Catalog.APIの在庫確認機能と連携

**権限による制御**：
- システム内部処理のため、外部からの直接呼び出しは不可
- イベント駆動による自動実行

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 5 | 注文履歴画面 | 結果表示画面 | AwaitingValidationステータスの注文を表示 |

## 機能種別

バックグラウンド処理 / ステータス変更処理 / イベント駆動

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| OrderId | int | Yes | ステータス変更対象の注文ID | GracePeriodConfirmedIntegrationEventから取得 |

### 入力データソース

- イベントバス経由でGracePeriodConfirmedIntegrationEventを受信

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 成功時 | bool | true（ステータス変更成功） |
| 失敗時 | bool | false（注文が存在しない場合） |

### 出力先

- データベース（注文ステータス更新）
- イベントバス（OrderStatusChangedToAwaitingValidationIntegrationEvent）

## 処理フロー

### 処理シーケンス

```
1. 統合イベントの受信
   └─ GracePeriodConfirmedIntegrationEventをイベントバスから受信

2. GracePeriodConfirmedIntegrationEventHandlerの実行
   └─ イベントからOrderIdを取得
   └─ SetAwaitingValidationOrderStatusCommandを作成

3. コマンドの送信
   └─ MediatR.Send()でコマンドを送信

4. SetAwaitingValidationOrderStatusCommandHandlerの実行
   └─ IOrderRepositoryから注文を取得
   └─ 注文が存在しない場合はfalseを返却

5. Order.SetAwaitingValidationStatus()の実行
   └─ 事前条件: Submittedステータスであること
   └─ 条件を満たす場合、ステータスをAwaitingValidationに変更
   └─ OrderStatusChangedToAwaitingValidationDomainEventを発行

6. UnitOfWorkでの永続化
   └─ SaveEntitiesAsync()でデータベースに保存
   └─ ドメインイベントをディスパッチ

7. OrderStatusChangedToAwaitingValidationDomainEventHandlerの実行
   └─ 注文情報と購買者情報を取得
   └─ 注文アイテムの在庫確認リストを作成
   └─ OrderStatusChangedToAwaitingValidationIntegrationEventを発行
```

### フローチャート

```mermaid
flowchart TD
    A[GracePeriodConfirmedIntegrationEvent受信] --> B[EventHandler]
    B --> C[SetAwaitingValidationOrderStatusCommand作成]
    C --> D[MediatR.Send]
    D --> E[CommandHandler]
    E --> F[注文取得]
    F --> G{注文存在?}
    G -->|No| H[false返却]
    G -->|Yes| I{Submittedステータス?}
    I -->|No| J[何もしない]
    J --> K[true返却]
    I -->|Yes| L[ステータスをAwaitingValidationに変更]
    L --> M[DomainEvent発行]
    M --> N[データベース保存]
    N --> O[統合イベント発行]
    O --> P[在庫確認サービスへ通知]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-25-01 | 遷移元ステータス制限 | Submittedステータスからのみ遷移可能 | SetAwaitingValidationStatus呼び出し時 |
| BR-25-02 | 在庫確認リスト生成 | 注文アイテムからProductIdとUnitsのリストを生成 | 統合イベント発行時 |
| BR-25-03 | 猶予期間経過条件 | 注文日時から設定された猶予期間が経過していること | イベント発行元で検証 |

### 計算ロジック

特になし（ステータス遷移ロジックのみ）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 注文取得 | ordering.orders | SELECT | 注文IDで注文を取得 |
| ステータス更新 | ordering.orders | UPDATE | OrderStatusをAwaitingValidationに更新 |
| 購買者取得 | ordering.buyers | SELECT | 統合イベント発行用に購買者情報を取得 |

### テーブル別操作詳細

#### ordering.orders

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id | WHERE Id = @OrderNumber | 注文取得 |
| UPDATE | OrderStatus | 'AwaitingValidation' (2) | 列挙型の値 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | false返却 | 注文が存在しない | ログ出力のみ |
| - | 無視 | Submitted以外のステータス | 何もせず正常終了 |

### リトライ仕様

- イベントハンドラのリトライはイベントバスの設定に依存
- 冪等性が保証されているためリトライは安全

## トランザクション仕様

- UnitOfWorkパターンによりトランザクション管理
- SaveEntitiesAsync()でコミット
- 例外発生時は自動ロールバック
- ドメインイベントはトランザクションコミット後に発行

## パフォーマンス要件

- バックグラウンド処理のため、即時性は不要
- バッチ的に複数の注文を処理する場合も対応

## セキュリティ考慮事項

- 内部イベント駆動のため、外部からの直接アクセスは不可
- イベントバスの認証・認可に依存

## 備考

- GracePeriodManagerServiceとの連携で動作
- 在庫確認サービス（Catalog.API）への通知トリガー
- 注文ライフサイクルの重要な遷移ポイント

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、統合イベントとコマンドの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | GracePeriodConfirmedIntegrationEvent.cs | `src/Ordering.API/Application/IntegrationEvents/Events/GracePeriodConfirmedIntegrationEvent.cs` | 入力となる統合イベントの構造（OrderIdのみ） |
| 1-2 | SetAwaitingValidationOrderStatusCommand.cs | `src/Ordering.API/Application/Commands/SetAwaitingValidationOrderStatusCommand.cs` | コマンドの構造（OrderNumberのみ） |

**読解のコツ**: GracePeriodConfirmedIntegrationEventはOrderProcessor（外部サービス）から発行され、Ordering.APIで受信される。サービス間の連携はすべて統合イベントを介して行われる。

#### Step 2: イベントハンドラを理解する

統合イベントを受信してコマンドを発行する流れを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | GracePeriodConfirmedIntegrationEventHandler.cs | `src/Ordering.API/Application/IntegrationEvents/EventHandling/GracePeriodConfirmedIntegrationEventHandler.cs` | イベントからコマンドへの変換 |

**主要処理フロー**:
1. **15-29行目**: Handle()メソッド全体
2. **17行目**: ログ出力（イベント受信の記録）
3. **19行目**: SetAwaitingValidationOrderStatusCommandを作成
4. **28行目**: MediatR.Send()でコマンドを送信

#### Step 3: コマンドハンドラを理解する

コマンドがどのように処理されるかを追跡する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | SetAwaitingValidationOrderStatusCommandHandler.cs | `src/Ordering.API/Application/Commands/SetAwaitingValidationOrderStatusCommandHandler.cs` | コマンドハンドラの処理フロー |

**主要処理フロー**:
- **19-29行目**: Handle()メソッド
- **21行目**: リポジトリから注文を取得
- **22-25行目**: 注文が存在しない場合はfalseを返却
- **27行目**: Order.SetAwaitingValidationStatus()を呼び出し
- **28行目**: UnitOfWorkで永続化

#### Step 4: ドメインロジックを理解する

注文エンティティ内でのステータス変更を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Order.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | SetAwaitingValidationStatus()メソッド |

**主要処理フロー**:
- **99-106行目**: SetAwaitingValidationStatus()メソッド
- **101行目**: Submittedからのみ遷移可能をチェック
- **103行目**: ドメインイベント発行
- **104行目**: ステータスをAwaitingValidationに変更

#### Step 5: ドメインイベントハンドラを理解する

ドメインイベントがどのように統合イベントに変換されるかを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | OrderStatusChangedToAwaitingValidationDomainEventHandler.cs | `src/Ordering.API/Application/DomainEventHandlers/OrderStatusChangedToAwaitingValidationDomainEventHandler.cs` | 統合イベント発行処理 |

**主要処理フロー**:
- **23-35行目**: Handle()メソッド
- **27-28行目**: 注文と購買者情報を取得
- **30-31行目**: 注文アイテムから在庫確認リストを作成
- **33行目**: OrderStatusChangedToAwaitingValidationIntegrationEventを作成
- **34行目**: イベントバスに発行

### プログラム呼び出し階層図

```
GracePeriodConfirmedIntegrationEvent（イベントバス経由）
    │
    └─ GracePeriodConfirmedIntegrationEventHandler.Handle()
           │
           ├─ SetAwaitingValidationOrderStatusCommand 作成
           │
           └─ IMediator.Send()
                  │
                  └─ SetAwaitingValidationOrderStatusCommandHandler.Handle()
                         │
                         ├─ IOrderRepository.GetAsync()
                         │
                         ├─ Order.SetAwaitingValidationStatus()
                         │      │
                         │      └─ AddDomainEvent(OrderStatusChangedToAwaitingValidationDomainEvent)
                         │
                         └─ IUnitOfWork.SaveEntitiesAsync()
                                │
                                └─ MediatR Dispatch
                                       │
                                       └─ OrderStatusChangedToAwaitingValidationDomainEventHandler.Handle()
                                              │
                                              ├─ IOrderRepository.GetAsync()
                                              │
                                              ├─ IBuyerRepository.FindByIdAsync()
                                              │
                                              ├─ OrderStockItem リスト作成
                                              │
                                              └─ IOrderingIntegrationEventService.AddAndSaveEventAsync()
                                                     │
                                                     └─ OrderStatusChangedToAwaitingValidationIntegrationEvent
```

### データフロー図

```
[入力]                              [処理]                                [出力]

GracePeriodConfirmed ────▶ IntegrationEventHandler ────────────▶ コマンド発行
IntegrationEvent                    │
  │                                 ▼
  └─ OrderId              SetAwaitingValidation
                          OrderStatusCommandHandler
                                    │
                                    ▼
                          IOrderRepository.GetAsync() ◀────────── Orders Table
                                    │
                                    ▼
                          Order.SetAwaitingValidationStatus()
                                    │
                                    ├─ OrderStatus = AwaitingValidation
                                    │
                                    └─ DomainEvent
                                           │
                                           ▼
                          DomainEventHandler
                                    │
                                    ├─ OrderStockItem リスト
                                    │      └─ ProductId, Units
                                    │
                                    └─ OrderStatusChangedTo
                                       AwaitingValidationIntegrationEvent
                                           │
                                           ▼
                                    EventBus (RabbitMQ)
                                           │
                                           ▼
                                    Catalog.API（在庫確認）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| GracePeriodConfirmedIntegrationEvent.cs | `src/Ordering.API/Application/IntegrationEvents/Events/GracePeriodConfirmedIntegrationEvent.cs` | ソース | 入力統合イベント |
| GracePeriodConfirmedIntegrationEventHandler.cs | `src/Ordering.API/Application/IntegrationEvents/EventHandling/GracePeriodConfirmedIntegrationEventHandler.cs` | ソース | 統合イベントハンドラ |
| SetAwaitingValidationOrderStatusCommand.cs | `src/Ordering.API/Application/Commands/SetAwaitingValidationOrderStatusCommand.cs` | ソース | コマンド定義 |
| SetAwaitingValidationOrderStatusCommandHandler.cs | `src/Ordering.API/Application/Commands/SetAwaitingValidationOrderStatusCommandHandler.cs` | ソース | コマンドハンドラ |
| Order.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | ソース | 注文エンティティ |
| OrderStatusChangedToAwaitingValidationDomainEvent.cs | `src/Ordering.Domain/Events/OrderStatusChangedToAwaitingValidationDomainEvent.cs` | ソース | ドメインイベント |
| OrderStatusChangedToAwaitingValidationDomainEventHandler.cs | `src/Ordering.API/Application/DomainEventHandlers/OrderStatusChangedToAwaitingValidationDomainEventHandler.cs` | ソース | ドメインイベントハンドラ |
| OrderStatusChangedToAwaitingValidationIntegrationEvent.cs | `src/Ordering.API/Application/IntegrationEvents/Events/OrderStatusChangedToAwaitingValidationIntegrationEvent.cs` | ソース | 出力統合イベント |
