# 機能設計書 26-在庫確認済み処理

## 概要

本ドキュメントは、eShopシステムにおける在庫確認済み処理機能の設計を記述するものである。在庫確認サービスからの確認完了通知を受けて、注文ステータスを「StockConfirmed」に遷移させる機能仕様を定義する。

### 本機能の処理概要

在庫確認済み処理機能は、Catalog.APIなどの在庫確認サービスから「在庫あり」の通知を受けた際に、注文ステータスを「AwaitingValidation」から「StockConfirmed」に変更する機能である。この遷移により、決済処理への進行が可能になる。

**業務上の目的・背景**：ECサイトの注文処理において、在庫確認は決済処理の前提条件である。在庫がない商品に対して決済を行うことは、顧客体験の悪化や在庫管理の混乱を招く。在庫確認が完了した注文のみを決済フローに進めることで、スムーズな注文処理と正確な在庫管理を実現する。

**機能の利用シーン**：
- 在庫確認サービスが注文アイテムの在庫をチェックし、すべての商品の在庫が確認できた場合
- OrderStockConfirmedIntegrationEventを受信した際に実行される

**主要な処理内容**：
1. OrderStockConfirmedIntegrationEventを受信
2. SetStockConfirmedOrderStatusCommandを生成
3. 処理シミュレーション（10秒の遅延）
4. 注文ステータスをStockConfirmedに変更
5. OrderStatusChangedToStockConfirmedDomainEventを発行
6. ドメインイベントハンドラが統合イベントを発行し、決済サービスに通知

**関連システム・外部連携**：
- Catalog.API（在庫確認サービス）からOrderStockConfirmedIntegrationEventを受信
- イベントバスを通じてOrderStatusChangedToStockConfirmedIntegrationEventを発行
- PaymentProcessor（決済サービス）と連携

**権限による制御**：
- システム内部処理のため、外部からの直接呼び出しは不可
- イベント駆動による自動実行

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 5 | 注文履歴画面 | 結果表示画面 | StockConfirmedステータスの注文を表示 |

## 機能種別

バックグラウンド処理 / ステータス変更処理 / イベント駆動

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| OrderId | int | Yes | ステータス変更対象の注文ID | OrderStockConfirmedIntegrationEventから取得 |

### 入力データソース

- イベントバス経由でOrderStockConfirmedIntegrationEventを受信

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 成功時 | bool | true（ステータス変更成功） |
| 失敗時 | bool | false（注文が存在しない場合） |

### 出力先

- データベース（注文ステータス更新、説明文更新）
- イベントバス（OrderStatusChangedToStockConfirmedIntegrationEvent）

## 処理フロー

### 処理シーケンス

```
1. 統合イベントの受信
   └─ OrderStockConfirmedIntegrationEventをイベントバスから受信

2. OrderStockConfirmedIntegrationEventHandlerの実行
   └─ イベントからOrderIdを取得
   └─ SetStockConfirmedOrderStatusCommandを作成

3. コマンドの送信
   └─ MediatR.Send()でコマンドを送信

4. SetStockConfirmedOrderStatusCommandHandlerの実行
   └─ 処理シミュレーション（10秒遅延）
   └─ IOrderRepositoryから注文を取得
   └─ 注文が存在しない場合はfalseを返却

5. Order.SetStockConfirmedStatus()の実行
   └─ 事前条件: AwaitingValidationステータスであること
   └─ 条件を満たす場合、ステータスをStockConfirmedに変更
   └─ 説明文を設定
   └─ OrderStatusChangedToStockConfirmedDomainEventを発行

6. UnitOfWorkでの永続化
   └─ SaveEntitiesAsync()でデータベースに保存
   └─ ドメインイベントをディスパッチ

7. OrderStatusChangedToStockConfirmedDomainEventHandlerの実行
   └─ 注文情報と購買者情報を取得
   └─ OrderStatusChangedToStockConfirmedIntegrationEventを発行
```

### フローチャート

```mermaid
flowchart TD
    A[OrderStockConfirmedIntegrationEvent受信] --> B[EventHandler]
    B --> C[SetStockConfirmedOrderStatusCommand作成]
    C --> D[MediatR.Send]
    D --> E[CommandHandler]
    E --> F[10秒遅延シミュレーション]
    F --> G[注文取得]
    G --> H{注文存在?}
    H -->|No| I[false返却]
    H -->|Yes| J{AwaitingValidationステータス?}
    J -->|No| K[何もしない]
    K --> L[true返却]
    J -->|Yes| M[ステータスをStockConfirmedに変更]
    M --> N[説明文設定]
    N --> O[DomainEvent発行]
    O --> P[データベース保存]
    P --> Q[統合イベント発行]
    Q --> R[決済サービスへ通知]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-26-01 | 遷移元ステータス制限 | AwaitingValidationステータスからのみ遷移可能 | SetStockConfirmedStatus呼び出し時 |
| BR-26-02 | 在庫確認完了メッセージ | 説明文に「All the items were confirmed with available stock.」を設定 | ステータス変更時 |
| BR-26-03 | 処理遅延シミュレーション | 本番環境での実処理時間を模擬するため10秒の遅延を挿入 | コマンドハンドラ実行時 |

### 計算ロジック

特になし（ステータス遷移ロジックのみ）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 注文取得 | ordering.orders | SELECT | 注文IDで注文を取得 |
| ステータス更新 | ordering.orders | UPDATE | OrderStatusをStockConfirmedに更新、Descriptionを更新 |
| 購買者取得 | ordering.buyers | SELECT | 統合イベント発行用に購買者情報を取得 |

### テーブル別操作詳細

#### ordering.orders

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id | WHERE Id = @OrderNumber | 注文取得 |
| UPDATE | OrderStatus | 'StockConfirmed' (3) | 列挙型の値 |
| UPDATE | Description | 'All the items were confirmed with available stock.' | 固定メッセージ |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | false返却 | 注文が存在しない | ログ出力のみ |
| - | 無視 | AwaitingValidation以外のステータス | 何もせず正常終了 |

### リトライ仕様

- イベントハンドラのリトライはイベントバスの設定に依存
- 冪等性が保証されているためリトライは安全

## トランザクション仕様

- UnitOfWorkパターンによりトランザクション管理
- SaveEntitiesAsync()でコミット
- 例外発生時は自動ロールバック
- ドメインイベントはトランザクションコミット後に発行

## パフォーマンス要件

- 処理時間: 約10秒（シミュレーション遅延含む）
- バックグラウンド処理のため、即時性は不要

## セキュリティ考慮事項

- 内部イベント駆動のため、外部からの直接アクセスは不可
- イベントバスの認証・認可に依存

## 備考

- 10秒の遅延は本番環境での在庫確認処理時間のシミュレーション
- PaymentProcessorへの決済処理通知のトリガー
- 注文ライフサイクルの重要な遷移ポイント

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、統合イベントとコマンドの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | OrderStockConfirmedIntegrationEvent.cs | `src/Ordering.API/Application/IntegrationEvents/Events/OrderStockConfirmedIntegrationEvent.cs` | 入力となる統合イベントの構造（OrderIdのみ） |
| 1-2 | SetStockConfirmedOrderStatusCommand.cs | `src/Ordering.API/Application/Commands/SetStockConfirmedOrderStatusCommand.cs` | コマンドの構造（OrderNumberのみ） |

**読解のコツ**: OrderStockConfirmedIntegrationEventはCatalog.API（在庫確認サービス）から発行され、Ordering.APIで受信される。

#### Step 2: イベントハンドラを理解する

統合イベントを受信してコマンドを発行する流れを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | OrderStockConfirmedIntegrationEventHandler.cs | `src/Ordering.API/Application/IntegrationEvents/EventHandling/OrderStockConfirmedIntegrationEventHandler.cs` | イベントからコマンドへの変換 |

**主要処理フロー**:
1. **8-23行目**: Handle()メソッド全体
2. **10行目**: ログ出力（イベント受信の記録）
3. **12行目**: SetStockConfirmedOrderStatusCommandを作成
4. **21行目**: MediatR.Send()でコマンドを送信

#### Step 3: コマンドハンドラを理解する

コマンドがどのように処理されるかを追跡する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | SetStockConfirmedOrderStatusCommandHandler.cs | `src/Ordering.API/Application/Commands/SetStockConfirmedOrderStatusCommandHandler.cs` | コマンドハンドラの処理フロー |

**主要処理フロー**:
- **19-32行目**: Handle()メソッド
- **22行目**: 10秒の遅延シミュレーション（`await Task.Delay(10000, cancellationToken)`）
- **24行目**: リポジトリから注文を取得
- **25-28行目**: 注文が存在しない場合はfalseを返却
- **30行目**: Order.SetStockConfirmedStatus()を呼び出し
- **31行目**: UnitOfWorkで永続化

#### Step 4: ドメインロジックを理解する

注文エンティティ内でのステータス変更を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Order.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | SetStockConfirmedStatus()メソッド |

**主要処理フロー**:
- **108-117行目**: SetStockConfirmedStatus()メソッド
- **110行目**: AwaitingValidationからのみ遷移可能をチェック
- **112行目**: ドメインイベント発行
- **114行目**: ステータスをStockConfirmedに変更
- **115行目**: 説明文を設定

#### Step 5: ドメインイベントハンドラを理解する

ドメインイベントがどのように統合イベントに変換されるかを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | OrderStatusChangedToStockConfirmedDomainEventHandler.cs | `src/Ordering.API/Application/DomainEventHandlers/OrderStatusChangedToStockConfirmedDomainEventHandler.cs` | 統合イベント発行処理 |

**主要処理フロー**:
- **23-32行目**: Handle()メソッド
- **27-28行目**: 注文と購買者情報を取得
- **30行目**: OrderStatusChangedToStockConfirmedIntegrationEventを作成
- **31行目**: イベントバスに発行

### プログラム呼び出し階層図

```
OrderStockConfirmedIntegrationEvent（イベントバス経由）
    │
    └─ OrderStockConfirmedIntegrationEventHandler.Handle()
           │
           ├─ SetStockConfirmedOrderStatusCommand 作成
           │
           └─ IMediator.Send()
                  │
                  └─ SetStockConfirmedOrderStatusCommandHandler.Handle()
                         │
                         ├─ Task.Delay(10000) [シミュレーション]
                         │
                         ├─ IOrderRepository.GetAsync()
                         │
                         ├─ Order.SetStockConfirmedStatus()
                         │      │
                         │      └─ AddDomainEvent(OrderStatusChangedToStockConfirmedDomainEvent)
                         │
                         └─ IUnitOfWork.SaveEntitiesAsync()
                                │
                                └─ MediatR Dispatch
                                       │
                                       └─ OrderStatusChangedToStockConfirmedDomainEventHandler.Handle()
                                              │
                                              ├─ IOrderRepository.GetAsync()
                                              │
                                              ├─ IBuyerRepository.FindByIdAsync()
                                              │
                                              └─ IOrderingIntegrationEventService.AddAndSaveEventAsync()
                                                     │
                                                     └─ OrderStatusChangedToStockConfirmedIntegrationEvent
```

### データフロー図

```
[入力]                              [処理]                                [出力]

OrderStockConfirmed ─────▶ IntegrationEventHandler ────────────▶ コマンド発行
IntegrationEvent                    │
  │                                 ▼
  └─ OrderId              SetStockConfirmed
                          OrderStatusCommandHandler
                                    │
                                    ├─ Task.Delay(10000)
                                    │
                                    ▼
                          IOrderRepository.GetAsync() ◀────────── Orders Table
                                    │
                                    ▼
                          Order.SetStockConfirmedStatus()
                                    │
                                    ├─ OrderStatus = StockConfirmed
                                    │
                                    ├─ Description = "All the items were confirmed..."
                                    │
                                    └─ DomainEvent
                                           │
                                           ▼
                          DomainEventHandler
                                    │
                                    └─ OrderStatusChangedTo
                                       StockConfirmedIntegrationEvent
                                           │
                                           ▼
                                    EventBus (RabbitMQ)
                                           │
                                           ▼
                                    PaymentProcessor（決済処理）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| OrderStockConfirmedIntegrationEvent.cs | `src/Ordering.API/Application/IntegrationEvents/Events/OrderStockConfirmedIntegrationEvent.cs` | ソース | 入力統合イベント |
| OrderStockConfirmedIntegrationEventHandler.cs | `src/Ordering.API/Application/IntegrationEvents/EventHandling/OrderStockConfirmedIntegrationEventHandler.cs` | ソース | 統合イベントハンドラ |
| SetStockConfirmedOrderStatusCommand.cs | `src/Ordering.API/Application/Commands/SetStockConfirmedOrderStatusCommand.cs` | ソース | コマンド定義 |
| SetStockConfirmedOrderStatusCommandHandler.cs | `src/Ordering.API/Application/Commands/SetStockConfirmedOrderStatusCommandHandler.cs` | ソース | コマンドハンドラ |
| Order.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | ソース | 注文エンティティ |
| OrderStatusChangedToStockConfirmedDomainEvent.cs | `src/Ordering.Domain/Events/OrderStatusChangedToStockConfirmedDomainEvent.cs` | ソース | ドメインイベント |
| OrderStatusChangedToStockConfirmedDomainEventHandler.cs | `src/Ordering.API/Application/DomainEventHandlers/OrderStatusChangedToStockConfirmedDomainEventHandler.cs` | ソース | ドメインイベントハンドラ |
| OrderStatusChangedToStockConfirmedIntegrationEvent.cs | `src/Ordering.API/Application/IntegrationEvents/Events/OrderStatusChangedToStockConfirmedIntegrationEvent.cs` | ソース | 出力統合イベント |
