# 機能設計書 27-支払い完了処理

## 概要

本ドキュメントは、eShopシステムにおける支払い完了処理機能の設計を記述するものである。決済サービスからの支払い成功通知を受けて、注文ステータスを「Paid」に遷移させる機能仕様を定義する。

### 本機能の処理概要

支払い完了処理機能は、PaymentProcessorなどの決済サービスから「決済成功」の通知を受けた際に、注文ステータスを「StockConfirmed」から「Paid」に変更する機能である。この遷移により、発送処理への進行が可能になる。

**業務上の目的・背景**：ECサイトの注文処理において、決済完了は発送処理の前提条件である。決済が完了していない注文に対して商品を発送することは、金銭的リスクを生む。決済完了した注文のみを発送フローに進めることで、安全な取引を保証する。また、Paid状態への遷移は、外部Webhookへの通知トリガーとなり、外部システムとの連携を可能にする。

**機能の利用シーン**：
- 決済サービス（PaymentProcessor）が決済処理を完了し、成功を通知した場合
- OrderPaymentSucceededIntegrationEventを受信した際に実行される

**主要な処理内容**：
1. OrderPaymentSucceededIntegrationEventを受信
2. SetPaidOrderStatusCommandを生成
3. 処理シミュレーション（10秒の遅延）
4. 注文ステータスをPaidに変更
5. OrderStatusChangedToPaidDomainEventを発行
6. ドメインイベントハンドラが統合イベントを発行し、Webhooksサービスに通知

**関連システム・外部連携**：
- PaymentProcessor（決済サービス）からOrderPaymentSucceededIntegrationEventを受信
- イベントバスを通じてOrderStatusChangedToPaidIntegrationEventを発行
- Webhooks.APIが「OrderPaid」イベントを外部エンドポイントに配信

**権限による制御**：
- システム内部処理のため、外部からの直接呼び出しは不可
- イベント駆動による自動実行

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 5 | 注文履歴画面 | 結果表示画面 | Paidステータスの注文を表示 |
| 21 | ホーム画面（WebhookClient） | 連携画面 | OrderPaidイベントによるWebhook通知を受信 |

## 機能種別

バックグラウンド処理 / ステータス変更処理 / イベント駆動

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| OrderId | int | Yes | ステータス変更対象の注文ID | OrderPaymentSucceededIntegrationEventから取得 |

### 入力データソース

- イベントバス経由でOrderPaymentSucceededIntegrationEventを受信

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 成功時 | bool | true（ステータス変更成功） |
| 失敗時 | bool | false（注文が存在しない場合） |

### 出力先

- データベース（注文ステータス更新、説明文更新）
- イベントバス（OrderStatusChangedToPaidIntegrationEvent）
- Webhooks（外部エンドポイント）

## 処理フロー

### 処理シーケンス

```
1. 統合イベントの受信
   └─ OrderPaymentSucceededIntegrationEventをイベントバスから受信

2. OrderPaymentSucceededIntegrationEventHandlerの実行
   └─ イベントからOrderIdを取得
   └─ SetPaidOrderStatusCommandを作成

3. コマンドの送信
   └─ MediatR.Send()でコマンドを送信

4. SetPaidOrderStatusCommandHandlerの実行
   └─ 処理シミュレーション（10秒遅延）
   └─ IOrderRepositoryから注文を取得
   └─ 注文が存在しない場合はfalseを返却

5. Order.SetPaidStatus()の実行
   └─ 事前条件: StockConfirmedステータスであること
   └─ 条件を満たす場合、ステータスをPaidに変更
   └─ 説明文を設定
   └─ OrderStatusChangedToPaidDomainEventを発行

6. UnitOfWorkでの永続化
   └─ SaveEntitiesAsync()でデータベースに保存
   └─ ドメインイベントをディスパッチ

7. OrderStatusChangedToPaidDomainEventHandlerの実行
   └─ 注文情報と購買者情報を取得
   └─ 注文アイテムの在庫減算リストを作成
   └─ OrderStatusChangedToPaidIntegrationEventを発行

8. Webhooks.APIによる外部通知
   └─ 登録済みWebhookエンドポイントにOrderPaidイベントを配信
```

### フローチャート

```mermaid
flowchart TD
    A[OrderPaymentSucceededIntegrationEvent受信] --> B[EventHandler]
    B --> C[SetPaidOrderStatusCommand作成]
    C --> D[MediatR.Send]
    D --> E[CommandHandler]
    E --> F[10秒遅延シミュレーション]
    F --> G[注文取得]
    G --> H{注文存在?}
    H -->|No| I[false返却]
    H -->|Yes| J{StockConfirmedステータス?}
    J -->|No| K[何もしない]
    K --> L[true返却]
    J -->|Yes| M[ステータスをPaidに変更]
    M --> N[説明文設定]
    N --> O[DomainEvent発行]
    O --> P[データベース保存]
    P --> Q[統合イベント発行]
    Q --> R[Webhook通知]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-27-01 | 遷移元ステータス制限 | StockConfirmedステータスからのみ遷移可能 | SetPaidStatus呼び出し時 |
| BR-27-02 | 支払い完了メッセージ | 説明文に模擬銀行口座情報を含むメッセージを設定 | ステータス変更時 |
| BR-27-03 | 処理遅延シミュレーション | 本番環境での決済確認時間を模擬するため10秒の遅延を挿入 | コマンドハンドラ実行時 |
| BR-27-04 | 在庫減算リスト生成 | 注文アイテムからProductIdとUnitsのリストを生成（在庫減算用） | 統合イベント発行時 |

### 計算ロジック

特になし（ステータス遷移ロジックのみ）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 注文取得 | ordering.orders | SELECT | 注文IDで注文を取得 |
| ステータス更新 | ordering.orders | UPDATE | OrderStatusをPaidに更新、Descriptionを更新 |
| 購買者取得 | ordering.buyers | SELECT | 統合イベント発行用に購買者情報を取得 |

### テーブル別操作詳細

#### ordering.orders

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id | WHERE Id = @OrderNumber | 注文取得 |
| UPDATE | OrderStatus | 'Paid' (4) | 列挙型の値 |
| UPDATE | Description | 'The payment was performed at a simulated "American Bank checking bank account ending on XX35071"' | 模擬決済メッセージ |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | false返却 | 注文が存在しない | ログ出力のみ |
| - | 無視 | StockConfirmed以外のステータス | 何もせず正常終了 |

### リトライ仕様

- イベントハンドラのリトライはイベントバスの設定に依存
- 冪等性が保証されているためリトライは安全

## トランザクション仕様

- UnitOfWorkパターンによりトランザクション管理
- SaveEntitiesAsync()でコミット
- 例外発生時は自動ロールバック
- ドメインイベントはトランザクションコミット後に発行

## パフォーマンス要件

- 処理時間: 約10秒（シミュレーション遅延含む）
- バックグラウンド処理のため、即時性は不要

## セキュリティ考慮事項

- 内部イベント駆動のため、外部からの直接アクセスは不可
- イベントバスの認証・認可に依存
- 模擬決済情報は本番環境では適切な値に置き換え

## 備考

- 10秒の遅延は本番環境での決済検証処理時間のシミュレーション
- Webhooks.APIへの通知によりサードパーティシステム連携を実現
- 注文ライフサイクルの重要な遷移ポイント
- Paid状態はキャンセル不可（BR-21-01との関連）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、統合イベントとコマンドの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | OrderPaymentSucceededIntegrationEvent.cs | `src/Ordering.API/Application/IntegrationEvents/Events/OrderPaymentSucceededIntegrationEvent.cs` | 入力となる統合イベントの構造（OrderIdのみ） |
| 1-2 | SetPaidOrderStatusCommand.cs | `src/Ordering.API/Application/Commands/SetPaidOrderStatusCommand.cs` | コマンドの構造（OrderNumberのみ） |

**読解のコツ**: OrderPaymentSucceededIntegrationEventはPaymentProcessor（決済サービス）から発行され、Ordering.APIで受信される。

#### Step 2: イベントハンドラを理解する

統合イベントを受信してコマンドを発行する流れを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | OrderPaymentSucceededIntegrationEventHandler.cs | `src/Ordering.API/Application/IntegrationEvents/EventHandling/OrderPaymentSucceededIntegrationEventHandler.cs` | イベントからコマンドへの変換 |

**主要処理フロー**:
1. **8-23行目**: Handle()メソッド全体
2. **10行目**: ログ出力（イベント受信の記録）
3. **12行目**: SetPaidOrderStatusCommandを作成
4. **21行目**: MediatR.Send()でコマンドを送信

#### Step 3: コマンドハンドラを理解する

コマンドがどのように処理されるかを追跡する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | SetPaidOrderStatusCommandHandler.cs | `src/Ordering.API/Application/Commands/SetPaidOrderStatusCommandHandler.cs` | コマンドハンドラの処理フロー |

**主要処理フロー**:
- **19-32行目**: Handle()メソッド
- **22行目**: 10秒の遅延シミュレーション（`await Task.Delay(10000, cancellationToken)`）
- **24行目**: リポジトリから注文を取得
- **25-28行目**: 注文が存在しない場合はfalseを返却
- **30行目**: Order.SetPaidStatus()を呼び出し
- **31行目**: UnitOfWorkで永続化

#### Step 4: ドメインロジックを理解する

注文エンティティ内でのステータス変更を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Order.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | SetPaidStatus()メソッド |

**主要処理フロー**:
- **119-128行目**: SetPaidStatus()メソッド
- **121行目**: StockConfirmedからのみ遷移可能をチェック
- **123行目**: ドメインイベント発行
- **125行目**: ステータスをPaidに変更
- **126行目**: 説明文を設定（模擬銀行口座情報）

#### Step 5: ドメインイベントハンドラを理解する

ドメインイベントがどのように統合イベントに変換されるかを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | OrderStatusChangedToPaidDomainEventHandler.cs | `src/Ordering.API/Application/DomainEventHandlers/OrderStatusChangedToPaidDomainEventHandler.cs` | 統合イベント発行処理 |

**主要処理フロー**:
- **22-40行目**: Handle()メソッド
- **26-27行目**: 注文と購買者情報を取得
- **29-30行目**: 注文アイテムから在庫リストを作成
- **32-37行目**: OrderStatusChangedToPaidIntegrationEventを作成
- **39行目**: イベントバスに発行

### プログラム呼び出し階層図

```
OrderPaymentSucceededIntegrationEvent（イベントバス経由）
    │
    └─ OrderPaymentSucceededIntegrationEventHandler.Handle()
           │
           ├─ SetPaidOrderStatusCommand 作成
           │
           └─ IMediator.Send()
                  │
                  └─ SetPaidOrderStatusCommandHandler.Handle()
                         │
                         ├─ Task.Delay(10000) [シミュレーション]
                         │
                         ├─ IOrderRepository.GetAsync()
                         │
                         ├─ Order.SetPaidStatus()
                         │      │
                         │      └─ AddDomainEvent(OrderStatusChangedToPaidDomainEvent)
                         │
                         └─ IUnitOfWork.SaveEntitiesAsync()
                                │
                                └─ MediatR Dispatch
                                       │
                                       └─ OrderStatusChangedToPaidDomainEventHandler.Handle()
                                              │
                                              ├─ IOrderRepository.GetAsync()
                                              │
                                              ├─ IBuyerRepository.FindByIdAsync()
                                              │
                                              ├─ OrderStockItem リスト作成
                                              │
                                              └─ IOrderingIntegrationEventService.AddAndSaveEventAsync()
                                                     │
                                                     └─ OrderStatusChangedToPaidIntegrationEvent
                                                            │
                                                            └─ Webhooks.API → 外部エンドポイント
```

### データフロー図

```
[入力]                              [処理]                                [出力]

OrderPaymentSucceeded ───▶ IntegrationEventHandler ────────────▶ コマンド発行
IntegrationEvent                    │
  │                                 ▼
  └─ OrderId              SetPaidOrderStatusCommandHandler
                                    │
                                    ├─ Task.Delay(10000)
                                    │
                                    ▼
                          IOrderRepository.GetAsync() ◀────────── Orders Table
                                    │
                                    ▼
                          Order.SetPaidStatus()
                                    │
                                    ├─ OrderStatus = Paid
                                    │
                                    ├─ Description = "The payment was performed..."
                                    │
                                    └─ DomainEvent
                                           │
                                           ▼
                          DomainEventHandler
                                    │
                                    ├─ OrderStockItem リスト
                                    │      └─ ProductId, Units
                                    │
                                    └─ OrderStatusChangedTo
                                       PaidIntegrationEvent
                                           │
                                           ├─▶ EventBus (RabbitMQ)
                                           │
                                           └─▶ Webhooks.API
                                                  │
                                                  ▼
                                           外部Webhookエンドポイント
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| OrderPaymentSucceededIntegrationEvent.cs | `src/Ordering.API/Application/IntegrationEvents/Events/OrderPaymentSucceededIntegrationEvent.cs` | ソース | 入力統合イベント |
| OrderPaymentSucceededIntegrationEventHandler.cs | `src/Ordering.API/Application/IntegrationEvents/EventHandling/OrderPaymentSucceededIntegrationEventHandler.cs` | ソース | 統合イベントハンドラ |
| SetPaidOrderStatusCommand.cs | `src/Ordering.API/Application/Commands/SetPaidOrderStatusCommand.cs` | ソース | コマンド定義 |
| SetPaidOrderStatusCommandHandler.cs | `src/Ordering.API/Application/Commands/SetPaidOrderStatusCommandHandler.cs` | ソース | コマンドハンドラ |
| Order.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | ソース | 注文エンティティ |
| OrderStatusChangedToPaidDomainEvent.cs | `src/Ordering.Domain/Events/OrderStatusChangedToPaidDomainEvent.cs` | ソース | ドメインイベント |
| OrderStatusChangedToPaidDomainEventHandler.cs | `src/Ordering.API/Application/DomainEventHandlers/OrderStatusChangedToPaidDomainEventHandler.cs` | ソース | ドメインイベントハンドラ |
| OrderStatusChangedToPaidIntegrationEvent.cs | `src/Ordering.API/Application/IntegrationEvents/Events/OrderStatusChangedToPaidIntegrationEvent.cs` | ソース | 出力統合イベント |
