# 機能設計書 28-在庫不足キャンセル処理

## 概要

本ドキュメントは、eShopシステムにおける在庫不足キャンセル処理機能の設計を記述するものである。在庫確認サービスからの在庫不足通知を受けて、注文を自動的にキャンセルする機能仕様を定義する。

### 本機能の処理概要

在庫不足キャンセル処理機能は、Catalog.APIなどの在庫確認サービスから「在庫不足」の通知を受けた際に、注文ステータスを「AwaitingValidation」から「Cancelled」に変更する機能である。どの商品の在庫が不足しているかを説明文に記録し、顧客が理由を確認できるようにする。

**業務上の目的・背景**：ECサイトでは、注文時点と在庫確認時点で在庫状況が変化する可能性がある。複数の顧客が同時に同一商品を注文した場合、一部の注文は在庫不足となる。在庫が確保できない注文を自動的にキャンセルし、不足商品名を記録することで、顧客への説明責任を果たし、顧客体験の向上を図る。

**機能の利用シーン**：
- 在庫確認サービスが注文アイテムの在庫をチェックし、1つ以上の商品で在庫が不足している場合
- OrderStockRejectedIntegrationEventを受信した際に実行される

**主要な処理内容**：
1. OrderStockRejectedIntegrationEventを受信
2. 在庫不足の商品IDリストを抽出
3. SetStockRejectedOrderStatusCommandを生成
4. 処理シミュレーション（10秒の遅延）
5. 注文ステータスをCancelledに変更
6. 不足商品名を説明文に記録

**関連システム・外部連携**：
- Catalog.API（在庫確認サービス）からOrderStockRejectedIntegrationEventを受信
- 在庫不足の商品情報（HasStock=false）を含む

**権限による制御**：
- システム内部処理のため、外部からの直接呼び出しは不可
- イベント駆動による自動実行

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 5 | 注文履歴画面 | 結果表示画面 | キャンセルされた注文と理由を表示 |

## 機能種別

バックグラウンド処理 / ステータス変更処理 / イベント駆動 / 自動キャンセル

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| OrderId | int | Yes | ステータス変更対象の注文ID | OrderStockRejectedIntegrationEventから取得 |
| OrderStockItems | List<ConfirmedOrderStockItem> | Yes | 在庫確認結果リスト | HasStock=falseの商品を抽出 |

### 入力データソース

- イベントバス経由でOrderStockRejectedIntegrationEventを受信
- 各商品のHasStockフラグから在庫不足商品を特定

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 成功時 | bool | true（ステータス変更成功） |
| 失敗時 | bool | false（注文が存在しない場合） |

### 出力先

- データベース（注文ステータス更新、説明文に不足商品名を記録）

## 処理フロー

### 処理シーケンス

```
1. 統合イベントの受信
   └─ OrderStockRejectedIntegrationEventをイベントバスから受信

2. OrderStockRejectedIntegrationEventHandlerの実行
   └─ イベントからOrderIdを取得
   └─ OrderStockItemsからHasStock=falseの商品IDを抽出
   └─ SetStockRejectedOrderStatusCommandを作成

3. コマンドの送信
   └─ MediatR.Send()でコマンドを送信

4. SetStockRejectedOrderStatusCommandHandlerの実行
   └─ 処理シミュレーション（10秒遅延）
   └─ IOrderRepositoryから注文を取得
   └─ 注文が存在しない場合はfalseを返却

5. Order.SetCancelledStatusWhenStockIsRejected()の実行
   └─ 事前条件: AwaitingValidationステータスであること
   └─ 条件を満たす場合、ステータスをCancelledに変更
   └─ 不足商品IDから商品名を特定
   └─ 説明文に不足商品名を記録

6. UnitOfWorkでの永続化
   └─ SaveEntitiesAsync()でデータベースに保存
```

### フローチャート

```mermaid
flowchart TD
    A[OrderStockRejectedIntegrationEvent受信] --> B[EventHandler]
    B --> C[在庫不足商品IDを抽出]
    C --> D[SetStockRejectedOrderStatusCommand作成]
    D --> E[MediatR.Send]
    E --> F[CommandHandler]
    F --> G[10秒遅延シミュレーション]
    G --> H[注文取得]
    H --> I{注文存在?}
    I -->|No| J[false返却]
    I -->|Yes| K{AwaitingValidationステータス?}
    K -->|No| L[何もしない]
    L --> M[true返却]
    K -->|Yes| N[ステータスをCancelledに変更]
    N --> O[不足商品名を説明文に記録]
    O --> P[データベース保存]
    P --> M
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-28-01 | 遷移元ステータス制限 | AwaitingValidationステータスからのみ遷移可能 | SetCancelledStatusWhenStockIsRejected呼び出し時 |
| BR-28-02 | 在庫不足商品の特定 | HasStock=falseの商品IDのみを抽出 | イベントハンドラで実行 |
| BR-28-03 | 不足商品名記録 | 注文アイテムから該当商品名を取得し、説明文に記録 | ステータス変更時 |
| BR-28-04 | 処理遅延シミュレーション | 本番環境での処理時間を模擬するため10秒の遅延を挿入 | コマンドハンドラ実行時 |
| BR-28-05 | ドメインイベント非発行 | 在庫不足キャンセルではドメインイベントを発行しない | SetCancelledStatusWhenStockIsRejected内 |

### 計算ロジック

**不足商品名リストの生成**:
```
itemsStockRejectedProductNames = OrderItems
    .Where(c => orderStockRejectedItems.Contains(c.ProductId))
    .Select(c => c.ProductName)

Description = "The product items don't have stock: ({itemsStockRejectedProductNames})"
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 注文取得 | ordering.orders | SELECT | 注文IDで注文を取得（OrderItemsを含む） |
| ステータス更新 | ordering.orders | UPDATE | OrderStatusをCancelledに更新、Descriptionを更新 |

### テーブル別操作詳細

#### ordering.orders

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | Id | WHERE Id = @OrderNumber | 注文取得（Include OrderItems） |
| UPDATE | OrderStatus | 'Cancelled' (6) | 列挙型の値 |
| UPDATE | Description | "The product items don't have stock: ({商品名リスト})" | 動的メッセージ |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | false返却 | 注文が存在しない | ログ出力のみ |
| - | 無視 | AwaitingValidation以外のステータス | 何もせず正常終了 |

### リトライ仕様

- イベントハンドラのリトライはイベントバスの設定に依存
- 冪等性が保証されているためリトライは安全

## トランザクション仕様

- UnitOfWorkパターンによりトランザクション管理
- SaveEntitiesAsync()でコミット
- 例外発生時は自動ロールバック
- 注：このメソッドではドメインイベントを発行しない

## パフォーマンス要件

- 処理時間: 約10秒（シミュレーション遅延含む）
- バックグラウンド処理のため、即時性は不要

## セキュリティ考慮事項

- 内部イベント駆動のため、外部からの直接アクセスは不可
- イベントバスの認証・認可に依存

## 備考

- 10秒の遅延は本番環境での在庫拒否処理時間のシミュレーション
- SetCancelledStatus()とは異なり、ドメインイベントを発行しない
- 説明文に不足商品名を記録することで、顧客がキャンセル理由を確認可能
- 通常のキャンセル（顧客起因）と在庫不足キャンセル（システム起因）を区別

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、統合イベントとコマンドの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | OrderStockRejectedIntegrationEvent.cs | `src/Ordering.API/Application/IntegrationEvents/Events/OrderStockRejectedIntegrationEvent.cs` | 入力となる統合イベントの構造（OrderId, OrderStockItemsリスト） |
| 1-2 | SetStockRejectedOrderStatusCommand.cs | `src/Ordering.API/Application/Commands/SetStockRejectedOrderStatusCommand.cs` | コマンドの構造（OrderNumber, OrderStockItemsリスト） |

**読解のコツ**: OrderStockItemsには各商品の在庫状況（HasStock）が含まれる。HasStock=falseの商品が在庫不足商品である。

#### Step 2: イベントハンドラを理解する

統合イベントを受信してコマンドを発行する流れを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | OrderStockRejectedIntegrationEventHandler.cs | `src/Ordering.API/Application/IntegrationEvents/EventHandling/OrderStockRejectedIntegrationEventHandler.cs` | イベントからコマンドへの変換、在庫不足商品の抽出 |

**主要処理フロー**:
1. **6-26行目**: Handle()メソッド全体
2. **8行目**: ログ出力（イベント受信の記録）
3. **10-13行目**: HasStock=falseの商品IDを抽出
4. **15行目**: SetStockRejectedOrderStatusCommandを作成
5. **24行目**: MediatR.Send()でコマンドを送信

#### Step 3: コマンドハンドラを理解する

コマンドがどのように処理されるかを追跡する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | SetStockRejectedOrderStatusCommandHandler.cs | `src/Ordering.API/Application/Commands/SetStockRejectedOrderStatusCommandHandler.cs` | コマンドハンドラの処理フロー |

**主要処理フロー**:
- **19-33行目**: Handle()メソッド
- **22行目**: 10秒の遅延シミュレーション
- **24行目**: リポジトリから注文を取得
- **25-28行目**: 注文が存在しない場合はfalseを返却
- **30行目**: Order.SetCancelledStatusWhenStockIsRejected()を呼び出し
- **32行目**: UnitOfWorkで永続化

#### Step 4: ドメインロジックを理解する

注文エンティティ内でのステータス変更を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Order.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | SetCancelledStatusWhenStockIsRejected()メソッド |

**主要処理フロー**:
- **155-168行目**: SetCancelledStatusWhenStockIsRejected()メソッド
- **157行目**: AwaitingValidationからのみ遷移可能をチェック
- **159行目**: ステータスをCancelledに変更
- **161-163行目**: 不足商品IDから商品名を抽出
- **165行目**: 商品名をカンマ区切りで結合
- **166行目**: 説明文を設定（不足商品名を含む）

### プログラム呼び出し階層図

```
OrderStockRejectedIntegrationEvent（イベントバス経由）
    │
    └─ OrderStockRejectedIntegrationEventHandler.Handle()
           │
           ├─ HasStock=false の商品IDを抽出
           │      └─ FindAll(c => !c.HasStock).Select(c => c.ProductId)
           │
           ├─ SetStockRejectedOrderStatusCommand 作成
           │
           └─ IMediator.Send()
                  │
                  └─ SetStockRejectedOrderStatusCommandHandler.Handle()
                         │
                         ├─ Task.Delay(10000) [シミュレーション]
                         │
                         ├─ IOrderRepository.GetAsync()
                         │
                         ├─ Order.SetCancelledStatusWhenStockIsRejected()
                         │      │
                         │      ├─ OrderItems から商品名を抽出
                         │      │
                         │      ├─ OrderStatus = Cancelled
                         │      │
                         │      └─ Description = "The product items don't have stock..."
                         │
                         └─ IUnitOfWork.SaveEntitiesAsync()
```

### データフロー図

```
[入力]                              [処理]                                [出力]

OrderStockRejected ──────▶ IntegrationEventHandler ────────────▶ コマンド発行
IntegrationEvent                    │
  │                                 │
  ├─ OrderId                        ├─ 在庫不足商品IDを抽出
  │                                 │      └─ HasStock=false のみ
  └─ OrderStockItems                │
       │                            ▼
       ├─ ProductId          SetStockRejected
       └─ HasStock           OrderStatusCommandHandler
                                    │
                                    ├─ Task.Delay(10000)
                                    │
                                    ▼
                          IOrderRepository.GetAsync() ◀────────── Orders Table
                                    │                              (with OrderItems)
                                    ▼
                          Order.SetCancelledStatusWhenStockIsRejected()
                                    │
                                    ├─ 不足商品名を抽出
                                    │      └─ OrderItems.Where(ProductId in rejectedItems)
                                    │                 .Select(ProductName)
                                    │
                                    ├─ OrderStatus = Cancelled
                                    │
                                    └─ Description =
                                       "The product items don't have stock:
                                        ({商品名1}, {商品名2}, ...)"
                                           │
                                           ▼
                                    データベース保存
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| OrderStockRejectedIntegrationEvent.cs | `src/Ordering.API/Application/IntegrationEvents/Events/OrderStockRejectedIntegrationEvent.cs` | ソース | 入力統合イベント |
| OrderStockRejectedIntegrationEventHandler.cs | `src/Ordering.API/Application/IntegrationEvents/EventHandling/OrderStockRejectedIntegrationEventHandler.cs` | ソース | 統合イベントハンドラ |
| SetStockRejectedOrderStatusCommand.cs | `src/Ordering.API/Application/Commands/SetStockRejectedOrderStatusCommand.cs` | ソース | コマンド定義 |
| SetStockRejectedOrderStatusCommandHandler.cs | `src/Ordering.API/Application/Commands/SetStockRejectedOrderStatusCommandHandler.cs` | ソース | コマンドハンドラ |
| Order.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/Order.cs` | ソース | 注文エンティティ |
| OrderItem.cs | `src/Ordering.Domain/AggregatesModel/OrderAggregate/OrderItem.cs` | ソース | 注文アイテムエンティティ |
