# 機能設計書 29-決済シミュレーション

## 概要

本ドキュメントは、eShopシステムにおける決済シミュレーション機能の設計を記述するものである。PaymentProcessorサービスにおいて、注文の決済処理をシミュレートし、設定により成功/失敗を制御する機能仕様を定義する。

### 本機能の処理概要

決済シミュレーション機能は、在庫確認完了（StockConfirmed）の統合イベントを受信した際に、実際の決済ゲートウェイの代わりにシミュレーションを行う機能である。環境変数の設定値により決済の成功/失敗を制御し、結果に応じてOrderPaymentSucceededまたはOrderPaymentFailedの統合イベントを発行する。

**業務上の目的・背景**：本番環境では実際の決済ゲートウェイ（Stripe、PayPalなど）と連携するが、開発・テスト環境では実際の決済を行わずに注文フローの動作確認が必要である。本機能は、決済処理をシミュレートすることで、開発者がエンドツーエンドの注文フローをテストできるようにする。また、決済失敗のシナリオをテストするために、設定により意図的に失敗を発生させることができる。

**機能の利用シーン**：
- 開発・テスト環境での注文フローのテスト
- 決済成功シナリオのテスト（PaymentSucceeded=true）
- 決済失敗シナリオのテスト（PaymentSucceeded=false）
- CI/CD パイプラインでの自動テスト

**主要な処理内容**：
1. OrderStatusChangedToStockConfirmedIntegrationEventを受信
2. PaymentOptions.PaymentSucceeded設定値を確認
3. 設定値に応じてOrderPaymentSucceededまたはOrderPaymentFailedイベントを生成
4. イベントバスに統合イベントを発行

**関連システム・外部連携**：
- Ordering.APIからOrderStatusChangedToStockConfirmedIntegrationEventを受信
- イベントバスを通じてOrderPaymentSucceeded/OrderPaymentFailedIntegrationEventを発行
- Ordering.APIが発行されたイベントを受信して処理

**権限による制御**：
- システム内部処理のため、外部からの直接呼び出しは不可
- イベント駆動による自動実行

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | なし | - | バックグラウンドサービスのため画面なし |

## 機能種別

バックグラウンド処理 / シミュレーション / イベント駆動

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| OrderId | int | Yes | 決済対象の注文ID | OrderStatusChangedToStockConfirmedIntegrationEventから取得 |

### 入力データソース

- イベントバス経由でOrderStatusChangedToStockConfirmedIntegrationEventを受信
- PaymentOptions設定（環境変数/設定ファイル）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 成功時 | OrderPaymentSucceededIntegrationEvent | 決済成功イベント（OrderIdを含む） |
| 失敗時 | OrderPaymentFailedIntegrationEvent | 決済失敗イベント（OrderIdを含む） |

### 出力先

- イベントバス（RabbitMQ）

## 処理フロー

### 処理シーケンス

```
1. 統合イベントの受信
   └─ OrderStatusChangedToStockConfirmedIntegrationEventをイベントバスから受信

2. OrderStatusChangedToStockConfirmedIntegrationEventHandlerの実行
   └─ ログ出力（イベント受信の記録）

3. 設定値の確認
   └─ IOptionsMonitor<PaymentOptions>からPaymentSucceeded値を取得

4. 結果イベントの生成
   └─ PaymentSucceeded == true の場合
      └─ OrderPaymentSucceededIntegrationEventを生成
   └─ PaymentSucceeded == false の場合
      └─ OrderPaymentFailedIntegrationEventを生成

5. イベントの発行
   └─ IEventBus.PublishAsync()でイベントを発行
   └─ ログ出力（イベント発行の記録）
```

### フローチャート

```mermaid
flowchart TD
    A[OrderStatusChangedToStockConfirmedIntegrationEvent受信] --> B[EventHandler]
    B --> C[ログ出力]
    C --> D{PaymentSucceeded設定値}
    D -->|true| E[OrderPaymentSucceededIntegrationEvent生成]
    D -->|false| F[OrderPaymentFailedIntegrationEvent生成]
    E --> G[EventBus.PublishAsync]
    F --> G
    G --> H[ログ出力]
    H --> I[完了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-29-01 | 決済結果制御 | PaymentSucceeded設定値に基づき決済成功/失敗を決定 | 常に適用 |
| BR-29-02 | シミュレーション目的 | 実際の決済ゲートウェイを使用せずにシミュレート | 開発・テスト環境 |
| BR-29-03 | イベント発行必須 | 必ず成功または失敗のイベントを発行 | 常に適用 |

### 計算ロジック

特になし（設定値による分岐のみ）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| なし | - | - | データベース操作なし |

### テーブル別操作詳細

PaymentProcessorサービスはデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 例外 | イベント発行失敗 | ログ出力、リトライ |

### リトライ仕様

- イベント発行はイベントバスのリトライポリシーに従う
- イベントハンドラのリトライはイベントバスの設定に依存

## トランザクション仕様

- データベース操作がないためトランザクション管理は不要
- イベント発行は非同期で行われる

## パフォーマンス要件

- 処理時間: 即時（シミュレーションのため遅延なし）
- スループット: イベントバスの性能に依存

## セキュリティ考慮事項

- 内部イベント駆動のため、外部からの直接アクセスは不可
- PaymentOptions設定は環境変数または設定ファイルで管理
- 本番環境では実際の決済ゲートウェイに置き換え

## 備考

- 本機能は開発・テスト用のシミュレーションである
- 本番環境では実際の決済ゲートウェイ（Stripe、PayPal等）との連携が必要
- PaymentSucceeded設定はIOptionsMonitorで取得するため、実行時に変更可能
- ASP.NET Core Aspireのコンテナオーケストレーションで動作

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、設定クラスと統合イベントの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | PaymentOptions.cs | `src/PaymentProcessor/PaymentOptions.cs` | 設定クラスの構造（PaymentSucceededプロパティ） |
| 1-2 | OrderStatusChangedToStockConfirmedIntegrationEvent.cs | `src/PaymentProcessor/IntegrationEvents/Events/OrderStatusChangedToStockConfirmedIntegrationEvent.cs` | 入力イベントの構造 |
| 1-3 | OrderPaymentSucceededIntegrationEvent.cs | `src/PaymentProcessor/IntegrationEvents/Events/OrderPaymentSucceededIntegrationEvent.cs` | 成功イベントの構造 |
| 1-4 | OrderPaymentFailedIntegrationEvent.cs | `src/PaymentProcessor/IntegrationEvents/Events/OrderPaymentFailedIntegrationEvent.cs` | 失敗イベントの構造 |

**読解のコツ**: PaymentOptionsはIOptionsMonitor<T>で取得されるため、アプリケーション実行中に設定値が変更されても即座に反映される。

#### Step 2: イベントハンドラを理解する

統合イベントを受信して処理する流れを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | OrderStatusChangedToStockConfirmedIntegrationEventHandler.cs | `src/PaymentProcessor/IntegrationEvents/EventHandling/OrderStatusChangedToStockConfirmedIntegrationEventHandler.cs` | イベントハンドラの処理フロー |

**主要処理フロー**:
1. **3-7行目**: コンストラクタ（IEventBus, IOptionsMonitor<PaymentOptions>, ILogger依存性注入）
2. **9-33行目**: Handle()メソッド全体
3. **11行目**: ログ出力（イベント受信の記録）
4. **13行目**: 結果イベント変数の宣言
5. **21-28行目**: PaymentSucceeded設定値による分岐
6. **21-24行目**: true の場合はOrderPaymentSucceededIntegrationEventを生成
7. **25-28行目**: false の場合はOrderPaymentFailedIntegrationEventを生成
8. **30行目**: ログ出力（イベント発行の記録）
9. **32行目**: イベントバスに発行

#### Step 3: サービス構成を理解する

PaymentProcessorサービスの構成を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Program.cs | `src/PaymentProcessor/Program.cs` | サービスの構成、イベントハンドラの登録 |

**読解のコツ**: PaymentProcessorはASP.NET Core Workerサービスとして実装されており、Webサーバーを持たない。イベントバスからのイベント受信とイベント発行のみを行う。

### プログラム呼び出し階層図

```
OrderStatusChangedToStockConfirmedIntegrationEvent（イベントバス経由）
    │
    └─ OrderStatusChangedToStockConfirmedIntegrationEventHandler.Handle()
           │
           ├─ Logger.LogInformation() [イベント受信ログ]
           │
           ├─ IOptionsMonitor<PaymentOptions>.CurrentValue.PaymentSucceeded
           │      │
           │      ├─ true → OrderPaymentSucceededIntegrationEvent 生成
           │      │
           │      └─ false → OrderPaymentFailedIntegrationEvent 生成
           │
           ├─ Logger.LogInformation() [イベント発行ログ]
           │
           └─ IEventBus.PublishAsync()
                  │
                  └─ OrderPaymentSucceeded/Failed IntegrationEvent
                         │
                         └─ Ordering.API へ
```

### データフロー図

```
[入力]                              [処理]                                [出力]

Ordering.API
    │
    └─ OrderStatusChangedTo ────────▶ PaymentProcessor ──────────────────▶ Ordering.API
       StockConfirmed                     │
       IntegrationEvent                   │
         │                                ▼
         └─ OrderId               EventHandler.Handle()
                                          │
                                          ▼
                                  PaymentOptions
                                    │
                                    ├─ PaymentSucceeded = true
                                    │      │
                                    │      └─▶ OrderPaymentSucceeded
                                    │           IntegrationEvent
                                    │                │
                                    │                ▼
                                    │         SetPaidOrderStatusCommand
                                    │
                                    └─ PaymentSucceeded = false
                                           │
                                           └─▶ OrderPaymentFailed
                                                IntegrationEvent
                                                     │
                                                     ▼
                                              注文キャンセル処理
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| PaymentOptions.cs | `src/PaymentProcessor/PaymentOptions.cs` | ソース | 設定クラス |
| OrderStatusChangedToStockConfirmedIntegrationEvent.cs | `src/PaymentProcessor/IntegrationEvents/Events/OrderStatusChangedToStockConfirmedIntegrationEvent.cs` | ソース | 入力統合イベント |
| OrderPaymentSucceededIntegrationEvent.cs | `src/PaymentProcessor/IntegrationEvents/Events/OrderPaymentSucceededIntegrationEvent.cs` | ソース | 成功統合イベント |
| OrderPaymentFailedIntegrationEvent.cs | `src/PaymentProcessor/IntegrationEvents/Events/OrderPaymentFailedIntegrationEvent.cs` | ソース | 失敗統合イベント |
| OrderStatusChangedToStockConfirmedIntegrationEventHandler.cs | `src/PaymentProcessor/IntegrationEvents/EventHandling/OrderStatusChangedToStockConfirmedIntegrationEventHandler.cs` | ソース | イベントハンドラ |
| Program.cs | `src/PaymentProcessor/Program.cs` | ソース | サービスエントリーポイント |
